import {
    MessageType,
    ICreateSpreadMessage,
    IDestroySpreadMessage,
    IUser
} from './message-protocol';

interface IFrameInstance {
    iframe: HTMLIFrameElement;
    user: IUser;
    isReady: boolean;
}

export class UserManager {
    private mockUsers: IUser[] = [
        { id: '1', name: 'User1', color: 'red' },
        { id: '2', name: 'User2', color: 'green' },
        { id: '3', name: 'User3', color: 'blue' },
        { id: '4', name: 'User4', color: 'purple' },
    ];
    private existUsers: string[] = [];
    private dom: HTMLDivElement;
    private iframeCache = new Map<string, IFrameInstance>();
    private pendingCreates = new Map<string, { user: IUser; hostId: string }>();

    constructor(domId: string) {
        this.dom = document.getElementById(domId) as HTMLDivElement;
        this.setupMessageListener();
    }

    private setupMessageListener() {
        window.addEventListener('message', (event) => {
            const message = event.data;

            switch (message.type) {
                case MessageType.IFRAME_READY:
                    this.handleIframeReady(event.source as Window);
                    break;

                case MessageType.SPREAD_READY:
                    console.log(`Spread ready for user: ${message.payload.userId}`);
                    const instance = this.iframeCache.get(message.payload.userId);
                    if (instance) {
                        instance.isReady = true;
                    }
                    break;
            }
        });
    }

    private handleIframeReady(source: Window) {
        for (const [userId, pending] of this.pendingCreates) {
            const instance = this.iframeCache.get(userId);
            if (instance && instance.iframe.contentWindow === source) {
                const createMsg: ICreateSpreadMessage = {
                    type: MessageType.CREATE_SPREAD,
                    payload: {
                        user: pending.user,
                        roomId: 'room1'
                    }
                };
                instance.iframe.contentWindow.postMessage(createMsg, '*');
                this.pendingCreates.delete(userId);
                break;
            }
        }
    }

    private createIframe(hostId: string, user: IUser) {
        const container = document.getElementById(hostId);
        if (!container) return;

        const iframe = document.createElement('iframe');
        iframe.src = './iframe.html';
        iframe.className = 'spread-iframe';
        iframe.id = `iframe-${user.id}`;

        container.innerHTML = '';
        container.appendChild(iframe);

        this.iframeCache.set(user.id, {
            iframe,
            user,
            isReady: false
        });

        this.pendingCreates.set(user.id, { user, hostId });
    }

    private destroyIframe(userId: string) {
        const instance = this.iframeCache.get(userId);
        if (!instance) return;

        if (instance.iframe.contentWindow) {
            const destroyMsg: IDestroySpreadMessage = {
                type: MessageType.DESTROY_SPREAD
            };
            instance.iframe.contentWindow.postMessage(destroyMsg, '*');
        }

        instance.iframe.remove();

        this.iframeCache.delete(userId);
    }

    addUser() {
        const userId = this.getMockUserId();
        if (!userId) return;

        const index = this.existUsers.length;
        if (index < 4) {
            const user = this.mockUsers[index];
            this.createIframe(`root${index + 1}`, user);
            this.existUsers.push(userId);
            this.updateUI();
        }
    }

    removeUser(userId: string) {
        this.destroyIframe(userId);
        this.existUsers = this.existUsers.filter(item => item !== userId);
        this.updateUI();
    }

    private updateUI() {
        const count = this.existUsers.length;

        const wrapper1 = document.getElementById("wrapper1");
        const wrapper2 = document.getElementById("wrapper2");
        const wrapper3 = document.getElementById("wrapper3");
        const wrapper4 = document.getElementById("wrapper4");
        const container1 = document.getElementById("container1");
        const container2 = document.getElementById("container2");

        if (count > 1) {
            wrapper1!.style.width = '50%';
            wrapper2!.style.display = 'block';
        } else {
            wrapper1!.style.width = '100%';
            wrapper2!.style.display = 'none';
        }

        if (count > 3) {
            wrapper3!.style.width = '50%';
            wrapper4!.style.display = 'block';
        } else {
            wrapper3!.style.width = '100%';
            wrapper4!.style.display = 'none';
        }

        if (count < 3) {
            container1!.style.height = 'calc(100% - 40px)';
            container2!.style.display = 'none';
        } else {
            container1!.style.height = 'calc(50% - 20px)';
            container2!.style.display = 'flex';
        }

        this.renderUserManager();
    }

    private getMockUserId(): string | undefined {
        for (let i = 0; i < this.mockUsers.length; i++) {
            const id = this.mockUsers[i].id;
            if (this.existUsers.findIndex(item => item === id) === -1) {
                return id;
            }
        }
        return undefined;
    }

    private getUserInfo(userId: string): IUser | undefined {
        return this.mockUsers.find(item => item.id === userId);
    }

    private renderUserManager() {
        if (!this.dom) return;

        while (this.dom.firstChild) {
            this.dom.removeChild(this.dom.firstChild);
        }

        this.existUsers.forEach((item, index) => {
            this.createUser(item, index !== 0 && index === this.existUsers.length - 1);
        });

        this.createAddButton();
    }

    private createUser(userId: string, addRemoveBtn: boolean) {
        const userInfo = this.getUserInfo(userId);
        if (!userInfo) return;

        const userBox = document.createElement('div');
        userBox.className = 'user-box';
        userBox.textContent = userInfo.name;
        userBox.style.cursor = "normal";

        if (addRemoveBtn) {
            userBox.style.cursor = "pointer";
            const deleteBtn = document.createElement('span');
            deleteBtn.className = 'delete-btn';

            userBox.addEventListener('click', () => {
                this.removeUser(userId);
            });
            userBox.appendChild(deleteBtn);
        }

        this.dom.appendChild(userBox);
    }

    private createAddButton() {
        if (this.existUsers.length < this.mockUsers.length) {
            const addButton = document.createElement('button');
            addButton.textContent = '+';
            addButton.className = 'add-user-button';

            addButton.addEventListener('click', () => {
                this.addUser();
            });

            this.dom.appendChild(addButton);
        }
    }
}
