import * as OT from '@mescius/js-collaboration-ot-client';
import {redirect, useLoaderData, useNavigate} from 'react-router-dom';
import {useEffect, useMemo, useState} from 'react';
import {Button, Empty, Flex, Layout, message, notification, Popconfirm, Spin, Timeline} from 'antd';
import {Content} from 'antd/es/layout/layout';
import Sider from 'antd/es/layout/Sider';
import { RollbackOutlined } from '@ant-design/icons';

import {konva_type, rich_text_type, spreadsheet_type} from '../ot-types';
import {normalizeError, translateToLocalDate} from '../util/util';
import KonvaViewer from '../components/viewer/konva';
import QuillViewer from '../components/viewer/quill';
import SpreadSheetsViewer from '../components/viewer/spreadsheets';
import {CustomHeader, PageType} from "../components/Header";
import {docReq, IDoc} from '../request/doc';
import { connect } from '../request/req';

export async function loader({ params }: any) {
    try {
        const doc = await docReq.get(params.docId);
        const ops = await docReq.getOps(params.docId);

        return { doc, ops };
    } catch (error) {
        notification.error({
            message: "fetch doc error!",
            description: normalizeError(error)
        });
        return redirect('/all-files');
    }
}

interface IOpEx {
    userId: string;
    username: string;
    docId: string;
    version: number;
    date: string;
}

export default function DocHistory() {
    const { doc, ops } = useLoaderData() as { doc: IDoc, ops: IOpEx[] };
    const [version, setVersion] = useState<number>();
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [data, setData] = useState<any>(null);
    const navigate = useNavigate();

    useEffect(() => {
        setIsLoading(true);
        docReq.getHistorySnapshot(doc.id, version).then((snapshot) => {
            console.log('fetch snapshot', snapshot);
            setData(snapshot.data);
            setIsLoading(false);
        });
    }, [version, doc]);

    const onClick = (v: number) => {
        setVersion(v);
    };

    const handleRestore = async (op: IOpEx) => {
        try {
            const conn = connect(doc.id);
            const sharedDoc = new OT.SharedDoc(conn) as any;
            await sharedDoc.subscribe();

            // The restore method accepts snapshotVersion (op.version + 1)
            const snapshotVersion = op.version + 1;
            await sharedDoc.restore(snapshotVersion);

            message.success(`Successfully restored to version ${snapshotVersion}`);

            // Clean up connection
            conn.close();
            sharedDoc.destroy();

            // Navigate back to edit page after restore
            setTimeout(() => {
                navigate(`/open/${doc.id}`);
            }, 500);
        } catch (error) {
            message.error('Failed to restore version');
            console.error('Failed to restore:', error);
        }
    };

    const ContentCom = useMemo(() => {
        const contentStyle: React.CSSProperties = {
            padding: 50,
            background: 'rgba(0, 0, 0, 0.05)',
            borderRadius: 4,
        };

        const content = <div style={contentStyle} />;
        if (isLoading) {
            return <Flex justify = "center" align="center" style={{ height: "100%" }}><Spin tip="Loading" size="large">
                {content}
            </Spin></Flex>;
        } else if (data === null) {
            return <Flex justify = "center" align="center" style={{ height: "100%" }}><Empty /></Flex>;
        } else {
            return <HistoryViewer type={doc.type} data={data} />
        }
    }, [data, doc, isLoading])

    return (
        <CustomHeader type={PageType.History} doc={doc}>
            <Layout>
                <Content >{ContentCom}</Content >
                <Sider width={260} style={{ background: "#f5f5f5", paddingTop: "20px" }}>
                    <Timeline
                        style={{
                            padding: '5px 8px',
                            height: "100%",
                            overflow: "auto"
                        }}
                        items={
                            ops.slice().reverse().map((op, index) => {
                                const snapshotVersion = op.version + 1;
                                const isSelected = version === snapshotVersion;
                                const isLatest = index === 0;
                                return {
                                    color: isSelected ? 'blue' : 'gray',
                                    children: <TimelineItem
                                        op={op}
                                        snapshotVersion={snapshotVersion}
                                        isSelected={isSelected}
                                        isLatest={isLatest}
                                        onClick={onClick}
                                        onRestore={handleRestore}
                                    />
                                };
                            })
                        }
                    ></Timeline>
                </Sider>
            </Layout>
        </CustomHeader>
    )
}

function HistoryViewer(props: { type: string, data: any }) {
    const { data, type } = props;

    let editor;
    if (type === konva_type.uri) {
        editor = <KonvaViewer data={data} />
    } else if (type === spreadsheet_type.uri) {
        editor = <SpreadSheetsViewer data={data} />
    } else if (type === rich_text_type.uri) {
        editor = <QuillViewer data={data} />
    } else {
        console.error('error');
    }

    return editor;
}

interface ITimelineItemProps {
    op: IOpEx;
    snapshotVersion: number;
    isSelected: boolean;
    isLatest: boolean;
    onClick: (version: number) => void;
    onRestore: (op: IOpEx) => void;
}

function TimelineItem(props: ITimelineItemProps) {
    const { op, snapshotVersion, isSelected, isLatest, onClick, onRestore } = props;
    const [isHovered, setIsHovered] = useState(false);

    const selectedBg = '#e6f4ff';
    const hoverBg = '#fafafa';
    const selectedBorderColor = '#1890ff';
    const defaultBorderColor = '#e8e8e8';
    const shadowDefault = '0 1px 2px rgba(0, 0, 0, 0.05)';
    const shadowHover = '0 2px 4px rgba(0, 0, 0, 0.08)';
    const shadowSelected = '0 2px 8px rgba(24, 144, 255, 0.2)';

    return (
        <div
            onClick={(e: React.MouseEvent<HTMLDivElement>) => {
                // Prevent click if clicking on the restore button
                if ((e.target as HTMLElement).closest('.restore-btn-wrapper')) {
                    return;
                }
                onClick(snapshotVersion);
            }}
            onMouseEnter={() => setIsHovered(true)}
            onMouseLeave={() => setIsHovered(false)}
            style={{
                userSelect: 'none',
                padding: '10px 12px',
                backgroundColor: isSelected ? selectedBg : (isHovered ? hoverBg : '#fff'),
                border: `1px solid ${isSelected ? selectedBorderColor : defaultBorderColor}`,
                borderRadius: '8px',
                cursor: 'pointer',
                boxShadow: isSelected ? shadowSelected : (isHovered ? shadowHover : shadowDefault),
                minWidth: 0,
                width: '100%',
            }}
        >
            {/* Header: Version + Restore Button */}
            <div style={{
                display: 'flex',
                justifyContent: 'space-between',
                alignItems: 'center',
                marginBottom: '5px'
            }}>
                <div style={{
                    display: 'flex',
                    alignItems: 'center',
                    flex: 1,
                    minWidth: 0,
                    overflow: 'hidden'
                }}>
                    {/* Version Number */}
                    <span style={{
                        fontWeight: 600,
                        fontSize: '13px',
                        color: '#262626',
                        whiteSpace: 'nowrap'
                    }}>
                        Version {snapshotVersion}
                    </span>
                </div>

                {/* Restore Button - Hidden for latest version */}
                {!isLatest && (
                    <div className="restore-btn-wrapper" style={{ flexShrink: 0 }}>
                        <Popconfirm
                            title="Restore Version"
                            description={`Are you sure you want to restore to version ${snapshotVersion}?`}
                            onConfirm={() => onRestore(op)}
                            okText="OK"
                            cancelText="Cancel"
                        >
                            <Button
                                danger
                                size="small"
                                icon={<RollbackOutlined />}
                                onClick={(e) => e.stopPropagation()}
                                style={{
                                    borderRadius: '4px',
                                    fontSize: '12px',
                                    fontWeight: 500,
                                    height: '24px',
                                    padding: '0 8px'
                                }}
                            >
                                Restore
                            </Button>
                        </Popconfirm>
                    </div>
                )}
            </div>

            {/* Separator Line */}
            <div style={{
                height: '1px',
                width: '100%',
                background: 'linear-gradient(to right, transparent, rgba(0, 0, 0, 0.06), transparent)',
                marginBottom: '5px'
            }} />

            {/* User Info and Timestamp */}
            <div style={{
                display: 'flex',
                flexDirection: 'column',
                gap: '4px'
            }}>
                <span style={{
                    fontSize: '13px',
                    color: '#595959',
                    fontWeight: isSelected ? 500 : 400,
                    overflow: 'hidden',
                    textOverflow: 'ellipsis',
                    whiteSpace: 'nowrap'
                }}>
                    {op.username}
                </span>
                <span style={{
                    fontSize: '12px',
                    color: '#8c8c8c',
                    whiteSpace: 'nowrap'
                }}>
                    {translateToLocalDate(op.date)}
                </span>
            </div>
        </div>
    );
}