/* eslint-disable @typescript-eslint/no-explicit-any */
import * as OT from '@mescius/js-collaboration-ot-client';
import { Presence } from "@mescius/js-collaboration-presence-client";
import Quill from 'quill';
import 'quill/dist/quill.snow.css';
import QuillCursors from "quill-cursors";
import { IDoc } from '../../request/doc';

import { useEffect, useRef, useState } from 'react';
import { rich_text_type } from '../../ot-types';
import { IUser } from '../../request/user';
import { connect } from '../../request/req';
import {CustomHeader, PageType} from "../Header";

Quill.register('modules/cursors', QuillCursors);

OT.TypesManager.register(rich_text_type);

interface IPresence {
    user: IUser;
}

export default function SharedQuill(props: { doc: IDoc, user: IUser, readonly: boolean }) {
    const ref = useRef<HTMLDivElement>(null);
    const { doc, user } = props;

    useEffect(() => {
        const conn = connect(doc.id);
        const sharedDoc = new OT.SharedDoc(conn);
        const presence = new Presence<IPresence>(conn);
        const div = ref.current as HTMLDivElement;

        const quill = new Quill(div, { theme: 'snow', readOnly: props.readonly, bounds: div, modules: { cursors: true } });
        const cursors = quill.getModule('cursors');
        function updatePresenceList() {
            cursors.clearCursors();
            for (const id in presence.otherStates) {
                const data = presence.otherStates[id] as any;

                cursors.createCursor(id, data.user.username, data.user.color);
                cursors.moveCursor(id, data.selection);
            }
        }
        bind(quill, sharedDoc, presence);
        presence.subscribe().then(() => {
            updatePresenceList();
            presence.submitLocalState({user});
            // Update local presence on selection change
            quill.on("selection-change", (range) => {
                presence.submitLocalStateField('selection', range);
            });
            presence.on("add", () => updatePresenceList());
            presence.on("update", () => updatePresenceList());
            presence.on("remove", () => updatePresenceList());
        });

        return () => {
            conn.close();
            sharedDoc.destroy();
            presence.destroy();

            div.previousSibling?.remove();
            div.innerHTML = '';
        }
    }, [doc.id, user])

    return (
        <CustomHeader type={PageType.Open} doc={doc}>
            <div style={{ width: '100%', height: '100%', background: 'white', display: 'flex', flexDirection: 'column' }}>
                <div ref={ref}></div>
            </div>
        </CustomHeader>
    )
}

async function bind(quill: Quill, doc: OT.SharedDoc, presence: Presence<IPresence>) {
    await doc.subscribe();
    quill.setContents(doc.data as any);
    quill.on('text-change', function (delta: unknown, _oldDelta: unknown, source: string) {
        console.log('text-change');
        if (source !== 'user') return;
        doc.submitOp(delta, { source: doc.connection.id });
    });
    doc.on('op', function (op, source) {
        if (source === doc.connection.id) return;
        quill.updateContents(op as any);
        // Update the content will cause the selection to change, but Quill will not trigger the selection-change event.
        presence.submitLocalStateField('selection', quill.getSelection());
    });
    doc.on('restore', function (targetVersion, source) {
        if (source === doc.connection.id) return;
        quill.setContents(doc.data as any);
    });

    doc.on('error', (err) => console.error(err))
}
