declare module GC{
    module Data{

        export interface IRelationship{
            sourceTable: GC.Data.Table;
            sourceFieldName: string;
            sourceRelationshipName: string;
            targetTable: GC.Data.Table;
            targetFieldName: string;
            targetRelationshipName: string;
        }


        export interface ITables{
            [tableName: string]: GC.Data.Table
        }


        export interface IViews{
            [viewName: string]: GC.Data.View
        }


        export interface RowFormulaRule{
            /**
             * the formula
             */
            formula: string;
            /**
             * the row style when the formula evaluation is true
             */
            style: GC.Data.StyleOptions;
        }


        /**
         * @typedef GC.Data.AverageRuleOptions - The options of average rule.
         * @property {"averageRule"} ruleType - The rule type if you want to use average rule.
         * @property {GC.Data.AverageType} type - The average type.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type AverageRuleOptions = 
            {
                /**
                 * The rule type if you want to use average rule.
                 */
                ruleType: "averageRule";
                /**
                 * The average type.
                 */
                type: GC.Data.AverageType;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.AverageType
         * @type {"above"|"below"|"equalOrAbove"|"equalOrBelow"|"above1StdDev"|"below1StdDev"|"above2StdDev"|"below2StdDev"|"above3StdDev"|"below3StdDev"}
         * @description The average type.
         */
        export type AverageType = 
            "above"|"below"|"equalOrAbove"|"equalOrBelow"|"above1StdDev"|"below1StdDev"|"above2StdDev"|"below2StdDev"|"above3StdDev"|"below3StdDev"
        

        /**
         * @typedef GC.Data.CellButtonOptions
         * @property {string} position - The button's position in cell, which supports "left", "right", "leftOfText", "rightOfText".
         * @property {boolean} useButtonStyle - Whether the cellButton is a button style, default value is false.
         * @property {boolean} enabled - Whether the cell button responds to user actions, default value is true.
         * @property {number} width - The button's width. If it is set to null or undefined, the button width is auto fit based on the caption and image size.
         * @property {string} caption - The text of the button to display.
         * @property {string} imageSrc - When imageType is custom, can Specifies a image (base64) by imageSrc.
         * @property {Object} imageSize - The image's size by object {width: 16, height: 16}, default value is 16px * 16px.
         * @property {number} imageSize.width - The image width.
         * @property {number} imageSize.height - The image height.
         * @property {string} captionAlign - The alignment of image and caption, which supports "left", "right".
         * @property {string | Function} command - When click button, allow user to execute a command or user can execute a callback.
         * @property {string} imageType - The button's type (the type of image to display in the button). Provide some predefined type for cellButton, custom allow to Specifies icon.
         * It supports "none", "custom", "clear", "cancel", "ok", "dropdown", "ellipsis", "left", "right", "plus", "minus", "undo", "redo", "search", "separator", "spinLeft", "spinRight", "collapse", "expand".
         * @property {string} visibility - The button can be visible "always", "onSelected", "onEditing", default value is "always".
         * @property {string} hoverBackColor - The hover backColor of cell button when the button is visible and enable.
         * @property {string} buttonBackColor - The backColor of cell button when the button is enable.
         */
        export type CellButtonOptions = 
            {
                /**
                 * The button's position in cell, which supports "left", "right", "leftOfText", "rightOfText".
                 */
                position?: "left" | "right" | "leftOfText" | "rightOfText";
                /**
                 * Whether the cellButton is a button style, default value is false.
                 */
                useButtonStyle?: boolean;
                /**
                 * Whether the cell button responds to user actions, default value is true.
                 */
                enabled?: boolean;
                /**
                 * The button's width. If it is set to null or undefined, the button width is auto fit based on the caption and image size.
                 */
                width?: number;
                /**
                 * The text of the button to display.
                 */
                caption?: string;
                /**
                 * When imageType is custom, can Specifies a image (base64) by imageSrc.
                 */
                imageSrc?: string;
                /**
                 * The image's size by object {width: 16, height: 16}, default value is 16px * 16px.
                 */
                imageSize?: { width: number, height: number };
                /**
                 * The alignment of image and caption, which supports "left", "right".
                 */
                captionAlign?: "left" | "right";
                /**
                 * When click button, allow user to execute a command or user can execute a callback.
                 */
                command?: string | ((sheet: any, row: number, col: number, option: any) => void);
                /**
                 * The button's type (the type of image to display in the button). Provide some predefined type for cellButton, custom allow to Specifies icon.
                 */
                imageType?: "none" | "custom" | "clear" | "cancel" | "ok" | "dropdown" | "ellipsis" | "left" | "right" | "plus" | "minus" | "undo" | "redo" | "search" | "separator" | "spinLeft" | "spinRight" | "collapse" | "expand";
                /**
                 * The button can be visible "always", "onSelected", "onEditing", default value is "always".
                 */
                visibility?: "always" | "onSelected" | "onEditing";
                /**
                 * The hover backColor of cell button when the button is visible and enable.
                 */
                hoverBackColor?: string;
                /**
                 * The backColor of cell button when the button is enable.
                 */
                buttonBackColor?: string;
            }
        

        /**
         * @typedef GC.Data.CellValueComparisonOperator
         * @type {"equalsTo"|"notEqualsTo"|"greaterThan"|"greaterThanOrEqualsTo"|"lessThan"|"lessThanOrEqualsTo"|"between"|"notBetween"}
         * @description The cell value comparison operator.
         */
        export type CellValueComparisonOperator = 
            "equalsTo"|"notEqualsTo"|"greaterThan"|"greaterThanOrEqualsTo"|"lessThan"|"lessThanOrEqualsTo"|"between"|"notBetween"
        

        /**
         * @typedef GC.Data.CellValueRuleOptions - The options of cell value rule.
         * @property {"cellValueRule"} ruleType - The rule type if you want to use cell value rule.
         * @property {GC.Data.CellValueComparisonOperator} comparisonOperator - The comparison operator of cell value.
         * @property {GC.Data.CellValueType} value1 - The first value.
         * @property {GC.Data.CellValueType} value2 - The second value.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type CellValueRuleOptions = 
            {
                /**
                 * The rule type if you want to use cell value rule.
                 */
                ruleType: "cellValueRule";
                /**
                 * The comparison operator of cell value.
                 */
                comparisonOperator: GC.Data.CellValueComparisonOperator;
                /**
                 * The first value.
                 */
                value1: GC.Data.CellValueType;
                /**
                 * The second value.
                 */
                value2: GC.Data.CellValueType;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.CellValueType
         * @type {number|string|boolean|Date|FormulaString}
         * @description The cell value type.
         */
        export type CellValueType = 
            number|string|boolean|Date|FormulaString
        

        /**
         * @typedef GC.Data.CheckboxOptions
         * @property {string} type - The type of the cell type, supports "checkbox".
         * @property {string} caption - The caption.
         * @property {string} textTrue - The text when cell value is true.
         * @property {string} textIndeterminate - The text when cell value is indeterminate.
         * @property {string} textFalse - The text when cell value is false.
         * @property {string} textAlign - The text alignment relative to the check box, which supports "top", "bottom", "left", "right".
         * @property {boolean} isThreeState - Whether the check box supports three states.
         * @property {number} boxSize - The check box size.
         */
        export type CheckboxOptions = 
            {
                /**
                 * The type of the cell type, supports "checkbox".
                 */
                type: "checkbox";
                /**
                 * The caption.
                 */
                caption: string;
                /**
                 * The text when cell value is true.
                 */
                textTrue: string;
                /**
                 * The text when cell value is indeterminate.
                 */
                textIndeterminate: string;
                /**
                 * The text when cell value is false.
                 */
                textFalse: string;
                /**
                 * The text alignment relative to the check box, which supports "top", "bottom", "left", "right".
                 */
                textAlign: "top" | "bottom" | "left" | "right";
                /**
                 * Whether the check box supports three states.
                 */
                isThreeState: boolean;
                /**
                 * The check box size.
                 */
                boxSize: number;
            }
        

        /**
         * @typedef GC.Data.ColorPickerGroup
         * @property {string} name - The group name.
         * @property {string[][]} colors - The group colors.
         * @property {boolean} needScaleColor - Whether generate scale color group.
         */
        export type ColorPickerGroup = 
            {
                /**
                 * The group name.
                 */
                name: string;
                /**
                 * The group colors.
                 */
                colors: string[][];
                /**
                 * Whether generate scale color group.
                 */
                needScaleColor: boolean;
            }
        

        /**
         * @typedef GC.Data.ColorPickerOptions
         * @property {number} colorBlockSize - Every color cell's size.
         * @property {GC.Data.ColorPickerGroup[]} groups - The group of the color picker, every group accept a name as group name, and a color array as the group's colors.
         */
        export type ColorPickerOptions = 
            {
                /**
                 * Every color cell's size.
                 */
                colorBlockSize: number;
                /**
                 * The group of the color picker, every group accept a name as group name, and a color array as the group's colors.
                 */
                groups: GC.Data.ColorPickerGroup[];
            }
        

        /**
         * @typedef GC.Data.ColorString
         * @type {string}
         * @description The string type color.
         */
        export type ColorString = 
            string
        

        /**
         * @typedef GC.Data.ColumnBindingInfo
         * @property {string} name - The bound column of data through name.
         * @property {string} displayName - The display name of this column.
         * @property {string} formatter - The formatter for this column.
         * @property {number | string} size - The size of column.
         * @property {boolean} visible - The visibility of column.
         */
        export type ColumnBindingInfo = 
            {
                /**
                 * The bound column of data through name.
                 */
                name: string;
                /**
                 * The display name of this column.
                 */
                displayName?: string;
                /**
                 * The formatter for this column.
                 */
                formatter?: string;
                /**
                 * The size of column.
                 */
                size?: number | string;
                /**
                 * The visibility of column.
                 */
                visible?: boolean;
            }
        

        /**
         * @typedef GC.Data.ColumnType - The column type
         * @type {"Number"|"Text"|"Formula"|"Checkbox"|"Date"|"Currency"|"Percent"|"Phone"|"Email"|"URL"|"Lookup"|"CreatedTime"|"ModifiedTime"|"Attachment"|"Select"|"Barcode"}
         */
        export type ColumnType = 
            "Number"|"Text"|"Formula"|"Checkbox"|"Date"|"Currency"|"Percent"|"Phone"|"Email"|"URL"|"Lookup"|"CreatedTime"|"ModifiedTime"|"Attachment"|"Select"|"Barcode"
        

        /**
         * @typedef GC.Data.ComboBoxOptions
         * @property {string} type - The type of the cell type, supports "combobox".
         * @property {string} editorValueType - The editor value type, which supports "text", "index", "value".
         * @property {string[] | GC.Data.ItemOptions[]} items - The items, which supports string Array or Object Array which each item contains text and value.
         * @property {number} itemHeight - The height of each item.
         * @property {number} maxDropDownItems - The maximum item count of the drop-down list per page.
         * @property {boolean} editable - Whether the combo box is editable.
         */
        export type ComboBoxOptions = 
            {
                /**
                 * The type of the cell type, supports "combobox".
                 */
                type: "combobox";
                /**
                 * The editor value type, which supports "text", "index", "value".
                 */
                editorValueType?: "text" | "index" | "value";
                /**
                 * The items, which supports string Array or Object Array which each item contains text and value.
                 */
                items: string[] | GC.Data.ItemOptions[];
                /**
                 * The height of each item.
                 */
                itemHeight?: number;
                /**
                 * The maximum item count of the drop-down list per page.
                 */
                maxDropDownItems?: number;
                /**
                 * Whether the combo box is editable.
                 */
                editable?: boolean;
            }
        

        /**
         * @typedef GC.Data.CornerFold
         * @property {number} size - The size of the corner fold.
         * @property {string} position - The corner fold's position, which supports "leftTop", "rightTop", "leftBottom", "rightBottom".
         * @property {string} color - The color of the corner fold.
         */
        export type CornerFold = 
            {
                /**
                 * The size of the corner fold.
                 */
                size?: number;
                /**
                 * The corner fold's position, which supports "leftTop", "rightTop", "leftBottom", "rightBottom".
                 */
                position?: "leftTop" | "rightTop" | "leftBottom" | "rightBottom";
                /**
                 * The color of the corner fold.
                 */
                color?: string;
            }
        

        /**
         * @typedef GC.Data.DataBarAxisPosition
         * @type {"automatic"|"cellMidPoint"|"none"}
         * @description The data bar rule axis position.
         */
        export type DataBarAxisPosition = 
            "automatic"|"cellMidPoint"|"none"
        

        /**
         * @typedef GC.Data.DataBarRuleDirection
         * @type {"leftToRight"|"rightToLeft"}
         * @description The data bar rule direction.
         */
        export type DataBarRuleDirection = 
            "leftToRight"|"rightToLeft"
        

        /**
         * @typedef GC.Data.DataBarRuleNegativeOptions - The options of data bar rule negative data options.
         * @property {GC.Data.ColorString} negativeFillColor - The color of the negative fill.
         * @property {boolean} useNegativeFillColor - Whether the negative fill color is used to paint the negative value.
         * @property {GC.Data.ColorString} negativeBorderColor - The color of the negative border.
         * @property {boolean} useNegativeBorderColor - Whether the negative border color is used to paint the border for the negative value.
         * @property {GC.Data.ColorString} axisColor - The axis color of the data bar.
         * @property {GC.Data.DataBarAxisPosition} axisPosition - The axis position of the data bar.
         */
        export type DataBarRuleNegativeOptions = 
            {
                /**
                 * The color of the negative fill.
                 */
                negativeFillColor: GC.Data.ColorString;
                /**
                 * Whether the negative fill color is used to paint the negative value.
                 */
                useNegativeFillColor: boolean;
                /**
                 * The color of the negative border.
                 */
                negativeBorderColor: GC.Data.ColorString;
                /**
                 * Whether the negative border color is used to paint the border for the negative value.
                 */
                useNegativeBorderColor: boolean;
                /**
                 * The axis color of the data bar.
                 */
                axisColor: GC.Data.ColorString;
                /**
                 * The axis position of the data bar.
                 */
                axisPosition: GC.Data.DataBarAxisPosition;
            }
        

        /**
         * @typedef GC.Data.DataBarRuleOptions - The options of data bar rule.
         * @property {"dataBarRule"} ruleType - The rule type if you want to use data bar rule.
         * @property {GC.Data.ScaleValueType} minType - The minimum scale type.
         * @property {number} minValue - The minimum scale value.
         * @property {GC.Data.ScaleValueType} maxType - The maximum scale type.
         * @property {number} maxValue - The maximum scale value.
         * @property {GC.Data.ColorString} maxColor - The maximum scale color string.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         * @property {boolean} gradient - Whether the data bar is a gradient.
         * @property {boolean} showBarOnly - Whether to display the data bar without text.
         * @property {boolean} showBorder - Whether to paint the border.
         * @property {GC.Data.ColorString} borderColor - The color of the border.
         * @property {GC.Data.DataBarRuleDirection} barDirection - The data bar direction.
         * @property {GC.Data.DataBarRuleNegativeOptions} negativeData - The options of data bar rule negative data options.
         */
        export type DataBarRuleOptions = 
            {
                /**
                 * The rule type if you want to use data bar rule.
                 */
                ruleType: "dataBarRule";
                /**
                 * The minimum scale type.
                 */
                minType: GC.Data.ScaleValueType;
                /**
                 * The minimum scale value.
                 */
                minValue: number;
                /**
                 * The maximum scale type.
                 */
                maxType: GC.Data.ScaleValueType;
                /**
                 * The maximum scale value.
                 */
                maxValue: number;
                color: GC.Data.ColorString;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
                /**
                 * Whether the data bar is a gradient.
                 */
                gradient: boolean;
                /**
                 * Whether to display the data bar without text.
                 */
                showBarOnly: boolean;
                /**
                 * Whether to paint the border.
                 */
                showBorder: boolean;
                /**
                 * The color of the border.
                 */
                borderColor: GC.Data.ColorString;
                /**
                 * The data bar direction.
                 */
                barDirection: GC.Data.DataBarRuleDirection;
                /**
                 * The options of data bar rule negative data options.
                 */
                negativeData: GC.Data.DataBarRuleNegativeOptions;
            }
        

        /**
         * @typedef GC.Data.DateOccurringRuleOptions - The options of date occurring rule.
         * @property {"dateOccurringRule"} ruleType - The rule type if you want to use data occurring rule.
         * @property {GC.Data.DateOccurringType} type - The date occurring type.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type DateOccurringRuleOptions = 
            {
                /**
                 * The rule type if you want to use data occurring rule.
                 */
                ruleType: "dateOccurringRule";
                /**
                 * The date occurring type.
                 */
                type: GC.Data.DateOccurringType;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.DateOccurringType
         * @type {"today"|"yesterday"|"tomorrow"|"last7Days"|"thisMonth"|"lastMonth"|"nextMonth"|"thisWeek"|"lastWeek"|"nextWeek"|"nextQuarter"|"thisQuarter"|"lastQuarter"|"nextYear"|"thisYear"|"lastYear"}
         * @description The date occurring type.
         */
        export type DateOccurringType = 
            "today"|"yesterday"|"tomorrow"|"last7Days"|"thisMonth"|"lastMonth"|"nextMonth"|"thisWeek"|"lastWeek"|"nextWeek"|"nextQuarter"|"thisQuarter"|"lastQuarter"|"nextYear"|"thisYear"|"lastYear"
        

        /**
         * @typedef GC.Data.DateTimePickerOptions
         * @property {boolean} showTime - Whether the calendar need to display time part.
         * @property {string} calendarPage - The default page, which accepts "year", "month", "day".
         * @property {string} startDay - The start day of week, normal the start day is monday or sunday, there user can set any day as it's start day. It accepts "monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday".
         */
        export type DateTimePickerOptions = 
            {
                /**
                 * Whether the calendar need to display time part.
                 */
                showTime: boolean;
                /**
                 * The default page, which accepts "year", "month", "day".
                 */
                calendarPage: "year" | "month" | "day";
                /**
                 * The start day of week, normal the start day is monday or sunday, there user can set any day as it's start day. It accepts "monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday".
                 */
                startDay: "monday" | "tuesday" | "wednesday" | "thursday" | "friday" | "saturday" | "sunday";
            }
        

        /**
         * @typedef GC.Data.DateValidatorOptions
         * @property {"date"} type - The data validator type.
         * @property {GC.Data.CellValueComparisonOperator} comparisonOperator - The data validator comparison operator.
         * @property {Date | GC.Data.FormulaString} value1 - The data validator first value.
         * @property {Date | GC.Data.FormulaString} value2 - The data validator second value if validator comparison operator is "between" or "notBetween".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type DateValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "date";
                /**
                 * The data validator comparison operator.
                 */
                comparisonOperator: GC.Data.CellValueComparisonOperator;
                /**
                 * The data validator first value.
                 */
                value1: Date | GC.Data.FormulaString;
                /**
                 * The data validator second value if validator comparison operator is "between" or "notBetween".
                 */
                value2?: Date | GC.Data.FormulaString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.Decoration
         * @property {GC.Data.Icon[]} icons - The icons of the decoration in the style.
         * @property {GC.Data.CornerFold} cornerFold - The corner fold of the decoration in the style.
         */
        export type Decoration = 
            {
                /**
                 * The icons of the decoration in the style.
                 */
                icons?: GC.Data.Icon[];
                /**
                 * The corner fold of the decoration in the style.
                 */
                cornerFold?: GC.Data.CornerFold;
            }
        

        /**
         * @typedef GC.Data.DropDownOptions
         * @property {string} type - The type of drop down, which supports "colorPicker", "dateTimePicker", "timePicker", "monthPicker", "list", "slider", "calculator", "workflowList", "multiColumn".
         * @property {GC.Data.ColorPickerOptions | GC.Data.DateTimePickerOptions | GC.Data.TimePickerOptions | GC.Data.MonthPickerOptions | GC.Data.ListOptions | GC.Data.SliderOptions | GC.Data.WorkFlowOptions | GC.Data.MultiColumnOptions} option - The option of drop down.
         * @property {Function} submitCommand - A command name or a callback function which will be executed when submit drop down's value.
         */
        export type DropDownOptions = 
            {
                /**
                 * The type of drop down, which supports "colorPicker", "dateTimePicker", "timePicker", "monthPicker", "list", "slider", "calculator", "workflowList", "multiColumn".
                 */
                type: "colorPicker" | "dateTimePicker" | "timePicker" | "monthPicker" | "list" | "slider" | "calculator" | "workflowList" | "multiColumn";
                /**
                 * The option of drop down.
                 */
                option?: GC.Data.ColorPickerOptions | GC.Data.DateTimePickerOptions | GC.Data.TimePickerOptions | GC.Data.MonthPickerOptions | GC.Data.ListOptions | GC.Data.SliderOptions | GC.Data.WorkFlowOptions | GC.Data.MultiColumnOptions;
                /**
                 * A command name or a callback function which will be executed when submit drop down's value.
                 */
                submitCommand?: string | ((sheet: any, value: any, option: any) => void);
            }
        

        /**
         * @typedef GC.Data.DuplicateRuleOptions - The options of duplicate rule.
         * @property {"duplicateRule"} ruleType - The rule type if you want to use duplicate rule.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type DuplicateRuleOptions = 
            {
                /**
                 * The rule type if you want to use duplicate rule.
                 */
                ruleType: "duplicateRule";
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.ErrorStyle
         * @type {"stop" | "warning" | "information"}
         * @description The validator error style.
         */
        export type ErrorStyle = 
            "stop" | "warning" | "information"
        

        /**
         * @typedef GC.Data.FileUploadOptions
         * @property {string} type - The type of the cell type, supports "fileUpload".
         * @property {number} maxSize -The maximum file size that can be uploaded.
         * @property {string} accept - The file types that can be uploaded.
         * @property {boolean} isPreviewEnabled - Whether to display the file preview button.
         * @property {boolean} isDownloadEnabled - Whether to display the file download button.
         * @property {boolean} isClearEnabled - Whether to display the file clear button.
         * @property {number} marginTop - The margin top value.
         * @property {number} marginRight - The margin right value.
         * @property {number} marginBottom - The margin bottom value.
         * @property {number} marginLeft - The margin left value.
         * @property {string} valuePath - The valuePath is used to get the cell value from File.
         */
        export type FileUploadOptions = 
            {
                /**
                 * The type of the cell type, supports "fileUpload".
                 */
                type: "fileUpload";
                maxSize?: number;
                /**
                 * The file types that can be uploaded.
                 */
                accept?: string;
                /**
                 * Whether to display the file preview button.
                 */
                isPreviewEnabled?: boolean;
                /**
                 * Whether to display the file download button.
                 */
                isDownloadEnabled?: boolean;
                /**
                 * Whether to display the file clear button.
                 */
                isClearEnabled?: boolean;
                /**
                 * The margin top value.
                 */
                marginTop?: number;
                /**
                 * The margin right value.
                 */
                marginRight?: number;
                /**
                 * The margin bottom value.
                 */
                marginBottom?: number;
                /**
                 * The margin left value.
                 */
                marginLeft?: number;
                /**
                 * The valuePath is used to get the cell value from File.
                 */
                valuePath?: string;
            }
        

        /**
         * @typedef GC.Data.FormulaListValidatorOptions
         * @property {"formulaList"} type - The data validator type.
         * @property {GC.Data.FormulaString} formula - The data validator formula if the validator type if "formula" or "formulaList".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type FormulaListValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "formulaList";
                /**
                 * The data validator formula if the validator type if "formula" or "formulaList".
                 */
                formula: GC.Data.FormulaString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.FormulaRule
         * @property {GC.Data.FormulaString} formula - The formula.
         * @description The formula rule.
         */
        export type FormulaRule = 
            {
                /**
                 * the formula string
                 */
                formula: GC.Data.FormulaString;
            }
        

        /**
         * @typedef GC.Data.FormulaRuleOptions - The options of formula rule.
         * @property {"formulaRule"} ruleType - The rule type if you want to use formula rule.
         * @property {GC.Data.FormulaString} formula - The formula.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type FormulaRuleOptions = 
            {
                /**
                 * The rule type if you want to use formula rule.
                 */
                ruleType: "formulaRule";
                /**
                 * The formula.
                 */
                formula: GC.Data.FormulaString;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef {string} GC.Data.FormulaString - The formula type string, such as "=SUM(A1:B2)".
         */
        export type FormulaString = 
            string
        

        /**
         * @typedef GC.Data.FormulaValidatorOptions
         * @property {"formula"} type - The data validator type.
         * @property {GC.Data.FormulaString} formula - The data validator formula if the validator type if "formula" or "formulaList".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type FormulaValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "formula";
                /**
                 * The data validator formula if the validator type if "formula" or "formulaList".
                 */
                formula: GC.Data.FormulaString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.GradientFillOptions
         * @property {number} degree - The gradient fill degree.
         * @property {GC.Data.GradientStop[]} stops - The gradient fill stops.
         */
        export type GradientFillOptions = 
            {
                /**
                 * The gradient fill degree.
                 */
                degree?: number;
                /**
                 * The gradient fill stops.
                 */
                stops: GC.Data.GradientStop[];
            }
        

        /**
         * @typedef GC.Data.GradientPathFillOptions
         * @property {string} type - The gradient path fill type, supports "path."
         * @property {number} left - The gradient path fill left.
         * @property {number} right - The gradient path fill right.
         * @property {number} top - The gradient path fill top.
         * @property {number} bottom - The gradient path fill bottom.
         * @property {GC.Data.GradientStop[]} stops - The gradient path fill stops.
         */
        export type GradientPathFillOptions = 
            {
                /**
                 * The gradient path fill type, supports "path."
                 */
                type: "path";
                /**
                 * The gradient path fill left.
                 */
                left?: number;
                /**
                 * The gradient path fill right.
                 */
                right?: number;
                /**
                 * The gradient path fill top.
                 */
                top?: number;
                /**
                 * The gradient path fill bottom.
                 */
                bottom?: number;
                /**
                 * The gradient path fill stops.
                 */
                stops: GC.Data.GradientStop[];
            }
        

        /**
         * @typedef GC.Data.GradientStop
         * @property {string} color - The gradient stop color.
         * @property {number} position - The gradient stop position.
         */
        export type GradientStop = 
            {
                /**
                 * The gradient stop color.
                 */
                color: string;
                /**
                 * The gradient stop position.
                 */
                position: number;
            }
        

        /**
         * @typedef GC.Data.HeaderStyleOptions
         * @property {string | GC.Data.PatternFillOptions | GC.Data.GradientFillOptions | GC.Data.GradientPathFillOptions} backColor - The background color string or pattern fill options, gradient fill options, gradient path fill options.
         * @property {string} foreColor - The foreground color.
         * @property {string} hAlign - The horizontal alignment, which supports "left", "center", "right", "general".
         * @property {string} vAlign - The vertical alignment, which supports "top", "center", "bottom".
         * @property {string} font - The font.
         * @property {string} themeFont - The theme font.
         * @property {string} formatter - The formatter string.
         * @property {GC.Data.LineBorder} borderLeft - The left border.
         * @property {GC.Data.LineBorder} borderTop - The top border.
         * @property {GC.Data.LineBorder} borderRight - The right border.
         * @property {GC.Data.LineBorder} borderBottom - The bottom border.
         * @property {GC.Data.LineBorder} diagonalDown - The diagonal with LeftTop to bottomRight.
         * @property {GC.Data.LineBorder} diagonalUp - The diagonal with topRight to bottomLeft.
         * @property {boolean} locked - Whether the cell, row, or column is locked.
         * @property {number} textIndent - The text indent amount.
         * @property {boolean} wordWrap - Whether words wrap within the cell or cells.
         * @property {boolean} shrinkToFit - Whether content shrinks to fit the cell or cells.
         * @property {string} backgroundImage - The background image to display.
         * @property {string} backgroundImageLayout - The layout for the background image, which supports "stretch", "center", "zoom", "none".
         * @property {GC.Data.TextDecoration} textDecoration - The decoration added to text.
         * @property {string} name - The name.
         * @property {string} parentName - The name of the parent style.
         * @property {string} watermark - The watermark content.
         * @property {string} cellPadding - The cell padding.
         * @property {GC.Data.LabelOptions} labelOptions - The cell label options.
         * @property {boolean} isVerticalText - Whether to set the cell's text vertical.
         * @property {boolean} showEllipsis - Whether the text out of bounds shows ellipsis.
         * @property {GC.Data.Decoration} decoration - The decoration.
         */
        export type HeaderStyleOptions = 
            {
                /**
                 * The background color string or pattern fill options, gradient fill options, gradient path fill options.
                 */
                backColor?: string | GC.Data.PatternFillOptions | GC.Data.GradientFillOptions | GC.Data.GradientPathFillOptions;
                /**
                 * The foreground color.
                 */
                foreColor?: string;
                /**
                 * The horizontal alignment, which supports "left", "center", "right", "general".
                 */
                hAlign?: "left" | "center" | "right" | "general";
                /**
                 * The vertical alignment, which supports "top", "center", "bottom".
                 */
                vAlign?: "top" | "center" | "bottom";
                /**
                 * The font.
                 */
                font?: string;
                /**
                 * The theme font.
                 */
                themeFont?: string;
                /**
                 * The formatter string.
                 */
                formatter?: string;
                /**
                 * The left border.
                 */
                borderLeft?: GC.Data.LineBorder;
                /**
                 * The top border.
                 */
                borderTop?: GC.Data.LineBorder;
                /**
                 * The right border.
                 */
                borderRight?: GC.Data.LineBorder;
                /**
                 * The bottom border.
                 */
                borderBottom?: GC.Data.LineBorder;
                /**
                 * The diagonal with LeftTop to bottomRight.
                 */
                diagonalDown?: GC.Data.LineBorder;
                /**
                 * The diagonal with topRight to bottomLeft.
                 */
                diagonalUp?: GC.Data.LineBorder;
                /**
                 * Whether the cell, row, or column is locked.
                 */
                locked?: boolean;
                /**
                 * The text indent amount.
                 */
                textIndent?: number;
                /**
                 * Whether words wrap within the cell or cells.
                 */
                wordWrap?: boolean;
                /**
                 * Whether content shrinks to fit the cell or cells.
                 */
                shrinkToFit?: boolean;
                /**
                 * The background image to display.
                 */
                backgroundImage?: string;
                /**
                 * The layout for the background image, which supports "stretch", "center", "zoom", "none".
                 */
                backgroundImageLayout?: "stretch" | "center" | "zoom" | "none";
                /**
                 * The decoration added to text.
                 */
                textDecoration?: GC.Data.TextDecoration;
                /**
                 * The name.
                 */
                name?: string;
                /**
                 * The name of the parent style.
                 */
                parentName?: string;
                /**
                 * The watermark content.
                 */
                watermark?: string;
                /**
                 * The cell padding.
                 */
                cellPadding?: string;
                /**
                 * The cell label options.
                 */
                labelOptions?: GC.Data.LabelOptions;
                /**
                 * Whether to set the cell's text vertical.
                 */
                isVerticalText?: boolean;
                /**
                 * Whether the text out of bounds shows ellipsis.
                 */
                showEllipsis?: boolean;
                /**
                 * The decoration.
                 */
                decoration?: GC.Data.Decoration;
            }
        

        /**
         * @typedef GC.Data.HierarchyCustomParseHandler
         * @param {GC.Data.IHierarchyCustomParseOptions} options - the options for the handler.
         * @returns {any} Returns of the parentId of the current record.
         * @description Get the parentId of the current record.
         */
        export type HierarchyCustomParseHandler = 
            (options: GC.Data.IHierarchyCustomParseOptions) => any
        

        /**
         * @typedef GC.Data.HierarchyCustomUnparseHandler
         * @param {GC.Data.IHierarchyCustomUnparseOptions} options - the options for the handler.
         * @returns {any} Returns of the id of the current record.
         * @description Get the id of the current record.
         */
        export type HierarchyCustomUnparseHandler = 
            (options: GC.Data.IHierarchyCustomUnparseOptions) => any
        

        /**
         * @typedef GC.Data.HighlightStyle
         * @property {"circle" | "dogEar" | "icon"} type - The highlight type.
         * @property {GC.Data.ColorString} color - The highlight color.
         * @property {"topLeft" | "topRight" | "bottomRight" | "bottomLeft" | "outsideLeft" | "outsideRight"} position - The highlight position.
         * @property {string} image - The highlight image, it could be an url or base64 data.
         */
        export type HighlightStyle = 
            {
                /**
                 * The highlight type.
                 */
                type?: "circle" | "dogEar" | "icon";
                /**
                 * The highlight color.
                 */
                color?: string;
                /**
                 * The highlight position.
                 */
                position: "topLeft" | "topRight" | "bottomRight" | "bottomLeft" | "outsideLeft" | "outsideRight";
                /**
                 * The highlight image, it could be an url or base64 data.
                 */
                image?: string;
            }
        

        /**
         * @typedef GC.Data.HyperlinkOptions
         * @property {string} type - The type of the cell type, supports "hyperlink".
         * @property {string} linkColor - The color of the link.
         * @property {string} visitedLinkColor - The color of the visited link.
         * @property {string} text - The text of the link.
         * @property {string} linkToolTip - The tooltip of the link.
         * @property {string} target - The type of the link's target, which support "blank", "self", "parent", "top".
         * @property {boolean} activeOnClick - Whether to move to the active cell when clicked.
         */
        export type HyperlinkOptions = 
            {
                /**
                 * The type of the cell type, supports "hyperlink".
                 */
                type: "hyperlink";
                /**
                 * The color of the link.
                 */
                linkColor: string;
                /**
                 * The color of the visited link.
                 */
                visitedLinkColor: string;
                /**
                 * The text of the link.
                 */
                text: string;
                /**
                 * The tooltip of the link.
                 */
                linkToolTip: string;
                /**
                 * The type of the link's target, which support "blank", "self", "parent", "top".
                 */
                target: "blank" | "self" | "parent" | "top";
                /**
                 * Whether to move to the active cell when clicked.
                 */
                activeOnClick: boolean;
            }
        

        /**
         * @typedef GC.Data.IColumn - The column options.
         * @property {string} name - The unique name of the column.
         * @property {string} value - The value of the column, could be a field name of table from database, or formula which uses the fields names.
         * @property {string} type - The column type, any of "Number", "Text", "Formula", "Checkbox", "Date", "Currency", "Percent", "Phone", "Email", "URL", "Lookup", "CreatedTime", "ModifiedTime", "Attachment", "Select", "Barcode"
         * @property {string | string[]} caption - The caption of the column.
         * @property {number | string} width - The width of the column, support number in pixel, or star size.
         * @property {GC.Data.StyleOptions} style - The style of the column.
         * @property {(GC.Data.CellValueRuleOptions | GC.Data.SpecificTextRuleOptions | GC.Data.FormulaRuleOptions | GC.Data.DateOccurringRuleOptions | GC.Data.Top10RuleOptions | GC.Data.UniqueRuleOptions | GC.Data.DuplicateRuleOptions | GC.Data.AverageRuleOptions | GC.Data.TwoScaleRuleOptions | GC.Data.ThreeScaleRuleOptions | GC.Data.DataBarRuleOptions | GC.Data.IconSetRuleOptions)[]} conditionalFormats - The conditional formats array of the column.
         * @property {GC.Data.NumberValidatorOptions | GC.Data.DateValidatorOptions | GC.Data.TimeValidatorOptions | GC.Data.TextLengthValidatorOptions | GC.Data.FormulaValidatorOptions | GC.Data.FormulaListValidatorOptions | GC.Data.ListValidatorOptions} validator - The data validator of the column.
         * @property {boolean} isPrimaryKey - Mark the column as primary key column.
         * @property {boolean} readonly - Mark the column is readonly.
         * @property {boolean} required - Mark the column is required when insert a new row.
         * @property {Object} defaultValue - Provide the default value when insert a new row, could be a const or a formula.
         * @property {GC.Data.HeaderStyleOptions} headerStyle - The header style of the column.
         * @property {boolean} visible - Mark the column is visible.
         * @property {string} headerFit - The header fit mode, any of the "normal", "vertical" or "stack". The default value is "normal".
         * @property {string} dataType - The actual data type of original value, any of "string", "number", "boolean", "object", "formula", "array", "date" or "rowOrder". It is useful for a Date because a Date is a string in JSON data and need be converted.
         * @property {string} dataPattern - The data pattern for parsing string to value, such as formatter "dd/MM/yyyy" for a date string, truthy and falsy value pairs "Yes|No" for a boolean string, decimal separator "," for a numeric string.
         * @property {Object} dataMap - A simple map to display the original value more meaningful, its key could be a number or string, and its value could be a number, string or Date.
         * @property {boolean} spread - Whether to spread a column when its value is an object.
         * @property {string} dataName - The original name of the table field, using this property to map the original name to the specified name.
         * @property {boolean} indexed - Weather need to create the filter cache while creating the table.
         * @property {boolean} allowSort - Whether show sort after opening filer dialog.
         * @property {boolean} allowFilterByValue - Whether show filter by value after opening filer dialog.
         * @property {boolean} allowFilterByList - Whether show filter by list after opening filer dialog. If allowSort, allowFilterByValue and allowFilterByList are all false, not show the filter button in this column.
         * @property {string | GC.Data.ICrossOptions} cross - The cross options of the column.
         * @property {string | (string | number | boolean | Date)[] | GC.Data.ILookupOptions} lookup - Define the lookup for the column, only be used in the columns of the schema of the data source.
         * @property {boolean | GC.Data.IOutlineColumnOptions} outlineColumn - Define the outline column only when the data be the hierarchy data.
         * @property {GC.Data.ITriggerFormulaOption} trigger - The trigger formula of the column.
         */
        export type IColumn = 
            {
                /**
                 * The unique name of the column.
                 */
                name?: string;
                /**
                 * The value of the column, could be a field name of table from database, or formula which uses the fields names.
                 */
                value?: string;
                /**
                 * The column type, any of "Number", "Text", "Formula", "Checkbox", "Date", "Currency", "Percent", "Phone", "Email", "URL", "Lookup", "CreatedTime", "ModifiedTime", "Attachment", "Select", "Barcode"
                 */
                type?: string;
                /**
                 * The caption of the column.
                 */
                caption?: string | string[];
                /**
                 * The width of the column, support number in pixel, or star size.
                 */
                width?: number | string;
                /**
                 * The style of the column.
                 */
                style?: GC.Data.StyleOptions;
                /**
                 * The conditional formats array of the column.
                 */
                conditionalFormats?: Array<GC.Data.CellValueRuleOptions | GC.Data.SpecificTextRuleOptions | GC.Data.FormulaRuleOptions | GC.Data.DateOccurringRuleOptions | GC.Data.Top10RuleOptions | GC.Data.UniqueRuleOptions | GC.Data.DuplicateRuleOptions | GC.Data.AverageRuleOptions | GC.Data.TwoScaleRuleOptions | GC.Data.ThreeScaleRuleOptions | GC.Data.DataBarRuleOptions | GC.Data.IconSetRuleOptions>;
                /**
                 * The data validator of the column.
                 */
                validator?: GC.Data.NumberValidatorOptions | GC.Data.DateValidatorOptions | GC.Data.TimeValidatorOptions | GC.Data.TextLengthValidatorOptions | GC.Data.FormulaValidatorOptions | GC.Data.FormulaListValidatorOptions | GC.Data.ListValidatorOptions;
                /**
                 * Mark the column as primary key column.
                 */
                isPrimaryKey?: boolean;
                /**
                 * Mark the column is readonly.
                 */
                readonly?: boolean;
                /**
                 * Mark the column is required when insert a new row.
                 */
                required?: boolean;
                /**
                 * Provide the default value when insert a new row, could be a const or a formula.
                 */
                defaultValue?: any;
                /**
                 * The header style of the column.
                 */
                headerStyle?: GC.Data.HeaderStyleOptions;
                /**
                 * Mark the column is visible.
                 */
                visible?: boolean;
                /**
                 * The header fit mode, any of the "normal", "vertical" or "stack". The default value is "normal".
                 */
                headerFit?: "normal" | "vertical" | "stack";
                /**
                 * The actual data type of original value, any of "string", "number", "boolean", "object", "formula", "array", "date" or "rowOrder". It is useful for a Date because a Date is a string in JSON data and need be converted.
                 */
                dataType?: "string" | "number" | "boolean" | "object" | "formula" | "array" | "date" | "rowOrder" | "TaskStartDate" | "TaskFinishDate" | "TaskDuration" | "TaskSubject" | "TaskSchedulingMode" | "TaskComplete" | "TaskPredecessor";
                /**
                 * The data pattern for parsing string to value, such as formatter "dd/MM/yyyy" for a date string, truthy and falsy value pairs "Yes|No" for a boolean string, decimal separator "," for a numeric string.
                 */
                dataPattern?: string;
                /**
                 * A simple map to display the original value more meaningful, its key could be a number or string, and its value could be a number, string or Date.
                 */
                dataMap?: any;
                /**
                 * Whether to spread a column when its value is an object.
                 */
                spread?: boolean;
                /**
                 * The original name of the table field, using this property to map the original name to the specified name.
                 */
                dataName?: string;
                /**
                 * Weather need to create the filter cache while creating the table.
                 */
                indexed?: boolean;
                /**
                 * Whether show sort after opening filer dialog.
                 */
                allowSort?: boolean;
                /**
                 * Whether show filter by value after opening filer dialog.
                 */
                allowFilterByValue?: boolean;
                /**
                 * Whether show filter by list after opening filer dialog. If allowSort, allowFilterByValue and allowFilterByList are all false, not show the filter button in this column.
                 */
                allowFilterByList?: boolean;
                /**
                 * The cross options of the column.
                 */
                cross?: string | GC.Data.ICrossOptions;
                /**
                 * Define the lookup for the column, only be used in the columns of the schema of the data source.
                 */
                lookup?: string | (string | number | boolean | Date)[] | GC.Data.ILookupOptions;
                /**
                 * Define the outline column only when the data be the hierarchy data.
                 */
                outlineColumn?: boolean | GC.Data.IOutlineColumnOptions;
                /**
                 * The trigger formula of the column.
                 */
                trigger?: GC.Data.ITriggerFormulaOption;
            }
        

        /**
         * @typedef GC.Data.IColumnCollection - The column collection.
         * @property {GC.Data.IColumn} key - A key-value collection, which key is a string, and value is a GC.Data.IColumn object.
         */
        export type IColumnCollection = 
            {
                [key: string]: GC.Data.IColumn;
            }
        

        /**
         * @typedef GC.Data.Icon
         * @property {string} position - The icon's position, which supports "left", "right", "leftOfText", "rightOfText", "outsideLeft", "outsideRight".
         * @property {number} width - The painted width.
         * @property {number} height - The painted height.
         * @property {string} src - The icon path.
         */
        export type Icon = 
            {
                /**
                 * The icon's position, which supports "left", "right", "leftOfText", "rightOfText", "outsideLeft", "outsideRight".
                 */
                position?: "left" | "right" | "leftOfText" | "rightOfText" | "outsideLeft" | "outsideRight";
                /**
                 * The painted width.
                 */
                width?: number;
                /**
                 * The painted height.
                 */
                height?: number;
                /**
                 * The icon path.
                 */
                src?: string;
            }
        

        /**
         * @typedef GC.Data.IconCriteriaOptions - The icon criteria options.
         * @property {boolean} isGreaterThanOrEqualTo - If set to true, use the greater than or equal to operator to calculate the value.
         * @property {GC.Data.IconValueType} iconValueType - The icon value type.
         * @property {number|string} iconValue - The icon value.
         */
        export type IconCriteriaOptions = 
            {
                /**
                 * If set to true, use the greater than or equal to operator to calculate the value.
                 */
                isGreaterThanOrEqualTo: boolean;
                /**
                 * The icon value type.
                 */
                iconValueType: GC.Data.IconValueType;
                /**
                 * The icon value.
                 */
                iconValue: number|string;
            }
        

        /**
         * @typedef GC.Data.IconOptions - The icon options.
         * @property {GC.Data.IconSetType} iconSetType - The icon value type.
         * @property {number} iconIndex - The custom icon index
         */
        export type IconOptions = 
            {
                /**
                 * The icon value type.
                 */
                iconSetType: GC.Data.IconSetType;
                /**
                 * The custom icon index
                 */
                iconIndex: number;
            }
        

        /**
         * @typedef GC.Data.IconSetRuleOptions - The options of icon set rule.
         * @property {"iconSetRule"} ruleType - The rule type if you want to use icon set rule.
         * @property {GC.Data.IconSetType} iconSetType - The icon set type.
         * @property {boolean} reverseIconOrder - Whether to reverse icon order.
         * @property {boolean} showIconOnly - Whether to display the icon only.
         * @property {GC.Data.IconOptions[]} icons - The array of icon options.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type IconSetRuleOptions = 
            {
                /**
                 * The rule type if you want to use icon set rule.
                 */
                ruleType: "iconSetRule";
                /**
                 * The icon set type.
                 */
                iconSetType: GC.Data.IconSetType;
                /**
                 * Whether to reverse icon order.
                 */
                reverseIconOrder: boolean;
                /**
                 * Whether to display the icon only.
                 */
                showIconOnly: boolean;
                /**
                 * The array of icon options.
                 */
                icons: GC.Data.IconOptions[];
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.IconSetType
         * @type {"threeArrowsColored"|"threeArrowsGray"|"threeTriangles"|"threeStars"|"threeFlags"|"threeTrafficLightsUnrimmed"|"threeTrafficLightsRimmed"|"threeSigns"|"threeSymbolsCircled"|"threeSymbolsUncircled"|"fourArrowsColored"|"fourArrowsGray"|"fourRedToBlack"|"fourRatings"|"fourTrafficLights"|"fiveArrowsColored"|"fiveArrowsGray"|"fiveRatings"|"fiveQuarters"|"fiveBoxes"|"noIcons"}
         * @description The icon set type.
         */
        export type IconSetType = 
            "threeArrowsColored"|"threeArrowsGray"|"threeTriangles"|"threeStars"|"threeFlags"|"threeTrafficLightsUnrimmed"|"threeTrafficLightsRimmed"|"threeSigns"|"threeSymbolsCircled"|"threeSymbolsUncircled"|"fourArrowsColored"|"fourArrowsGray"|"fourRedToBlack"|"fourRatings"|"fourTrafficLights"|"fiveArrowsColored"|"fiveArrowsGray"|"fiveRatings"|"fiveQuarters"|"fiveBoxes"|"noIcons"
        

        /**
         * @typedef GC.Data.IconValueType
         * @type {"number"|"percent"|"formula"|"percentile"}
         * @description The icon value type.
         */
        export type IconValueType = 
            "number"|"percent"|"formula"|"percentile"
        

        /**
         * @typedef GC.Data.ICrossAttributeOptions
         * @property {string} value - To define the header of cross column.
         * @property {string} formatter - To format the value.
         * @description The options to define the header of cross column.
         */
        export type ICrossAttributeOptions = 
            {
                /**
                 * To define the header of cross column.
                 */
                value: string;
                /**
                 * To format the value.
                 */
                formatter?: string;
            }
        

        /**
         * @typedef GC.Data.ICrossOptions
         * @property {string} over - The key to find cross column.
         * @property {(string | GC.Data.ICrossAttributeOptions)[]} attributes - The attributes to define multi headers of the cross column.
         * @property {string} filter - The formula to filter out the cross columns.
         * @property {string} caption - The self-defined header.
         * @description The options to define the cross column.
         */
        export type ICrossOptions = 
            {
                /**
                 * The key to find cross column.
                 */
                over: string;
                /**
                 * The attributes to define multi headers of the cross column.
                 */
                attributes?: (string | GC.Data.ICrossAttributeOptions)[];
                /**
                 * The formula to filter out the cross columns.
                 */
                filter?: string;
                /**
                 * The self-defined header.
                 */
                caption?: string;
            }
        

        /**
         * @typedef GC.Data.ICrossViewOptions - The cross view options.
         * @property {GC.Data.ICrossOptions} key - A key-value collection, which key is a string, and value is a GC.Data.ICrossOptions object.
         */
        export type ICrossViewOptions = 
            {
                [key: string]: GC.Data.ICrossOptions;
            }
        

        /**
         * @typedef GC.Data.IDataSourceOption - The data source options.
         * @property {Array | string} data - Local data source, which could be an object array for JSON, a string for CSV or XML.
         * @property {GC.Data.IRemoteTransportOption} remote - Remote data source, which supports REST API, OData, GraphQL.
         * @property {boolean} autoSync - In auto sync mode, data changes will be synced to server immediately.
         * @property {boolean} batch - In batch mode, data changes will be held in data source, user need invoke submitChanges method to sync server.
         * @property {GC.Data.ISchemaOption} schema - The data source schema options.
         */
        export type IDataSourceOption = 
            {
                /**
                 * Local data source, which could be an object array for JSON, a string for CSV or XML.
                 */
                data?: any;
                /**
                 * Remote data source, which supports REST API, OData, GraphQL.
                 */
                remote?: GC.Data.IRemoteTransportOption;
                /**
                 * In auto sync mode, data changes will be synced to server immediately.
                 */
                autoSync?: boolean;
                /**
                 * In batch mode, data changes will be held in data source, user need invoke submitChanges method to sync server.
                 */
                batch?: boolean;
                /**
                 * The data source schema options.
                 */
                schema?: GC.Data.ISchemaOption;
            }
        

        /**
         * @typedef GC.Data.IFormulaCollection - The formula collection.
         * @property {string} key - A key-value collection, which key is a string, and value is a string.
         */
        export type IFormulaCollection = 
            {
                [key: string]: string;
            }
        

        /**
         * @typedef GC.Data.IHierarchyCustomParseOptions - The options to parse the parentId for the custom hierarchy type.
         * @property {any} data - The data of the current record.
         * @property {number} index - The record index.
         */
        export type IHierarchyCustomParseOptions = 
            {
                /**
                 * The data of the current record.
                 */
                data: any;
                /**
                 * The record index.
                 */
                index: number;
            }
        

        /**
         * @typedef GC.Data.IHierarchyCustomUnparseOptions - The options to unparse the record to the id for the custom hierarchy type.
         * @property {any} data - The data of the current record.
         * @property {number} index - The record index.
         * @property {any} parentData - The data of the parent record.
         */
        export type IHierarchyCustomUnparseOptions = 
            {
                /**
                 * The data of the current record.
                 */
                data: any;
                /**
                 * The record index.
                 */
                index: number;
                /**
                 * The data of the parent record.
                 */
                parentData: any;
            }
        

        /**
         * @typedef GC.Data.IHierarchyOption - The hierarchy options.
         * @property {string} type - Supports 'Parent', 'ChildrenPath', 'Level', 'Custom'.
         * @property {string} column - The column that has the data to build hierarchy.
         * @property {string | GC.Data.IHierarchyOutlineColumnOptions} outlineColumn - The definitions of the outline column of the column that show as hierarchy.
         * @property {GC.Data.IHierarchySummaryFieldCollection} summaryFields - The definitions of the collections of hierarchy summary fields.
         * @property {GC.Data.HierarchyCustomParseHandler} parse - The options to parse the parentId for the custom hierarchy type.
         * @property {GC.Data.HierarchyCustomUnparseHandler} unparse - The options to unparse the record to the id for the custom hierarchy type.
         */
        export type IHierarchyOption = 
            {
                /**
                 * Supports 'Parent', 'ChildrenPath', 'Level', 'Custom'.
                 */
                type: 'Parent' | 'ChildrenPath' | 'Level' | 'Custom';
                /**
                 * The column that has the data to build hierarchy.
                 */
                column: string;
                /**
                 * The definitions of the outline column of the column that show as hierarchy.
                 */
                outlineColumn?: string | GC.Data.IHierarchyOutlineColumnOptions;
                /**
                 * The definitions of the collections of hierarchy summary fields.
                 */
                summaryFields?: GC.Data.IHierarchySummaryFieldCollection;
                /**
                 * The options to parse the parentId for the custom hierarchy type.
                 */
                parse?: GC.Data.HierarchyCustomParseHandler;
                /**
                 * The options to unparse the record to the id for the custom hierarchy type.
                 */
                unparse?: GC.Data.HierarchyCustomUnparseHandler;
            }
        

        /**
         * @typedef GC.Data.IHierarchyOutlineColumnOptions - The outline column options of the column that show as hierarchy.
         * @property {string} value - Specified the column that show as hierarchy.
         * @property {boolean} showCheckBox - Whether to display the check box.
         * @property {boolean} showImage - Whether to display images.
         * @property {string[]} images - The images by level (url or base64Image).
         * @property {boolean} showIndicator - Whether to display the indicator.
         * @property {string} expandIndicator - The expand indicator (url or base64Image).
         * @property {string} collapseIndicator - The collapse indicator (url or base64Image).
         */
        export type IHierarchyOutlineColumnOptions = 
            {
                /**
                 * Specified the column that show as hierarchy.
                 */
                value: string;
                /**
                 * Whether to display the check box.
                 */
                showCheckBox?: boolean;
                /**
                 * Whether to display images.
                 */
                showImage?: boolean;
                /**
                 * The images by level (url or base64Image).
                 */
                images?: string[];
                /**
                 * Whether to display the indicator.
                 */
                showIndicator?: boolean;
                /**
                 * The expand indicator (url or base64Image).
                 */
                expandIndicator?: string;
                /**
                 * The collapse indicator (url or base64Image).
                 */
                collapseIndicator?: string;
            }
        

        /**
         * @typedef GC.Data.IHierarchySummaryFieldCollection - The hierarchy summary field collection.
         * @property {string} key - A key-value collection, which key is a field string, and value is a formula string.
         */
        export type IHierarchySummaryFieldCollection = 
            {
                [key: string]: string;
            }
        

        /**
         * @typedef GC.Data.ILookupOptions - The lookup options.
         * @property {string} name - The relation ship name.
         * @property {string[] | GC.Data.IColumn[]} columns - The columns definitions of the related table, used for multi-column drop down.
         */
        export type ILookupOptions = 
            {
                /**
                 * The relation ship name.
                 */
                name: string;
                /**
                 * The columns definitions of the related table, used for multi-column drop down.
                 */
                columns: string[] | GC.Data.IColumn[];
            }
        

        /**
         * @typedef GC.Data.IOutlineColumnOptions - The outline column options of the column that show as hierarchy.
         * @property {boolean} showCheckBox - Whether to display the check box.
         * @property {boolean} showImage - Whether to display images.
         * @property {string[]} images - The images by level (url or base64Image).
         * @property {boolean} showIndicator - Whether to display the indicator.
         * @property {string} expandIndicator - The expand indicator (url or base64Image).
         * @property {string} collapseIndicator - The collapse indicator (url or base64Image).
         */
        export type IOutlineColumnOptions = 
            {
                /**
                 * Whether to display the check box.
                 */
                showCheckBox?: boolean;
                /**
                 * Whether to display images.
                 */
                showImage?: boolean;
                /**
                 * The images by level (url or base64Image).
                 */
                images?: string[];
                /**
                 * Whether to display the indicator.
                 */
                showIndicator?: boolean;
                /**
                 * The expand indicator (url or base64Image).
                 */
                expandIndicator?: string;
                /**
                 * The collapse indicator (url or base64Image).
                 */
                collapseIndicator?: string;
            }
        

        /**
         * @typedef {RequestInit} GC.Data.IRemoteFetchOption - The remote request options similar as RequestInit of the Fetch API.
         */
        export type IRemoteFetchOption = 
            RequestInit
        

        /**
         * @typedef GC.Data.IRemoteReadRequestOption - The remote request options for read operation.
         * @property {string} url - Request url.
         * @property {string} method - Request method, such as 'GET', 'POST', 'PUT', 'DELETE', default value is 'GET'.
         * @property {GC.Data.IRemoteFetchOption} options - Request options, similar as RequestInit of the Fetch API, optional.
         * @property {string} adapter - Adapt different data protocol, supports "rest", "odata", "odata4", "graphql".
         * @property {Object} body - Request body.
         */
        export type IRemoteReadRequestOption = 
            {
                /**
                 * Request url.
                 */
                url: string;
                /**
                 * Request method, such as 'GET', 'POST', 'PUT', 'DELETE', default value is 'GET'.
                 */
                method?: 'GET' | 'POST' | 'PUT' | 'DELETE';
                /**
                 * Request options, similar as RequestInit of the Fetch API, optional.
                 */
                options?: GC.Data.IRemoteFetchOption;
                /**
                 * Adapt different data protocol, supports "rest", "odata", "odata4", "graphql".
                 */
                adapter?: string;
                /**
                 * Request body.
                 */
                body?: Object;
            }
        

        /**
         * @typedef GC.Data.IRemoteRequestOption - The remote request options.
         * @property {string} url - Request url.
         * @property {string} method - Request method, such as 'GET', 'POST', 'PUT', 'DELETE', default value is 'GET'.
         * @property {GC.Data.IRemoteFetchOption} options - Request options, similar as RequestInit of the Fetch API, optional.
         */
        export type IRemoteRequestOption = 
            {
                /**
                 * Request url.
                 */
                url: string;
                /**
                 * Request method, such as 'GET', 'POST', 'PUT', 'DELETE', default value is 'GET'.
                 */
                method?: 'GET' | 'POST' | 'PUT' | 'DELETE';
                /**
                 * Request options, similar as RequestInit of the Fetch API, optional.
                 */
                options?: GC.Data.IRemoteFetchOption;
            }
        

        /**
         * @typedef GC.Data.IRemoteTransportOption - The remote transport options.
         * @property {GC.Data.IRemoteReadRequestOption | GC.Data.RemoteReadHandler} read - The options for read operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} create - The options for create operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} update - The options for update operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} delete - The options for delete operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} batch - The options for batch operation.
         * @property {GC.Data.IRemoteReadRequestOption | GC.Data.RemoteReadHandler} getColumns - The options for getting columns operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} addColumn - The options for adding column operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} updateColumn - The options for updating column operation.
         * @property {GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler} removeColumn - The options for deleting column operation.
         */
        export type IRemoteTransportOption = 
            {
                /**
                 * The options for read operation.
                 */
                read: GC.Data.IRemoteReadRequestOption | GC.Data.RemoteReadHandler;
                /**
                 * The options for create operation.
                 */
                create?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for update operation.
                 */
                update?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for delete operation.
                 */
                delete?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for batch operation.
                 */
                batch?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for getting columns operation.
                 */
                getColumns?: GC.Data.IRemoteReadRequestOption | GC.Data.RemoteReadHandler;
                /**
                 * The options for adding column operation.
                 */
                addColumn?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for updating column operation.
                 */
                updateColumn?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
                /**
                 * The options for deleting column operation.
                 */
                removeColumn?: GC.Data.IRemoteRequestOption | GC.Data.RemoteChangeHandler;
            }
        

        /**
         * @typedef GC.Data.ISchemaOption - The data source schema options.
         * @property {string} type - Supports 'json', 'csv', 'xml', 'columnJson'.
         * @property {string} dataPath - Data reader use this to get data source collection after parse, return entire parsed object if this is not specified.
         * @property {GC.Data.IColumnCollection} columns - The definitions of all columns of current table.
         * @property {GC.Data.IFormulaCollection} window - The definitions of all formulas of window chaining.
         * @property {GC.Data.IHierarchyOption} hierarchy - The definitions of hierarchy options.
         */
        export type ISchemaOption = 
            {
                /**
                 * Supports 'json', 'csv', 'xml', 'columnJson'.
                 */
                type?: string;
                /**
                 * Data reader use this to get data source collection after parse, return entire parsed object if this is not specified.
                 */
                dataPath?: string;
                /**
                 * The definitions of all columns of current table.
                 */
                columns?: GC.Data.IColumnCollection;
                /**
                 * The definitions of all formulas of window chaining.
                 */
                window?: GC.Data.IFormulaCollection;
                /**
                 * The definitions of hierarchy options.
                 */
                hierarchy?: GC.Data.IHierarchyOption;
            }
        

        /**
         * @typedef GC.Data.ItemOptions
         * @property {string} text - The item text.
         * @property {Object} value - The item value, could be a number, string, boolean, Date.
         */
        export type ItemOptions = 
            {
                /**
                 * The item text.
                 */
                text: string;
                /**
                 * The item value, could be a number, string, boolean, Date.
                 */
                value: any;
            }
        

        /**
         * @typedef GC.Data.ITriggerFormulaOption
         * @property {string} when - Indicates when to trigger the formula.
         * @property {string} formula - The formula to be triggered.
         * @property {string} fields - The watched fields which be updated could trigger the formula, "*" be set for any field, be omitted when on new.
         * @description The options to define the trigger formula.
         */
        export type ITriggerFormulaOption = 
            {
                /**
                 * Indicates when to trigger the formula.
                 */
                when: "onNew" | "onNewAndUpdate";
                /**
                 * The formula to be triggered.
                 */
                formula: string;
                /**
                 * The watched fields which be updated could trigger the formula, "*" be set for any field, be omitted when on new.
                 */
                fields?: string;
            }
        

        /**
         * @typedef GC.Data.LabelOptions
         * @property {string} alignment - The cell label position, which supports "topLeft", "topCenter", "topRight", "bottomLeft", "bottomCenter", "bottomRight".
         * @property {string} visibility - The cell label visibility, which supports "visible", "hidden", "auto".
         * @property {string} font - The cell label font.
         * @property {string} foreColor - The cell label foreground color.
         * @property {string} margin - The cell label margin.
         */
        export type LabelOptions = 
            {
                /**
                 * The cell label position, which supports "topLeft", "topCenter", "topRight", "bottomLeft", "bottomCenter", "bottomRight".
                 */
                alignment: "topLeft" | "topCenter" | "topRight" | "bottomLeft" | "bottomCenter" | "bottomRight";
                /**
                 * The cell label visibility, which supports "visible", "hidden", "auto".
                 */
                visibility: "visible" | "hidden" | "auto";
                /**
                 * The cell label font.
                 */
                font: string;
                /**
                 * The cell label foreground color.
                 */
                foreColor: string;
                /**
                 * The cell label margin.
                 */
                margin: string;
            }
        

        /**
         * @typedef GC.Data.LineBorder
         * @property {string} color - The border color.
         * @property {string} style - The border style, which supports "empty", "thin", "medium", "dashed", "dotted", "thick", "double", "hair", "mediumDashed", "dashDot", "mediumDashDot", "dashDotDot", "mediumDashDotDot", "slantedDashDot".
         */
        export type LineBorder = 
            {
                /**
                 * The border color.
                 */
                color: string;
                /**
                 * The border style, which supports "empty", "thin", "medium", "dashed", "dotted", "thick", "double", "hair", "mediumDashed", "dashDot", "mediumDashDot", "dashDotDot", "mediumDashDotDot", "slantedDashDot".
                 */
                style: "empty" | "thin" | "medium" | "dashed" | "dotted" | "thick" | "double" | "hair" | "mediumDashed" | "dashDot" | "mediumDashDot" | "dashDotDot" | "mediumDashDotDot" | "slantedDashDot";
            }
        

        /**
         * @typedef GC.Data.ListItem
         * @property {string} text - The item text.
         * @property {string} value - The item value.
         * @property {string} icon - The item icon.
         */
        export type ListItem = 
            {
                /**
                 * The item text.
                 */
                text?: string;
                /**
                 * The item value.
                 */
                value?: string;
                /**
                 * The item icon.
                 */
                icon?: string;
            }
        

        /**
         * @typedef GC.Data.ListLayout
         * @property {string} direction - The layout direction, which accepts "horizontal", "vertical".
         * @property {string} displayAs - The layout display, which accepts "inline", "popup", "tree".
         * @property {boolean} collapsible - Whether the list can be collapsed.
         */
        export type ListLayout = 
            {
                /**
                 * The layout direction, which accepts "horizontal", "vertical".
                 */
                direction?: "horizontal" | "vertical";
                /**
                 * The layout display, which accepts "inline", "popup", "tree".
                 */
                displayAs?: "inline" | "popup" | "tree";
                /**
                 * Whether the list can be collapsed.
                 */
                collapsible?: boolean;
            }
        

        /**
         * @typedef GC.Data.ListOptions
         * @property {string} text - The text.
         * @property {GC.Data.ListLayout} layout - The layout.
         * @property {boolean} multiSelect - Whether to set the list to multi-select.
         * @property {string} valueType - The cell value type of dropdown list result, which accepts "string", "array".
         * @property {GC.Data.ListItem[] | GC.Data.ListOptions[] | Function} items - The items. Usually, it accepts an array. Specially, when it accepts a function, the function should returns a DOM element.
         * @property {Function} onItemSelected - It is used when items or sub-items are selected.
         */
        export type ListOptions = 
            {
                /**
                 * The text.
                 */
                text?: string;
                /**
                 * The layout.
                 */
                layout?: GC.Data.ListLayout;
                /**
                 * Whether to set the list to multi-select.
                 */
                multiSelect?: boolean;
                /**
                 * The cell value type of dropdown list result, which accepts "string", "array".
                 */
                valueType?: "string" | "array";
                /**
                 * a function returns a DOM element
                 */
                items: GC.Data.ListItem[] | GC.Data.ListOptions[] | (() => HTMLElement);
                /**
                 * used when items or sub-items is (() => HTMLElement)
                 */
                onItemSelected?: (e: MouseEvent) => string;
            }
        

        /**
         * @typedef GC.Data.ListString
         * @type {string}
         * @description The list type string, such as "1,2,3".
         */
        export type ListString = 
            string
        

        /**
         * @typedef GC.Data.ListValidatorOptions
         * @property {"list"} type - The data validator type.
         * @property {GC.Data.ListString} source - The data validator source.
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type ListValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "list";
                /**
                 * The data validator source.
                 */
                source: GC.Data.ListString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.MaskType
         * @property {string} pattern - The mask pattern.
         * @property {string} placeholder - The mask placeholder.
         * @property {boolean} excludeLiteral - The mask exclude literal.
         * @property {boolean} excludePlaceholder - The mask exclude placeholder.
         */
        export type MaskType = 
            {
                /**
                 * The mask pattern.
                 */
                pattern: string;
                /**
                 * The mask placeholder.
                 */
                placeholder?: string;
                /**
                 * The mask exclude literal.
                 */
                excludeLiteral?: boolean;
                /**
                 * The mask exclude placeholder.
                 */
                excludePlaceholder?: boolean;
            }
        

        /**
         * @typedef GC.Data.MonthPickerOptions
         * @property {number} startYear - The month picker's start year, default value is ten year ago.
         * @property {number} stopYear - The month picker's stop year, default value is this year.
         * @property {number} height - The dom's height of month picker, default value is 300.
         */
        export type MonthPickerOptions = 
            {
                /**
                 * The month picker's start year, default value is ten year ago.
                 */
                startYear: number;
                /**
                 * The month picker's stop year, default value is this year.
                 */
                stopYear: number;
                /**
                 * The dom's height of month picker, default value is 300.
                 */
                height: number;
            }
        

        /**
         * @typedef GC.Data.MultiColumnOptions
         * @property {number} width - User can customize the width for multi-column dropdown. By default, it is the min value between 800 and the result of auto calculated all the column width.
         * @property {number} height - User can customize the height for multi-column dropdown. By default, it is the min value between 400 and the result of auto calculated all the row height.
         * @property {string | Object[]} dataSource - User should specify the dataSource for binding.The dataSource is required, supports array or formula which return an array.
         * @property {GC.Data.ColumnBindingInfo[]} bindingInfos - User can customize the bound column infos.
         */
        export type MultiColumnOptions = 
            {
                /**
                 * User can customize the width for multi-column dropdown. By default, it is the min value between 800 and the result of auto calculated all the column width.
                 */
                width?: number;
                /**
                 * User can customize the height for multi-column dropdown. By default, it is the min value between 400 and the result of auto calculated all the row height.
                 */
                height?: number;
                /**
                 * User should specify the dataSource for binding.The dataSource is required, supports array or formula which return an array.
                 */
                dataSource: string | any[];
                /**
                 * User can customize the bound column infos.
                 */
                bindingInfos?: GC.Data.ColumnBindingInfo[];
            }
        

        /**
         * @typedef GC.Data.NumberValidatorOptions
         * @property {"number"} type - The data validator type.
         * @property {GC.Data.CellValueComparisonOperator} comparisonOperator - The data validator comparison operator.
         * @property {number | GC.Data.FormulaString} value1 - The data validator first value.
         * @property {number | GC.Data.FormulaString} value2 - The data validator second value if validator comparison operator is "between" or "notBetween".
         * @property {boolean} isIntegerValue - Is it an integer value if the validator type if "number".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type NumberValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "number";
                /**
                 * The data validator comparison operator.
                 */
                comparisonOperator: GC.Data.CellValueComparisonOperator;
                /**
                 * The data validator first value.
                 */
                value1: number | GC.Data.FormulaString;
                /**
                 * The data validator second value if validator comparison operator is "between" or "notBetween".
                 */
                value2?: number | GC.Data.FormulaString;
                /**
                 * Is it an integer value if the validator type if "number".
                 */
                isIntegerValue: boolean;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.PatternFillOptions
         * @property {string} type - The pattern fill type, which supports "solid", "darkGray", "mediumGray", "lightGray", "gray125", "gray0625", "darkHorizontal", "darkVertical", "darkDown", "darkUp", "darkGrid", "darkTrellis", "lightHorizontal", "lightVertical", "lightDown", "lightUp", "lightGrid" and "lightTrellis".
         * @property {string} patternColor - the pattern fill color.
         * @property {string} backgroundColor - the pattern fill background color.
         */
        export type PatternFillOptions = 
            {
                /**
                 * The pattern fill type, which supports "solid", "darkGray", "mediumGray", "lightGray", "gray125", "gray0625", "darkHorizontal", "darkVertical", "darkDown", "darkUp", "darkGrid", "darkTrellis", "lightHorizontal", "lightVertical", "lightDown", "lightUp", "lightGrid" and "lightTrellis".
                 */
                type: "solid" | "darkGray" | "mediumGray" | "lightGray" | "gray125" | "gray0625" | "darkHorizontal" | "darkVertical" | "darkDown" | "darkUp" | "darkGrid" | "darkTrellis" | "lightHorizontal" | "lightVertical" | "lightDown" | "lightUp" | "lightGrid" | "lightTrellis";
                /**
                 * the pattern fill color.
                 */
                patternColor: string;
                /**
                 * the pattern fill background color.
                 */
                backgroundColor?: string;
            }
        

        /**
         * @typedef GC.Data.RadioButtonCheckboxListOptions
         * @property {string} type - The type of the cell type, supports "radioButtonList","checkboxList".
         * @property {GC.Data.ItemOptions[]} items - The items, which supports Object Array which each item contains text and value.
         * @property {string} direction - The the radio box or check box's orders, which support "horizontal", "vertical".
         * @property {number} maxColumnCount - The the radio box or check box's column count.
         * @property {number} maxRowCount - The the radio box or check box's row count.
         * @property {boolean} isFlowLayout - The the radio box or check box's layout is auto fit.
         * @property {Object} itemSpacing - The space for two items.
         * @property {number} itemSpacing.horizontal - The space for two items in horizontal direction.
         * @property {number} itemSpacing.vertical - The space for two items in vertical direction.
         * @property {string} textAlign - The text alignment relative to the radio box or check box, which supports "left", "right".
         * @property {number} boxSize - The radio box or check box size.
         */
        export type RadioButtonCheckboxListOptions = 
            {
                /**
                 * The type of the cell type, supports "radioButtonList","checkboxList".
                 */
                type: "radioButtonList" | "checkboxList";
                /**
                 * The items, which supports Object Array which each item contains text and value.
                 */
                items: GC.Data.ItemOptions[];
                /**
                 * The the radio box or check box's orders, which support "horizontal", "vertical".
                 */
                direction: "horizontal" | "vertical";
                /**
                 * The the radio box or check box's column count.
                 */
                maxColumnCount: number;
                /**
                 * The the radio box or check box's row count.
                 */
                maxRowCount: number;
                /**
                 * The the radio box or check box's layout is auto fit.
                 */
                isFlowLayout: boolean;
                /**
                 * The space for two items.
                 */
                itemSpacing: { horizontal: number; vertical: number };
                /**
                 * The text alignment relative to the radio box or check box, which supports "left", "right".
                 */
                textAlign: "left" | "right";
                /**
                 * The radio box or check box size.
                 */
                boxSize: number;
            }
        

        /**
         * @typedef GC.Data.RemoteChangeHandler
         * @param {Object | Object[]} data - the data changes.
         * @param {GC.Data.IRemoteFetchOption} options - the options for the handler, similar as RequestInit of the Fetch API.
         * @returns {Promise<Object | Object[]>} Returns of the handler.
         * @description handle data changes.
         */
        export type RemoteChangeHandler = 
            (data: Object | Object[], options: GC.Data.IRemoteFetchOption) => Promise<Object | Object[]>
        

        /**
         * @typedef GC.Data.RemoteReadHandler
         * @param {GC.Data.IRemoteFetchOption} options - the options for the handler, similar as RequestInit of the Fetch API.
         * @returns {Promise<Object>} Returns of the handler.
         * @description handle reading data.
         */
        export type RemoteReadHandler = 
            (options: Object) => Promise<Object[]>
        

        /**
         * @typedef GC.Data.ScaleValueType
         * @type {"number"|"lowestValue"|"highestValue"|"percent"|"percentile"|"automin"|"formula"|"automax"}
         * @description The scale value type.
         */
        export type ScaleValueType = 
            "number"|"lowestValue"|"highestValue"|"percent"|"percentile"|"automin"|"formula"|"automax"
        

        /**
         * @typedef GC.Data.SliderOptions
         * @property {boolean} scaleVisible - Whether the thumb can drag over tick only, default value is false.
         * @property {number} max - The maximum value the slider can slide to, default value is 100.
         * @property {number} min - The minimum value the slider can slide to, default value is 0.
         * @property {number} step - The granularity that the slider can step through values. Must greater than 0, and be divided by (max - min). When marks no null, step can be null, default value is 1.
         * @property {boolean} tooltipVisible - If true, tooltip will show always, or it will not show anyway, even if dragging or hovering, default value is false.
         * @property {number} width - The slider's width when direction is horizontal, default value is 350.
         * @property {number} height - The slider's height when direction is vertical, default value is 350.
         * @property {string} direction - The direction of slider, which accepts "horizontal", "vertical", and default value is horizontal.
         * @property {number[]} marks - The tick mark of slider.
         * @property {string} formatString - The display formatter, that used to tooltip or marks.
         */
        export type SliderOptions = 
            {
                /**
                 * Whether the thumb can drag over tick only, default value is false.
                 */
                scaleVisible: boolean;
                /**
                 * The maximum value the slider can slide to, default value is 100.
                 */
                max: number;
                /**
                 * The minimum value the slider can slide to, default value is 0.
                 */
                min: number;
                /**
                 * The granularity that the slider can step through values. Must greater than 0, and be divided by (max - min). When marks no null, step can be null, default value is 1.
                 */
                step: number;
                /**
                 * If true, tooltip will show always, or it will not show anyway, even if dragging or hovering, default value is false.
                 */
                tooltipVisible: boolean;
                /**
                 * The slider's width when direction is horizontal, default value is 350.
                 */
                width: number;
                /**
                 * The slider's height when direction is vertical, default value is 350.
                 */
                height: number;
                /**
                 * The direction of slider, which accepts "horizontal", "vertical", and default value is horizontal.
                 */
                direction: "horizontal" | "vertical";
                /**
                 * The tick mark of slider.
                 */
                marks: number[];
                /**
                 * The display formatter, that used to tooltip or marks.
                 */
                formatString: string;
            }
        

        /**
         * @typedef GC.Data.SpecificTextComparisonOperator
         * @type {"contains"|"doesNotContain"|"beginsWith"|"endsWith"}
         * @description The specific text comparison operator.
         */
        export type SpecificTextComparisonOperator = 
            "contains"|"doesNotContain"|"beginsWith"|"endsWith"
        

        /**
         * @typedef GC.Data.SpecificTextRuleOptions - The options of specific text rule.
         * @property {"specificTextRule"} ruleType - The rule type if you want to use specific text rule.
         * @property {GC.Data.SpecificTextComparisonOperator} comparisonOperator - The comparison operator of specific text.
         * @property {string} text - The specific text.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type SpecificTextRuleOptions = 
            {
                /**
                 * The rule type if you want to use specific text rule.
                 */
                ruleType: "specificTextRule";
                /**
                 * The comparison operator of specific text.
                 */
                comparisonOperator: GC.Data.SpecificTextComparisonOperator;
                /**
                 * The specific text.
                 */
                text: string;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.StateRule
         * @property {GC.Data.StateRuleDirection} direction - The state rule direction.
         * @property {GC.Data.RowColumnStates} state - The state.
         * @property {GC.Data.ViewArea} [area] - The rule area.
         * @description The state rule.
         */
        export type StateRule = 
            {
                /**
                 * the state rule direction
                 */
                direction: GC.Data.StateRuleDirection;
                /**
                 * the formula string
                 */
                state: GC.Data.RowColumnStates;
                /**
                 * the position where the rule takes an effect
                 */
                area?: GC.Data.ViewArea;
            }
        

        /**
         * @typedef GC.Data.StyleOptions
         * @property {string | GC.Data.PatternFillOptions | GC.Data.GradientFillOptions | GC.Data.GradientPathFillOptions} backColor - The background color string or pattern fill options, gradient fill options, gradient path fill options.
         * @property {string} foreColor - The foreground color.
         * @property {string} hAlign - The horizontal alignment, which supports "left", "center", "right", "general".
         * @property {string} vAlign - The vertical alignment, which supports "top", "center", "bottom".
         * @property {string} font - The font.
         * @property {string} themeFont - The theme font.
         * @property {string} formatter - The formatter string.
         * @property {GC.Data.LineBorder} borderLeft - The left border.
         * @property {GC.Data.LineBorder} borderTop - The top border.
         * @property {GC.Data.LineBorder} borderRight - The right border.
         * @property {GC.Data.LineBorder} borderBottom - The bottom border.
         * @property {GC.Data.LineBorder} diagonalDown - The diagonal with LeftTop to bottomRight.
         * @property {GC.Data.LineBorder} diagonalUp - The diagonal with topRight to bottomLeft.
         * @property {boolean} locked - Whether the cell, row, or column is locked.
         * @property {number} textIndent - The text indent amount.
         * @property {boolean} wordWrap - Whether words wrap within the cell or cells.
         * @property {boolean} shrinkToFit - Whether content shrinks to fit the cell or cells.
         * @property {string} backgroundImage - The background image to display.
         * @property {GC.Data.CheckboxOptions | GC.Data.ComboBoxOptions | GC.Data.HyperlinkOptions | GC.Data.FileUploadOptions | GC.Data.RadioButtonCheckboxListOptions} cellType - The cell type.
         * @property {string} backgroundImageLayout - The layout for the background image, which supports "stretch", "center", "zoom", "none".
         * @property {boolean} tabStop - Whether the user can set focus to the cell using the Tab key.
         * @property {GC.Data.TextDecoration} textDecoration - The decoration added to text.
         * @property {string} imeMode - The input method editor mode, which supports "auto", "active", "inactive", "disabled".
         * @property {string} name - The name.
         * @property {string} parentName - The name of the parent style.
         * @property {string} watermark - The watermark content.
         * @property {string} cellPadding - The cell padding.
         * @property {GC.Data.LabelOptions} labelOptions - The cell label options.
         * @property {boolean} isVerticalText - Whether to set the cell's text vertical.
         * @property {number} textOrientation - The cell text rotation angle.
         * @property {boolean} showEllipsis - Whether the text out of bounds shows ellipsis.
         * @property {GC.Data.CellButtonOptions[]} cellButtons - The cell buttons.
         * @property {GC.Data.DropDownOptions[]} dropDowns - The drop downs.
         * @property {GC.Data.Decoration} decoration - The decoration.
         * @property {GC.Data.MaskType} mask - The mask.
         */
        export type StyleOptions = 
            {
                /**
                 * The background color string or pattern fill options, gradient fill options, gradient path fill options.
                 */
                backColor?: string | GC.Data.PatternFillOptions | GC.Data.GradientFillOptions | GC.Data.GradientPathFillOptions;
                /**
                 * The foreground color.
                 */
                foreColor?: string;
                /**
                 * The horizontal alignment, which supports "left", "center", "right", "general".
                 */
                hAlign?: "left" | "center" | "right" | "general";
                /**
                 * The vertical alignment, which supports "top", "center", "bottom".
                 */
                vAlign?: "top" | "center" | "bottom";
                /**
                 * The font.
                 */
                font?: string;
                /**
                 * The theme font.
                 */
                themeFont?: string;
                /**
                 * The formatter string.
                 */
                formatter?: string;
                /**
                 * The left border.
                 */
                borderLeft?: GC.Data.LineBorder;
                /**
                 * The top border.
                 */
                borderTop?: GC.Data.LineBorder;
                /**
                 * The right border.
                 */
                borderRight?: GC.Data.LineBorder;
                /**
                 * The bottom border.
                 */
                borderBottom?: GC.Data.LineBorder;
                /**
                 * The diagonal with LeftTop to bottomRight.
                 */
                diagonalDown?: GC.Data.LineBorder;
                /**
                 * The diagonal with topRight to bottomLeft.
                 */
                diagonalUp?: GC.Data.LineBorder;
                /**
                 * Whether the cell, row, or column is locked.
                 */
                locked?: boolean;
                /**
                 * The text indent amount.
                 */
                textIndent?: number;
                /**
                 * Whether words wrap within the cell or cells.
                 */
                wordWrap?: boolean;
                /**
                 * Whether content shrinks to fit the cell or cells.
                 */
                shrinkToFit?: boolean;
                /**
                 * The background image to display.
                 */
                backgroundImage?: string;
                /**
                 * The cell type.
                 */
                cellType?: GC.Data.CheckboxOptions | GC.Data.ComboBoxOptions | GC.Data.HyperlinkOptions | GC.Data.FileUploadOptions | GC.Data.RadioButtonCheckboxListOptions;
                /**
                 * The layout for the background image, which supports "stretch", "center", "zoom", "none".
                 */
                backgroundImageLayout?: "stretch" | "center" | "zoom" | "none";
                /**
                 * Whether the user can set focus to the cell using the Tab key.
                 */
                tabStop?: boolean;
                /**
                 * The decoration added to text.
                 */
                textDecoration?: GC.Data.TextDecoration;
                /**
                 * The input method editor mode, which supports "auto", "active", "inactive", "disabled".
                 */
                imeMode?: "auto" | "active" | "inactive" | "disabled";
                /**
                 * The name.
                 */
                name?: string;
                /**
                 * The name of the parent style.
                 */
                parentName?: string;
                /**
                 * The watermark content.
                 */
                watermark?: string;
                /**
                 * The cell padding.
                 */
                cellPadding?: string;
                /**
                 * The cell label options.
                 */
                labelOptions?: GC.Data.LabelOptions;
                /**
                 * Whether to set the cell's text vertical.
                 */
                isVerticalText?: boolean;
                /**
                 * The cell text rotation angle.
                 */
                textOrientation?: number;
                /**
                 * Whether the text out of bounds shows ellipsis.
                 */
                showEllipsis?: boolean;
                /**
                 * The cell buttons.
                 */
                cellButtons?: GC.Data.CellButtonOptions[];
                /**
                 * The drop downs.
                 */
                dropDowns?: GC.Data.DropDownOptions[];
                /**
                 * The decoration.
                 */
                decoration?: GC.Data.Decoration;
                /**
                 * The mask.
                 */
                mask?: GC.Data.MaskType;
            }
        

        /**
         * @typedef GC.Data.StyleRule
         * @property {GC.Data.FormulaRule | GC.Data.StateRule} [rule] - The style rule.
         * @property {GC.Data.StyleOptions} [style] - The style rule style.
         * @description The style rule.
         */
        export type StyleRule = 
            {
                /**
                 * the formula rule or the state rule
                 */
                rule?: GC.Data.FormulaRule | GC.Data.StateRule;
                /**
                 * the style if the rule is satisfied
                 */
                style?: GC.Data.StyleOptions;
            }
        

        /**
         * @typedef GC.Data.StyleRules
         * @property {GC.Data.StyleRule} name - The style rule.
         * @description The style rules.
         */
        export type StyleRules = 
            {
                /**
                 * the style rule
                 */
                name: GC.Data.StyleRule
            }
        

        /**
         * @typedef GC.Data.TextDecoration
         * @type {"underline" | "lineThrough" | "overline" | "doubleUnderline" | "none" | "lineThroughUnderline" | "lineThroughDoubleUnderline"}
         * @description The text decoration.
         */
        export type TextDecoration = 
            "underline" | "lineThrough" | "overline" | "doubleUnderline" | "none" | "lineThroughUnderline" | "lineThroughDoubleUnderline"
        

        /**
         * @typedef GC.Data.TextLengthString
         * @type {string}
         * @description The text length type string, such as "10".
         */
        export type TextLengthString = 
            string
        

        /**
         * @typedef GC.Data.TextLengthValidatorOptions
         * @property {"textLength"} type - The data validator type.
         * @property {GC.Data.CellValueComparisonOperator} comparisonOperator - The data validator comparison operator.
         * @property {GC.Data.TextLengthString | GC.Data.FormulaString} value1 - The data validator first value.
         * @property {GC.Data.TextLengthString | GC.Data.FormulaString} value2 - The data validator second value if validator comparison operator is "between" or "notBetween".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type TextLengthValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "textLength";
                /**
                 * The data validator comparison operator.
                 */
                comparisonOperator: GC.Data.CellValueComparisonOperator;
                /**
                 * The data validator first value.
                 */
                value1: GC.Data.TextLengthString | GC.Data.FormulaString;
                /**
                 * The data validator second value if validator comparison operator is "between" or "notBetween".
                 */
                value2?: GC.Data.TextLengthString | GC.Data.FormulaString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.ThreeScaleRuleOptions - The options of three scale rule.
         * @property {"threeScaleRule"} ruleType - The rule type if you want to use three scale rule.
         * @property {GC.Data.ScaleValueType} minType - The minimum scale type.
         * @property {number} minValue - The minimum scale value.
         * @property {GC.Data.ColorString} minColor - The minimum scale color string.
         * @property {GC.Data.ScaleValueType} midType - The midpoint scale type.
         * @property {number} midValue - The midpoint scale value.
         * @property {GC.Data.ColorString} midColor - The midpoint scale color string.
         * @property {GC.Data.ScaleValueType} maxType - The maximum scale type.
         * @property {number} maxValue - The maximum scale value.
         * @property {GC.Data.ColorString} maxColor - The maximum scale color string.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type ThreeScaleRuleOptions = 
            {
                /**
                 * The rule type if you want to use three scale rule.
                 */
                ruleType: "threeScaleRule";
                /**
                 * The minimum scale type.
                 */
                minType: GC.Data.ScaleValueType;
                /**
                 * The minimum scale value.
                 */
                minValue: number;
                /**
                 * The minimum scale color string.
                 */
                minColor: GC.Data.ColorString;
                /**
                 * The midpoint scale type.
                 */
                midType: GC.Data.ScaleValueType;
                /**
                 * The midpoint scale value.
                 */
                midValue: number;
                /**
                 * The midpoint scale color string.
                 */
                midColor: GC.Data.ColorString;
                /**
                 * The maximum scale type.
                 */
                maxType: GC.Data.ScaleValueType;
                /**
                 * The maximum scale value.
                 */
                maxValue: number;
                /**
                 * The maximum scale color string.
                 */
                maxColor: GC.Data.ColorString;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.TimePickerOptions
         * @property {GC.Data.TimePickerValue} max - The maximum value the time picker can display, the value need hour, minute and second attribute.
         * @property {GC.Data.TimePickerValue} min - The minimum value the time picker can display, the value need hour, minute and second attribute.
         * @property {GC.Data.TimePickerValue} step - The granularity that the time picker can step through values.
         * @property {string} formatString - The format of the time item.
         * @property {number} height - The container of the time picker's size.
         */
        export type TimePickerOptions = 
            {
                /**
                 * The maximum value the time picker can display, the value need hour, minute and second attribute.
                 */
                max: GC.Data.TimePickerValue;
                /**
                 * The minimum value the time picker can display, the value need hour, minute and second attribute.
                 */
                min: GC.Data.TimePickerValue;
                /**
                 * The granularity that the time picker can step through values.
                 */
                step: GC.Data.TimePickerValue;
                /**
                 * The format of the time item.
                 */
                formatString: string;
                /**
                 * The container of the time picker's size.
                 */
                height: number;
            }
        

        /**
         * @typedef GC.Data.TimePickerValue
         * @property {number} hour - The hour of the value.
         * @property {number} minute - The minute of the value.
         * @property {number} second - The second of the value.
         */
        export type TimePickerValue = 
            {
                /**
                 * The hour of the value.
                 */
                hour: number;
                /**
                 * The minute of the value.
                 */
                minute: number;
                /**
                 * The second of the value.
                 */
                second: number;
            }
        

        /**
         * @typedef GC.Data.TimeValidatorOptions
         * @property {"time"} type - The data validator type.
         * @property {GC.Data.CellValueComparisonOperator} comparisonOperator - The data validator comparison operator.
         * @property {Date | GC.Data.FormulaString} value1 - The data validator first value.
         * @property {Date | GC.Data.FormulaString} value2 - The data validator second value if validator comparison operator is "between" or "notBetween".
         * @property {boolean} ignoreBlank - Whether to ignore empty value.
         * @property {boolean} inCellDropdown - Whether to display a drop-down button.
         * @property {GC.Data.ErrorStyle} errorStyle - The data validator error style.
         * @property {string} errorTitle - The data validator error title.
         * @property {string} errorMessage - The data validator error message.
         * @property {boolean} showErrorMessage - Whether to show error message.
         * @property {string} inputMessage - The data validator input message.
         * @property {string} inputTitle - The data validator input title.
         * @property {boolean} showInputMessage - Whether to show input message.
         * @property {GC.Data.HighlightStyle} highlightStyle - The data validator highlight style.
         */
        export type TimeValidatorOptions = 
            {
                /**
                 * The data validator type.
                 */
                type: "time";
                /**
                 * The data validator comparison operator.
                 */
                comparisonOperator: GC.Data.CellValueComparisonOperator;
                /**
                 * The data validator first value.
                 */
                value1: Date | GC.Data.FormulaString;
                /**
                 * The data validator second value if validator comparison operator is "between" or "notBetween".
                 */
                value2?: Date | GC.Data.FormulaString;
                /**
                 * The data validator error style.
                 */
                errorStyle: GC.Data.ErrorStyle;
                /**
                 * The data validator error title.
                 */
                errorTitle: string;
                /**
                 * The data validator error message.
                 */
                errorMessage: string;
                /**
                 * The data validator highlight style.
                 */
                highlightStyle: GC.Data.HighlightStyle;
                /**
                 * Whether to ignore empty value.
                 */
                ignoreBlank: boolean;
                /**
                 * Whether to display a drop-down button.
                 */
                inCellDropdown: boolean;
                /**
                 * The data validator input message.
                 */
                inputMessage: string;
                /**
                 * The data validator input title.
                 */
                inputTitle: string;
                /**
                 * Whether to show error message.
                 */
                showErrorMessage: boolean;
                /**
                 * Whether to show input message.
                 */
                showInputMessage: boolean;
            }
        

        /**
         * @typedef GC.Data.Top10RuleOptions - The options of top-10 rule.
         * @property {"top10Rule"} ruleType - The rule type if you want to use top-10 rule.
         * @property {GC.Data.Top10Type} type - The top-10 type.
         * @property {number} rank - The number of top or bottom items to apply the style to.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type Top10RuleOptions = 
            {
                /**
                 * The rule type if you want to use top-10 rule.
                 */
                ruleType: "top10Rule";
                /**
                 * The top-10 type.
                 */
                type: GC.Data.Top10Type;
                /**
                 * The number of top or bottom items to apply the style to.
                 */
                rank: number;
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.Top10Type
         * @type {"top"|"bottom"}
         * @description The top-10 type.
         */
        export type Top10Type = 
            "top"|"bottom"
        

        /**
         * @typedef GC.Data.TwoScaleRuleOptions - The options of two scale rule.
         * @property {"twoScaleRule"} ruleType - The rule type if you want to use two scale rule.
         * @property {GC.Data.ScaleValueType} minType - The minimum scale type.
         * @property {number} minValue - The minimum scale value.
         * @property {GC.Data.ColorString} minColor - The minimum scale color string.
         * @property {GC.Data.ScaleValueType} maxType - The maximum scale type.
         * @property {number} maxValue - The maximum scale value.
         * @property {GC.Data.ColorString} maxColor - The maximum scale color string.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type TwoScaleRuleOptions = 
            {
                /**
                 * The rule type if you want to use two scale rule.
                 */
                ruleType: "twoScaleRule";
                /**
                 * The minimum scale type.
                 */
                minType: GC.Data.ScaleValueType;
                /**
                 * The minimum scale value.
                 */
                minValue: number;
                /**
                 * The minimum scale color string.
                 */
                minColor: GC.Data.ColorString;
                /**
                 * The maximum scale type.
                 */
                maxType: GC.Data.ScaleValueType;
                /**
                 * The maximum scale value.
                 */
                maxValue: number;
                /**
                 * The maximum scale color string.
                 */
                maxColor: GC.Data.ColorString;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.UniqueRuleOptions - The options of unique rule.
         * @property {"uniqueRule"} ruleType - The rule type if you want to use unique rule.
         * @property {GC.Data.StyleOptions} style - The style that is applied to the cell when the condition is met.
         * @property {number} priority - The priority of the rule.
         * @property {boolean} stopIfTrue - Whether rules with lower priority are applied before this rule.
         */
        export type UniqueRuleOptions = 
            {
                /**
                 * The rule type if you want to use unique rule.
                 */
                ruleType: "uniqueRule";
                /**
                 * The style that is applied to the cell when the condition is met.
                 */
                style: GC.Data.StyleOptions;
                /**
                 * The priority of the rule.
                 */
                priority: number;
                /**
                 * Whether rules with lower priority are applied before this rule.
                 */
                stopIfTrue: boolean;
            }
        

        /**
         * @typedef GC.Data.ViewOptions
         * @property {number} [defaultColumnWidth] - The default column width.
         * @property {GC.Data.StyleRules} [styleRules] - The style rules.
         * @property {boolean} [showCrossValueHeader] - mark the cross value headers are visible.
         * @property {GC.Data.ICrossViewOptions} [cross] - define the cross column options.
         * @description The options for the view.
         */
        export type ViewOptions = 
            {
                /**
                 * the default column width
                 */
                defaultColumnWidth?: number;
                /**
                 * the style rules
                 */
                styleRules?: GC.Data.StyleRules;
                /**
                 * mark the cross value headers are visible.
                 */
                showCrossValueHeader?: boolean;
                /**
                 * define the cross column options.
                 */
                cross?: GC.Data.ICrossViewOptions;
            }
        

        /**
         * @typedef GC.Data.WorkFlowItem
         * @property {string} value - The item value.
         * @property {number[]|string[]} transitions - The item transitions.
         */
        export type WorkFlowItem = 
            {
                /**
                 * The item value.
                 */
                value: string;
                /**
                 * The item transitions.
                 */
                transitions: number[]|string[];
            }
        

        /**
         * @typedef GC.Data.WorkFlowOptions
         * @property {GC.Data.WorkFlowItem[]} items - The items.
         */
        export type WorkFlowOptions = 
            {
                /**
                 * The items.
                 */
                items: GC.Data.WorkFlowItem[];
            }
        
        /**
         * Specifies the type of row and column state.
         * @enum {number}
         */
        export enum RowColumnStates{
            /** When mouse hover on the row and the column , its state include "hover" state.
             * @type {number}
             */
            hover= 1,
            /** When the column is locked, its state include "readonly" state. This state only support for column.
             * @type {number}
             */
            readonly= 4,
            /** When the row or the column is focus, its state include "active" state.
             * @type {number}
             */
            active= 16,
            /** When the cell is in the selection range, the cell row and column state include "selected" state.
             * @type {number}
             */
            selected= 32,
            /** When cell value is changed, cell row and column state include "dirty" state.
             * @type {number}
             */
            dirty= 64,
            /** When inserted a row, its state include "inserted" state. This state only support for row.
             * @type {number}
             */
            inserted= 128,
            /** When updated a row, its state include "updated" state. This state only support for row.
             * @type {number}
             */
            updated= 256,
            /** When pin a row/column, its state include "pin" state.
             * @type {number}
             */
            pin= 1024,
            /** When the column has primary key, its state include "primaryKey" state. This state only support for column.
             * @type {number}
             */
            primaryKey= 2048,
            /** When the value of the column are required, its state include "required" state. This state only support for column.
             * @type {number}
             */
            required= 4096
        }

        /**
         * Defines the direction in which the style rule is applied.
         * @enum {number}
         * @example
         * ```javascript
         * view.addStyleRule("hoverRowStyle", GC.Data.View.StateRuleDirection, { backColor: "green" }, { state: GC.Data.RowColumnStates.hover });
         * ```
         */
        export enum StateRuleDirection{
            /** Specifies the style rule will not be applied.
             * @type {number}
             */
            none= 0,
            /** Specifies the style rule will apply to the row direction.
             * @type {number}
             */
            row= 1,
            /** Specifies the style rule will apply to the column direction.
             * @type {number}
             */
            column= 2,
            /** Specifies the style rule will apply to both row and column direction.
             * @type {number}
             */
            both= 3
        }

        /**
         * Specifies the view area.
         * @enum {number}
         */
        export enum ViewArea{
            /** Indicates the column header.
             * @type {number}
             */
            colHeader= 1,
            /** Indicates the viewport.
             * @type {number}
             */
            viewport= 3
        }


        export class DataManager{
            /**
             * Represents the data manager.
             * @class
             * @example
             * ```javascript
             * // Create a data manager
             * var dataManager = new GC.Data.DataManager();
             * ```
             */
            constructor();
            /**
             * Represents the relationship array. Each relationship includes the following field,
             * @property {GC.Data.Table} sourceTable - The source table.
             * @property {string} sourceFieldName - The field name of the source table.
             * @property {string} sourceRelationshipName - The relationship name which can be used in source table.
             * @property {GC.Data.Table} targetTable - The target table.
             * @property {string} targetFieldName - The field name of the target table.
             * @property {string} targetRelationshipName - The relationship name which can be used in target table.
             * @type {Object[]}
             */
            relationships: GC.Data.IRelationship[];
            /**
             * Represents the table collection. Its key is table name, and value is GC.Data.Table instance.
             * @type {Object}
             */
            tables: GC.Data.ITables;
            /**
             * Adds a relationship into the data manager.
             * @param {GC.Data.Table} sourceTable - The source table, which foreign key is target table's primary key.
             * @param {string} sourceFieldName - The source field name.
             * @param {string} sourceRelationshipName - The source relationship name.
             * @param {GC.Data.Table} targetTable - The target table, which primary key is source table's foreign key.
             * @param {string} targetFieldName - The target field name.
             * @param {string} targetRelationshipName - The target relationship name.
             * @returns {Object} Returns the relationship.
             * @example
             * ```javascript
             * // Add relationship between products table and categories table
             * dataManager.addRelationship(productTable, "categoryId", "categories", categoriesTable, "id", "products");
             * ```
             */
            addRelationship(sourceTable: GC.Data.Table,  sourceFieldName: string,  sourceRelationshipName: string,  targetTable: GC.Data.Table,  targetFieldName: string,  targetRelationshipName: string): GC.Data.IRelationship;
            /**
             * Add a table into the data manager.
             * @param {string} name - The table name.
             * @param {GC.Data.IDataSourceOption} dataSourceOption - The data source option for creating a table, which contains the following properties.
             * @returns {GC.Data.Table} Returns the table.
             * @example
             * ```javascript
             * // Add a sample table to read data
             * var tableName = "products";
             * var dataSourceOption = {
             *      remote: {
             *         read: {
             *             url: "https://demodata.mescius.io/northwind/api/v1/orders"
             *         }
             *     },
             *     schema: {
             *         columns: {
             *             orderId: {dataName: "id"},
             *             orderDate: {dataType: "date", dataPattern: "yyyy-MM-dd hh:mm:ss.000"},
             *             requiredDate: {dataType: "date"},
             *             shippedDate: {dataType: "date"},
             *             shipVia: {dataMap: {1: "Speedy Express", 2: "United Package", 3: "Federal Shipping"}}
             *         }
             *     }
             * };
             * var dataManager = new GC.Data.DataManager();
             * var productTable = dataManager.addTable(tableName, dataSourceOption);
             * ```
             */
            addTable(name: string,  dataSourceOption: GC.Data.IDataSourceOption): GC.Data.Table;
            /**
             * Removes a relationship from the data manager by source relationship name.
             * @param {string} name - The source relationship name.
             * @example
             * ```javascript
             * // Remove a relationship from the data manager by source relationship name
             * dataManager.removeRelationship("categories");
             * ```
             */
            removeRelationship(name: string): void;
            /**
             * Removes a table from the data manager.
             * @param {string} name - The table name.
             * @example
             * ```javascript
             * // Remove a table from data manager by table name
             * dataManager.removeTable("products");
             * ```
             */
            removeTable(name: string): void;
        }

        export class Table{
            /**
             * Represents the table.
             * @class
             * @param {string} name - The table name.
             * @param {GC.Data.IDataSourceOption} dataSourceOption - The data source of table.
             */
            constructor(name: string,  dataSourceOption: GC.Data.IDataSourceOption);
            /**
             * Represents the default columns of the table, it is only available after table is fetched. The key is column name, the value is column information.
             * @type {Object}
             */
            columns: GC.Data.IColumnCollection;
            /**
             * Represents the name of the table.
             * @type {string}
             */
            name: string;
            /**
             * Represents the data source options of the table.
             * @type {GC.Data.IDataSourceOption}
             */
            options: GC.Data.IDataSourceOption;
            /**
             * Represents the view collection of the table. The key is view name, the value is GC.Data.View instance.
             * @type {Object}
             */
            views: GC.Data.IViews;
            /**
             * Adds a view, which host table is current table.
             * @param {string} name - The view name.
             * @param {string[] | Object[]} [columnInfos] - The column information, each column includes the following properties.
             * @param {boolean} includeDefaultColumns - Whether to include current table's default columns when column information are empty. Its default value is true.
             * @param {GC.Data.ViewOptions} [options] - The view options.
             * @property {string} name - The unique name of the column.
             * @property {string} [value] - The value of the column, could be a field name of table from database, or formula which uses the fields names.
             * @property {string | string[]} [caption] - The caption of the column.
             * @property {number | string} [width] - The width of the column, support number in pixel, or star size.
             * @property {GC.Data.StyleOptions} [style] - The column style options.
             * @property {(GC.Data.CellValueRuleOptions | GC.Data.SpecificTextRuleOptions | GC.Data.FormulaRuleOptions | GC.Data.DateOccurringRuleOptions | GC.Data.Top10RuleOptions | GC.Data.UniqueRuleOptions | GC.Data.DuplicateRuleOptions | GC.Data.AverageRuleOptions | GC.Data.TwoScaleRuleOptions | GC.Data.ThreeScaleRuleOptions | GC.Data.DataBarRuleOptions | GC.Data.IconSetRuleOptions)[]} [conditionalFormats] - The conditional rules array.
             * @property {GC.Data.NumberValidatorOptions | GC.Data.DateValidatorOptions | GC.Data.TimeValidatorOptions | GC.Data.TextLengthValidatorOptions | GC.Data.FormulaValidatorOptions | GC.Data.FormulaListValidatorOptions | GC.Data.ListValidatorOptions} [validators] - The default data validator.
             * @property {boolean} [isPrimaryKey] - Mark the column as primary key column.
             * @property {boolean} [readonly] - Mark the column is readonly.
             * @property {boolean} [required] - Mark the column is required when insert a new row.
             * @property {Object} [defaultValue] - Provide the default value when insert a new row, could be a const or a formula.
             * @property {GC.Data.HeaderStyleOptions} [style] - The column header style options.
             * @returns {GC.Data.View} Returns the view.
             * @example
             * ```javascript
             * // Add a view by string array columns
             * productTable.addView("productView", [
             *     "id", "name", "reorderLevel", "unitPrice", "unitsInStock", "unitsOnOrder"
             * ]);
             *
             * // Add a view by customized columns
             * productTable.addView("productView", [{
             *     value: "id",
             *     caption: "ID",
             *     isPrimaryKey: true
             * }, {
             *     value: "name",
             *     caption: "NAME",
             *     required: true
             * }, {
             *     value: "quantityPerUnit",
             *     caption: "QUANTITY PER UNIT"
             * }, {
             *     value: "unitPrice",
             *     caption: "UNIT PRICE"
             * }, {
             *     value: "unitsInStock",
             *     caption: "UNITS IN STOCK",
             *     readonly: true
             * }, {
             *     value: "unitsOnOrder",
             *     caption: "UNITS ON ORDER"
             * }, {
             *     value: "reorderLevel",
             *     caption: "REORDER LEVEL"
             * }, {
             *     value: "discontinued",
             *     caption: "DISCONTINUED",
             *     defaultValue: false
             * });
             *
             * // Add a view with relationship columns
             * var supplierRelationship = dataManager.addRelationship(productTable, "supplierId", "supplier", supplierTable, "id", "products");
             * productTable.addView("productWithSupplierView", [{
             *     value: "id",
             *     caption: "ID"
             * }, {
             *     value: "name",
             *     caption: "NAME"
             * }, {
             *     value: "supplier.companyName", // relationship
             *     caption: "SUPPLIER NAME"
             * }, {
             *     value: "supplier.contactName", // relationship
             *     caption: "SUPPLIER CONTACT NAME"
             * }, {
             *     value: "supplier.contactTitle", // relationship
             *     caption: "SUPPLIER CONTACT TITLE"
             * });
             *
             * // Add a view with calc field columns
             * var supplierRelationship = dataManager.addRelationship(productTable, "supplierId", "supplier", supplierTable, "id", "products");
             * productTable.addView("productWithSupplierView", [{
             *     value: "id",
             *     caption: "ID"
             * }, {
             *     value: "name",
             *     caption: "NAME"
             * }, {
             *     caption: "TOTAL PRICE",
             *     value: "=(unitsInStock + unitsOnOrder) * unitPrice"
             * }, {
             *     caption: "SUPPLIER'S INFO",
             *     value: "=CONCAT(supplierTable.companyName, ', ', supplierTable.contactName)"
             * });
             *
             * ```
             */
            addView(name: string,  columnInfos?: string[] | GC.Data.IColumn[],  includeDefaultColumns?: boolean,  options?: GC.Data.ViewOptions): GC.Data.View;
            /**
             * Clear all the indexed fields.
             * @example
             * ```javascript
             * // Clear all the indexed fields.
             * table.clearIndexes();
             * ```
             */
            clearIndexes(): void;
            /**
             * Create index for the fields.
             * @param {Array} fields The index fields.
             * @example
             * ```javascript
             * // Create indexes.
             * table.createIndexes(["name", "country", "project"]);
             * ```
             */
            createIndexes(fields: string[]): void;
            /**
             * Drop indexed fields.
             * @param {Array} fields The indexed fields.
             * @example
             * ```javascript
             * // Drop indexed fields.
             * table.dropIndexes(["name", "country", "project"]);
             * ```
             */
            dropIndexes(fields: string[]): void;
            /**
             * Requests the table data from local data source or remote data source by the data source option.
             * @param {boolean} [reload] Whether to reload the data from server side forcibly.
             * @returns {Promise} The resolving Promise thenable. You could get the data in Promise.then().
             * @example
             * ```javascript
             * // Use fetched data to build a tablesheet
             * productTable.fetch().then(function(data) {
             *     var productView = productTable.addView("productView");
             *     var tableSheet = spread.addSheetTab(0, "productTableSheet", GC.Spread.Sheets.SheetType.tableSheet);
             *     tableSheet.setDataView(productView);
             * });
             * ```
             */
            fetch(reload?: boolean): Promise<any>;
            /**
             * Get all the indexed fields.
             * @returns {Array} The indexed fields.
             * @example
             * ```javascript
             * // Indexed fields exist.
             * table.getIndexes(); // returns ["name", "country", "project"]
             * // No indexed field.
             * table.getIndexes(); // returns []
             * ```
             */
            getIndexes(): string[];
            /**
             * Removes a view.
             * @param {string} name - The name of the view to be removed.
             * @example
             * ```javascript
             * // Remove a view by name
             * dataManager.removeView("productView");
             * ```
             */
            removeView(name: string): void;
            /**
             * Search the records with search value, returns all the exact matching records.
             * @param {Object} value The search value. Always convert the non-string to string type.
             * @param {string} field The target field.
             * @returns {Array} The all matching records.
             * @example
             * ```javascript
             * // Search value and succeed
             * table.search("SpreadJS", "group"); // returns [ {id: 1, project: "DataManager", group: "SpreadJS"}, {id: 3, project: "TableSheet", group: "SpreadJS"} ]
             * // Search value and failed
             * table.search("WPS", "project"); // returns []
             * ```
             */
            search(value: any,  field: string): any[];
        }

        export class View{
            /**
             * Represents the view.
             * @class
             * @param {string} name - The view name.
             * @param {string[] | Object[]} [columnInfos] - The column information, each column includes the following properties.
             * @param {boolean} [includeDefaultColumns] - Whether to include current table's default columns when column information are empty. Its default value is true.
             * @param {GC.Data.ViewOptions} [options] - The view options.
             * @property {string} name - The unique name of the column.
             * @property {string} [value] - The value of the column, could be a field name of table from database, or formula which uses the fields names.
             * @property {string | string[]} [caption] - The caption of the column.
             * @property {number | string} [width] - The width of the column, support number in pixel, or star size.
             * @property {GC.Data.StyleOptions} [style] - The column style options.
             * @property {(GC.Data.CellValueRuleOptions | GC.Data.SpecificTextRuleOptions | GC.Data.FormulaRuleOptions | GC.Data.DateOccurringRuleOptions | GC.Data.Top10RuleOptions | GC.Data.UniqueRuleOptions | GC.Data.DuplicateRuleOptions | GC.Data.AverageRuleOptions | GC.Data.TwoScaleRuleOptions | GC.Data.ThreeScaleRuleOptions | GC.Data.DataBarRuleOptions | GC.Data.IconSetRuleOptions)[]} [conditionalFormats] - The conditional rules array.
             * @property {GC.Data.NumberValidatorOptions | GC.Data.DateValidatorOptions | GC.Data.TimeValidatorOptions | GC.Data.TextLengthValidatorOptions | GC.Data.FormulaValidatorOptions | GC.Data.FormulaListValidatorOptions | GC.Data.ListValidatorOptions} [validators] - The default data validator.
             * @property {boolean} [isPrimaryKey] - Mark the column as primary key column.
             * @property {boolean} [readonly] - Mark the column is readonly.
             * @property {boolean} [required] - Mark the column is required when insert a new row.
             * @property {Object} [defaultValue] - Provide the default value when insert a new row, could be a const or a formula.
             * @property {GC.Data.HeaderStyleOptions} [style] - The column header style options.
             * @returns {GC.Data.View} Returns the view.
             * /
             ///* field name: string
             /**
             * Represents the name of the view.
             * @type {string}
             */
            constructor(name: string,  columnInfos?: string[] | GC.Data.IColumn[],  includeDefaultColumns?: boolean,  options?: GC.Data.ViewOptions);
            /**
             * Whether to filter again after data is changed. Its default value is true.
             * @type {string}
             */
            autoFilter: boolean;
            /**
             * Whether to sort again after data is changed. Its default value is true.
             * @type {string}
             */
            autoSort: boolean;
            /**
             * Adds a column into current view.
             * @param {string | Object} column - The column string or object. When the parameter is a string, the column name and value are the string.
             * @example
             * ```javascript
             * // Add columns to view
             * var productTable = dataManager.addTable("products", {
             *      remote: {
             *         read: {
             *             url: "https://demodata.mescius.io/northwind/api/v1/products"
             *         }
             *     }
             * });
             * var productView = productTable.addView("productView", [ "id", "name" ]);
             * productTable.fetch().then(function() {
             *     productView.addColumn("reorderLevel");
             *     productView.addColumn({ value: "unitPrice", caption: "UNIT PRICE" });
             * });
             * ```
             */
            addColumn(column: string | GC.Data.IColumn): void;
            /**
             * Add a style rule into view.
             * @param {string} name - The style rule name.
             * @param {GC.Data.StyleOptions} [style] - The style if the rule is satisfied.
             * @param {GC.Data.FormulaRule | GC.Data.StateRule} [rule] - The style rule option.
             * @example
             * ```javascript
             * // add a style rule
             * view.addStyleRule("dirtyRowStyle", { backColor: "yellow" }, {
             *    direction: GC.Data.StateRuleDirection.row,
             *    state: GC.Data.RowColumnStates.dirty
             * });
             * ```
             */
            addStyleRule(name: string,  style?: GC.Data.StyleOptions,  rule?: GC.Data.FormulaRule | GC.Data.StateRule): void;
            /**
             * clear all the style rules from view.
             * @example
             * ```javascript
             * // remove a style rule
             * view.clearStyleRules();
             * ```
             */
            clearStyleRules(): void;
            /**
             * Requests the view data from its host table and related table.
             * @param {boolean} [reload] Whether to reload the data from server side forcibly.
             * @returns {Promise} The resolving Promise thenable. You could get the data in Promise.then().
             * @example
             * ```javascript
             * // Set tablesheet with custom view
             * var tablesheet = spread.addSheetTab(0, "TableSheet1", GC.Spread.Sheets.SheetType.tableSheet);
             * var dataManager = new GC.Data.DataManager();
             * var productTable = dataManager.addTable("productTable", {
             *     remote: {
             *         read: {
             *             url: "https://demodata.mescius.io/northwind/api/v1/products"
             *         }
             *     }
             * });
             * var productView = productTable.addView("productView", [
             *     "id", "name", "reorderLevel", "unitPrice", "unitsInStock", "unitsOnOrder"
             * ]);
             * productView.fetch().then(function () {
             *     // set data source with a View
             *     tablesheet.setDataView(productView);
             * });
             * ```
             */
            fetch(reload?: boolean): Promise<any>;
            /**
             * Gets a column of current view.
             * @param {number} [index] - The column index.
             * @returns {Object | Object[]} Returns a column by the specified index, or returns all columns when the parameter is omitted.
             * @example
             * ```javascript
             * // Get all columns
             * var allColumns = productWithSupplierView.getColumn();
             * // Get the second column
             * var column1 = productWithSupplierView.getColumn(1);
             * ```
             */
            getColumn(index?: number): GC.Data.IColumn | GC.Data.IColumn[];
            /**
             * Get a style rule or all style rules.
             * @param {string} [name] - The style rule name.
             * @returns {GC.Data.StyleRule | GC.Data.StyleRules | undefined} Returns a style rule or all style rules.
             * @example
             * ```javascript
             * // get a style rule
             * view.getStyleRule("dirtyRowStyle");
             * // get all style rule
             * view.getStyleRule();
             * ```
             */
            getStyleRule(name?: string): GC.Data.StyleRule | GC.Data.StyleRules | undefined;
            /**
             * Get the view host table data source length.
             * @returns {number} The host table data source length.
             * @example
             * ```javascript
             * // after fetch data, can get the view data source length.
             * let dataSourceLength = productView.length();
             * ```
             */
            length(): number;
            /**
             * Removes a column from current view.
             * @param column - The column value.
             * @example
             * ```javascript
             * // Remove a column
             * productWithSupplierView.removeColumn("discontinued");
             * ```
             */
            removeColumn(column: string): void;
            /**
             * Add a style rule by name from view.
             * @param {string} name - The style rule name.
             * @example
             * ```javascript
             * // remove a style rule
             * view.removeStyleRule("dirtyRowStyle");
             * ```
             */
            removeStyleRule(name: string): void;
            /**
             * Get the visible data length in the current view.
             * @returns {number} The visible data length in the current view.
             * @example
             * ```javascript
             * // after fetch data, can get the view visibleLength.
             * let viewVisibleLength = productView.visibleLength();
             * ```
             */
            visibleLength(): number;
        }
    }

    module Pivot{

        export interface ICustomSortCallBack{
            (fieldItemNameArray: any[], sortInfo: GC.Pivot.IPivotCustomSortInfo) : any[];
        }


        export interface IPivotCustomSortInfo{
            sortType: GC.Pivot.SortType;
            isDate?: boolean;
        }


        /**
         * @typedef {object} GC.Pivot.IDataPosition
         * @property {boolean} [display]
         * @property {GC.Pivot.DataPosition}
         * @property {number}
         */
        export type IDataPosition = 
            {
                display?: boolean;
                positionType: GC.Pivot.DataPosition;
                positionIndex: number;
            }
        

        /**
         * @typedef {object} GC.Pivot.IPivotCaptionConditionFilterInfo
         * @property {GC.Pivot.PivotConditionType} conType
         * @property {string[]} val
         * @property {GC.Pivot.PivotCaptionFilterOperator} operator
         */
        export type IPivotCaptionConditionFilterInfo = 
            {
                conType: GC.Pivot.PivotConditionType;
                val: string[];
                operator: GC.Pivot.PivotCaptionFilterOperator;
            }
        

        /**
         * @typedef {object} GC.Pivot.IPivotDateConditionFilterInfo
         * @property {GC.Pivot.PivotConditionType} conType
         * @property {Date[]} val
         * @property {GC.Pivot.PivotDateFilterOperator} operator
         * @property {boolean} isWholeDay
         * @property {boolean} isDynamicEndDate
         * @property {boolean} isParallel
         * @property {GC.Pivot.PivotAdvancedDateFilterBy} by
         */
        export type IPivotDateConditionFilterInfo = 
            {
                conType: GC.Pivot.PivotConditionType;
                val: Date[];
                operator: GC.Pivot.PivotDateFilterOperator;
                isWholeDay: boolean;
                isDynamicEndDate?: boolean;
                isParallel?: boolean;
                by?: GC.Pivot.PivotAdvancedDateFilterBy;
            }
        

        /**
         * @typedef {object} GC.Pivot.IPivotTop10ConditionFilterInfo
         * @property {GC.Pivot.PivotConditionType} conType
         * @property {number} val
         * @property {GC.Pivot.PivotTop10FilterType} type
         * @property {GC.Pivot.PivotTop10FilterOperator} operator
         */
        export type IPivotTop10ConditionFilterInfo = 
            {
                conType: GC.Pivot.PivotConditionType;
                val: number;
                type: GC.Pivot.PivotTop10FilterType;
                operator: GC.Pivot.PivotTop10FilterOperator;
            }
        

        /**
         * @typedef {object} GC.Pivot.IPivotValueConditionInfo
         * @property {GC.Pivot.PivotConditionType} conType
         * @property {number[]} val
         * @property {GC.Pivot.PivotValueFilterOperator} operator
         */
        export type IPivotValueConditionInfo = 
            {
                conType: GC.Pivot.PivotConditionType;
                val: number[];
                operator: GC.Pivot.PivotValueFilterOperator;
            }
        

        /**
         * @typedef {object} GC.Pivot.IPivotViewSortInfo
         * @property {GC.Pivot.SortType} sortType sort field by ascending or descending or custom.
         * @property {string} sortValueFieldName sort field by specified value area field's value.
         * @property {GC.Spread.Pivot.IPivotReference[]} sortByPivotReferences sort field by specified row or column. sortValueFieldName is must if use this property.
         * @property {GC.Pivot.ICustomSortCallBack} customSortCallback sort field by custom way. should return the ordered field items array.
         * @property {(string | number)[]} customSortList custom sort list for custom sort.
         */
        export type IPivotViewSortInfo = 
            {
                sortType?: GC.Pivot.SortType;
                sortValueFieldName?: string;
                sortByPivotReferences?: GC.Spread.Pivot.IPivotReference[];
                customSortCallback?: GC.Pivot.ICustomSortCallBack;
                customSortList?: (string | number)[];
            }
        

        /**
         * @typedef GC.Pivot.IStringGroupItems
         * @property {Object.<string, string[] | number[]>}  -The group field items name and the matched items text.
         */
        export type IStringGroupItems = 
            {
                [groupItemName: string]: (string | number)[];
            }
        

        /**
         * @typedef {object} GC.Pivot.IValueInfo
         * @property {string} sourceName value field source name.
         * @property {GC.Pivot.SubtotalType} subtotalType value field subtotal type.
         */
        export type IValueInfo = 
            {
                sourceName: string;
                subtotalType: GC.Pivot.SubtotalType;
            }
        
        /**
         * indicates the area where the Values is located
         * @enum {number}
         */
        export enum DataPosition{
            /**
             * Values is displayed in the row area
             */
            row= 0,
            /**
             * Values is displayed in the column area
             */
            col= 1,
            /**
             * Values does not show
             */
            none= 2
        }

        /**
         * indicates the date group type
         * @enum {number}
         */
        export enum DateGroupType{
            /**
             * seconds
             */
            seconds= 0,
            /**
             * minutes
             */
            minutes= 1,
            /**
             * hours
             */
            hours= 2,
            /**
             * days
             */
            days= 3,
            /**
             * months
             */
            months= 4,
            /**
             * quarters
             */
            quarters= 5,
            /**
             * years
             */
            years= 6
        }

        /**
         * Describes the type of advanced date filter
         * @enum {number}
         */
        export enum PivotAdvancedDateFilterBy{
            /**
             * filter by month
             */
            month= 1,
            /**
             * filter by quarter
             */
            quarter= 2,
            /**
             * filter by year
             */
            year= 3
        }

        /**
         * Describe the type of caption filter
         * @enum {number}
         */
        export enum PivotCaptionFilterOperator{
            /**
             * Indicates whether the string is equal to a specified string.
             */
            equalsTo= 0,
            /**
             * Indicates whether the string is not equal to a specified string.
             */
            notEqualsTo= 1,
            /**
             * Indicates whether the string starts with a specified string.
             */
            beginsWith= 2,
            /**
             * Indicates whether the string does not start with a specified string.
             */
            doesNotBeginWith= 3,
            /**
             * Indicates whether the string ends with a specified string.
             */
            endsWith= 4,
            /**
             * Indicates whether the string does not end with a specified string.
             */
            doesNotEndWith= 5,
            /**
             * Indicates whether the string contains a specified string.
             */
            contains= 6,
            /**
             * Indicates whether the string does not contain a specified string.
             */
            doesNotContain= 7,
            /**
             * Indicates whether a value is greater than the parameter value.
             */
            greaterThan= 8,
            /**
             * Indicates whether a value is greater than or equal to the parameter value.
             */
            greaterThanOrEqualsTo= 9,
            /**
             * Indicates whether a value is less than the parameter value.
             */
            lessThan= 10,
            /**
             * Indicates whether a value is less than or equal to the parameter value.
             */
            lessThanOrEqualsTo= 11,
            /**
             * Indicates whether the number is between parameter values.
             */
            between= 12,
            /**
             * Indicates whether the number is not between parameter values.
             */
            notBetween= 13
        }

        /**
         * describe the type of filter
         *  @enum {number}
         */
        export enum PivotConditionType{
            /**
             * Indicates a text or number condition type.
             */
            caption= 0,
            /**
             * Indicates a date type condition.
             */
            date= 1,
            /**
             * Indicates number condition.
             */
            value= 2,
            /**
             * Indicates a specific condition which use for finding the top10 number .
             */
            top10= 3
        }

        /**
         * Indicates the field data type of pivot table.
         * @enum {number}
         */
        export enum PivotDataType{
            /**
             * number type
             */
            number= 0,
            /**
             * string type
             */
            string= 1,
            /**
             * date type
             */
            date= 2
        }

        /**
         * Describe the type of date filter
         * @enum {number}
         */
        export enum PivotDateFilterOperator{
            /**
             * Indicates whether the date time is equal to a certain time.
             */
            equalsTo= 0,
            /**
             * Indicates whether the date time is not equal to a certain time.
             */
            notEqualsTo= 1,
            /**
             * Indicates whether the date time is before a certain time.
             */
            before= 2,
            /**
             * Indicates whether the date time is before or equal to a certain time.
             */
            beforeEqualsTo= 3,
            /**
             * Indicates whether the date time is after a certain time.
             */
            after= 4,
            /**
             * Indicates whether the date time is after or equal to a certain time.
             */
            afterEqualsTo= 5,
            /**
             * Indicates whether the date time is between the certain time.
             */
            Between= 6,
            /**
             * Indicates whether the date time is not between the certain time.
             */
            notBetween= 7,
            /**
             * Specifies today.
             */
            today= 8,
            /**
             * Specifies yesterday.
             */
            yesterday= 9,
            /**
             * Specifies tomorrow.
             */
            tomorrow= 10,
            /**
             * Specifies this month.
             */
            thisMonth= 12,
            /**
             * Specifies last month.
             */
            lastMonth= 13,
            /**
             * Specifies next month.
             */
            nextMonth= 14,
            /**
             * Specifies this week.
             */
            thisWeek= 15,
            /**
             * Specifies last week.
             */
            lastWeek= 16,
            /**
             * Specifies next week.
             */
            nextWeek= 17,
            /**
             * Specifies next Quarter.
             */
            nextQuarter= 18,
            /**
             * Specifies this Quarter.
             */
            thisQuarter= 19,
            /**
             * Specifies last Quarter.
             */
            lastQuarter= 20,
            /**
             * Specifies next Year.
             */
            nextYear= 21,
            /**
             * Specifies this Year.
             */
            thisYear= 22,
            /**
             * Specifies last Year.
             */
            lastYear= 23,
            /**
             * Indicates the first quarter of a year.
             */
            Q1= 24,
            /**
             * Indicates the second quarter of a year.
             */
            Q2= 25,
            /**
             * Indicates the third quarter of a year.
             */
            Q3= 26,
            /**
             * Indicates the fourth quarter of a year.
             */
            Q4= 27,
            /**
             * Indicates the January of a year.
             */
            M1= 28,
            /**
             * Indicates the February of a year.
             */
            M2= 29,
            /**
             * Indicates the March of a year.
             */
            M3= 30,
            /**
             * Indicates the April of a year.
             */
            M4= 31,
            /**
             * Indicates the May of a year.
             */
            M5= 32,
            /**
             * Indicates the June of a year.
             */
            M6= 33,
            /**
             * Indicates the July of a year.
             */
            M7= 34,
            /**
             * Indicates the August of a year.
             */
            M8= 35,
            /**
             * Indicates the September of a year.
             */
            M9= 36,
            /**
             * Indicates the October of a year.
             */
            M10= 37,
            /**
             * Indicates the November of a year.
             */
            M11= 38,
            /**
             * Indicates the December of a year.
             */
            M12= 39,
            /**
             * Indicates the form Jan-1 to today in current year.
             */
            yearToDate= 40,
            /**
             * Indicates the form customize start time to end time.
             */
            dateToDate= 41
        }

        /**
         * indicates the pivot table show value as  type
         * @enum {number}
         */
        export enum PivotShowDataAs{
            /**
             * no calculation
             */
            normal= 0,
            /**
             *  % of Grand Total
             */
            percentOfTotal= 1,
            /**
             * % of row Grand Total
             */
            percentOfRow= 2,
            /**
             * % of Column Grand Total
             */
            percentOfCol= 3,
            /**
             * % of
             */
            percent= 4,
            /**
             * % of Parent Row Total
             */
            percentOfParentRow= 5,
            /**
             * % of Parent Column Total
             */
            percentOfParentCol= 6,
            /**
             * % of Parent Total
             */
            percentOfParent= 7,
            /**
             * Difference from
             */
            difference= 8,
            /**
             * % difference from
             */
            percentDiff= 9,
            /**
             * running total in
             */
            runTotal= 10,
            /**
             * % of running total in
             */
            percentOfRunningTotal= 11,
            /**
             * rank smallest to largest
             */
            rankAscending= 12,
            /**
             * rank largest to smallest
             */
            rankDescending= 13,
            /**
             * index
             */
            index= 14
        }

        /**
         * Indicates the view layout type of pivot table.
         * @enum {number}
         */
        export enum PivotShowDataAsBaseItemType{
            /**
             * show data as base on a certain item of base field
             */
            item= 0,
            /**
             * show data as base on the next item of base field
             */
            next= 1,
            /**
             * show data as base on the previous item of base field
             */
            previous= 2
        }

        /**
         * Indicates the pivot field type of pivot table.
         * @enum {number}
         */
        export enum PivotSourceFieldType{
            /**
             * this is a calc field mark
             */
            isCalcField= 0,
            /**
             * this is a group field mark
             */
            isGroupField= 1
        }

        /**
         * Indicates that the interception is the front or back
         * @enum {number}
         */
        export enum PivotTop10FilterOperator{
            /**
             * Indicates that the interception is the front
             */
            top= 1,
            /**
             * Indicates that the interception is the back
             */
            bottom= -1
        }

        /**
         * Describes the type of filter
         * @enum {number}
         */
        export enum PivotTop10FilterType{
            /**
             * filter by count
             */
            count= 0,
            /**
             * filter by percent
             */
            percent= 1,
            /**
             * filter by sum
             */
            sum= 2
        }

        /**
         * describe the type of filter
         * @enum {number}
         */
        export enum PivotValueFilterOperator{
            /**
             * Determines whether a  value is equal to the parameter value.
             */
            equalsTo= 0,
            /**
             * Determines whether a  value is not equal to the parameter value.
             */
            notEqualsTo= 1,
            /**
             * Determines whether a  value is greater than the parameter value.
             */
            greaterThan= 2,
            /**
             * Determines whether a  value is greater than or equal to the parameter value.
             */
            greaterThanOrEqualsTo= 3,
            /**
             * Determines whether a  value is less than the parameter value.
             */
            lessThan= 4,
            /**
             * Determines whether a  value is less than or equal to the parameter value.
             */
            lessThanOrEqualsTo= 5,
            /**
             * Determines whether a  value is between the two parameter values.
             */
            between= 6,
            /**
             *Determines whether a cell value is not between the two parameter values.
             */
            notBetween= 7
        }

        /**
         * Indicates the pivot table sort type.
         * @enum {number}
         */
        export enum SortType{
            /**
             * Sort in ascending order
             */
            asc= 0,
            /**
             * Sort in descending order
             */
            desc= 1,
            /**
             * User-defined sort type
             */
            custom= 2
        }

        /**
         * Indicates the type of data subtotal
         * @enum {number}
         */
        export enum SubtotalType{
            /**
             * Average
             */
            average= 0,
            /**
             * Count
             */
            count= 1,
            /**
             * CountNums
             */
            countNums= 2,
            /**
             * Max
             */
            max= 3,
            /**
             * Min
             */
            min= 4,
            /**
             * Product
             */
            product= 5,
            /**
             * StdDev
             */
            stdDev= 6,
            /**
             * StdDevp
             */
            stdDevp= 7,
            /**
             * Sum
             */
            sum= 8,
            /**
             * Var
             */
            varr= 9,
            /**
             * Varp
             */
            varp= 10
        }

    }

    module Spread{
        module CalcEngine{
            /**
             * Gets or Sets the Excel Compatible Mode for CalcEngine. Default value is FALSE.
             * When ExcelCompatibleCalcMode disabled, SpreadJS will auto convert text to number for calculation.
             * When ExcelCompatibleCalcMode enabled, treated the text differently when provided as argument directly or in array / reference.
             * For example, A1 is text value "1", SUM(A1, 1) will be 2 if disabled, and 1 if enabled.
             * @example
             * ```javascript
             * GC.Spread.CalcEngine.ExcelCompatibleCalcMode = true;
             * ```
             */
            var ExcelCompatibleCalcMode: boolean;
            /**
             * Represents the type node in the formula expression tree.
             * @enum {number}
             */
            export enum ExpressionType{
                /**
                 * Specifies the unknown type
                 */
                unknow= 0,
                /**
                 * Specifies the reference type
                 */
                reference= 1,
                /**
                 * Specifies the number type
                 */
                number= 2,
                /**
                 * Specifies the string type
                 */
                string= 3,
                /**
                 * Specifies the boolean type
                 */
                boolean= 4,
                /**
                 * Specifies the error type
                 */
                error= 5,
                /**
                 * Specifies the array type
                 */
                array= 6,
                /**
                 * Specifies the function type
                 */
                function= 7,
                /**
                 * Specifies the name type
                 */
                name= 8,
                /**
                 * Specifies the operator type
                 */
                operator= 9,
                /**
                 * Specifies the parenthesis type
                 */
                parentheses= 10,
                /**
                 * Specifies the missing argument type
                 */
                missingArgument= 11,
                /**
                 * Specifies the expand type
                 */
                expand= 12,
                /**
                 * Specifies the struct reference type
                 */
                structReference= 13,
                /**
                 * Specifies the local name type
                 */
                LocalName= 100
            }


            export class AsyncEvaluateContext{
                /**
                 * Represents an evaluate context for async functions.
                 * @class
                 * @param {Object} context The common evaluate context.
                 */
                constructor(context: any);
                /**
                 * Set the async function evaluate result to CalcEngine, CalcEngine uses this value to recalculate the cell that contains this async function and all dependents cells.
                 * @param {Object} value The async function evaluate result.
                 */
                setAsyncResult(value: any): void;
            }

            export class CalcArray{
                /**
                 * Represents an array in the formula result.
                 * @class
                 * @param {any[][]} array The array content of the formula result.
                 */
                constructor(array: any[][]);
            }

            export class CalcError{
                /**
                 * Represents an error in calculation.
                 * @class
                 * @param {string} error The description of the error.
                 */
                constructor(error: string);
                /**
                 * Parses the specified error from the string.
                 * @param {string} value The error string.
                 * @returns {GC.Spread.CalcEngine.CalcError} The calculation error.
                 */
                static parse(value:string): GC.Spread.CalcEngine.CalcError;
                /**
                 * Returns a string that represents this instance.
                 * @returns {string} The error string.
                 */
                toString(): string;
            }

            export class Expression{
                /**
                 * Provides the base class from which the classes that represent expression tree nodes are derived. This is an abstract class.
                 * @class
                 * @example
                 * ```javascript
                 * // the below code will return a Expression
                 * GC.Spread.Sheets.CalcEngine.formulaToExpression(sheet, "=1");
                 * // Convert a formula string to an expression tree
                 * const expression = GC.Spread.Sheets.CalcEngine.formulaToExpression(sheet, "=SUM(A1:B2)");
                 * console.log(expression.type); // Returns ExpressionType.function
                 * console.log(expression.functionName); // Returns "SUM"
                 * ```
                 */
                constructor(type: GC.Spread.CalcEngine.ExpressionType);
                /**
                 * Indicates the type of expression node in the formula parse tree. Each expression node represents a specific element in a formula, such as a number, string, reference, function, or operator.
                 * @type {GC.Spread.CalcEngine.ExpressionType}
                 * @example
                 * ```javascript
                 * console.log(GC.Spread.Sheets.CalcEngine.formulaToExpression(sheet, "=1").type === GC.Spread.CalcEngine.ExpressionType.number);
                 * ```
                 */
                type: GC.Spread.CalcEngine.ExpressionType;
            }
            module Functions{
                /**
                 * Defines a global custom function, which can be used in formulas.
                 * The global custom function can be called from any spreadsheet.
                 * @param {string} name The name of the function.
                 * @param {GC.Spread.CalcEngine.Functions.Function} fn The function to add.
                 * @returns {GC.Spread.CalcEngine.Functions.Function} The function that was added.
                 * @example
                 * ```javascript
                 * class FactorialFunction extends GC.Spread.CalcEngine.Functions.Function {
                 *     constructor () {
                 *         super('FACTORIAL', 1, 1, {
                 *             description: "Function to calculate the Fibonacci number.",
                 *             parameters: [{ name: 'n' }]
                 *         });
                 *     }
                 *     evaluate (n) {
                 *         var fib = [0, 1];
                 *         for (var i = 2; i <= n; i++) {
                 *           fib[i] = fib[i - 1] + fib[i - 2];
                 *         }
                 *         return fib[n];
                 *     }
                 * }
                 * GC.Spread.CalcEngine.Functions.defineGlobalCustomFunction("FACTORIAL", new FactorialFunction());
                 * ```
                 */
                function defineGlobalCustomFunction(name: string,  fn: GC.Spread.CalcEngine.Functions.Function): GC.Spread.CalcEngine.Functions.Function;
                /**
                 * Gets all of the global functions or one global function that specified by name.
                 * @param {string} name The name of the function.
                 * @returns {GC.Spread.CalcEngine.Functions.Function} If the name is empty, return all of the global functions, otherwise, return one function with the specified name.
                 */
                function findGlobalFunction(name?: string): any;
                /**
                 * If the name is empty, remove all of the global functions, otherwise, remove one function with the specified name.
                 * @param {string} name The name of the function.
                 */
                function removeGlobalFunction(name?: string): void;

                export interface IFunctionDescription{
                    /**
                     * The description of the function.
                     */
                    description: string;
                    /**
                     * The parameters' description array of the function.
                     */
                    parameters: GC.Spread.CalcEngine.Functions.IParameterDescription[];
                }


                export interface IParameterDescription{
                    /**
                     * The parameter name.
                     */
                    name: string;
                    /**
                     * Whether the parameter is repeatable.
                     */
                    repeatable?: boolean;
                    /**
                     * Whether the parameter is optional.
                     */
                    optional?: boolean;
                }

                /**
                 * Represents the asynchronous function evaluate mode.
                 * @enum {number}
                 */
                export enum AsyncFunctionEvaluateMode{
                    /**
                     * enum value is 0, Specifies that the async function evaluates the changed, referenced cells.
                     */
                    onRecalculation= 0,
                    /**
                     * enum value is 1, Specifies that the async function only evaluates once.
                     */
                    calculateOnce= 1,
                    /**
                     * enum value is 2, Specifies that the async function evaluates based on the interval.
                     */
                    onInterval= 2
                }


                export class AsyncFunction extends GC.Spread.CalcEngine.Functions.Function{
                    /**
                     * Represents an abstract base class for defining asynchronous functions.
                     * Asynchronous functions are typically used for operations that require server requests or other time-consuming tasks.
                     * @class
                     * @param {string} name The name of the function.
                     * @param {number} [minArgs] The minimum number of arguments for the function.
                     * @param {number} [maxArgs] The maximum number of arguments for the function.
                     * @param {GC.Spread.CalcEngine.Functions.IFunctionDescription} [description] The description of the function.
                     * @example
                     * ```javascript
                     * class WeatherFunction extends GC.Spread.CalcEngine.Functions.AsyncFunction {
                     *     constructor () {
                     *         super('WEATHER', 0, 0, {
                     *             description: "Get Weather",
                     *             parameters: []
                     *         });
                     *     }
                     *     evaluate (context) {
                     *         setTimeout(function () { context.setAsyncResult('sunny'); }, 100); // Here use setTimeout to simulate the server call.
                     *     }
                     * }
                     * spread.addCustomFunction(new WeatherFunction());
                     * spread.getActiveSheet().setFormula(0, 0, '=WEATHER()');
                     * ```
                     */
                    constructor(name: string,  minArgs?: number,  maxArgs?: number,  description?: GC.Spread.CalcEngine.Functions.IFunctionDescription);
                    /**
                     * Returns the default value of the evaluated function result before getting the async result.
                     * @returns {Object} The default value of the evaluated function result before getting the async result.
                     */
                    defaultValue(): any;
                    /**
                     * Returns the result of the function applied to the arguments.
                     * @param {GC.Spread.CalcEngine.AsyncEvaluateContext} context The evaluate context
                     * @param {Object} args Arguments for the function evaluation
                     * @returns {Object} The result of the function applied to the arguments.
                     */
                    evaluateAsync(context: GC.Spread.CalcEngine.AsyncEvaluateContext,  args: any): any;
                    /**
                     * Decides how to re-calculate the formula.
                     * @returns {GC.Spread.CalcEngine.Functions.AsyncFunctionEvaluateMode} The evaluate mode.
                     */
                    evaluateMode(): GC.Spread.CalcEngine.Functions.AsyncFunctionEvaluateMode;
                    /**
                     * Returns the interval.
                     * @returns {number} The interval in milliseconds.
                     */
                    interval(): number;
                }

                export class Function{
                    /**
                     * Represents an abstract base class for defining functions.
                     * @class
                     * @param {string} name The name of the function.
                     * @param {number} [minArgs] The minimum number of arguments for the function.
                     * @param {number} [maxArgs] The maximum number of arguments for the function.
                     * @param {GC.Spread.CalcEngine.Functions.IFunctionDescription} [functionDescription] The description object of the function.
                     * @example
                     * ```javascript
                     * class FactorialFunction extends GC.Spread.CalcEngine.Functions.Function {
                     *     constructor () {
                     *         super('FACTORIAL', 1, 1, {
                     *             description: "Function to calculate the Fibonacci number.",
                     *             parameters: [{ name: 'n' }]
                     *         });
                     *     }
                     *     evaluate (n) {
                     *         var fib = [0, 1];
                     *         for (var i = 2; i <= n; i++) {
                     *           fib[i] = fib[i - 1] + fib[i - 2];
                     *         }
                     *         return fib[n];
                     *     }
                     * }
                     * spread.addCustomFunction(new FactorialFunction());
                     * spread.getActiveSheet().setFormula(0, 0, '=FACTORIAL(10)');
                     * ```
                     */
                    constructor(name: string,  minArgs?: number,  maxArgs?: number,  functionDescription?: GC.Spread.CalcEngine.Functions.IFunctionDescription);
                    /**
                     * Represents the maximum number of arguments for the function.
                     * @type {number}
                     */
                    maxArgs: number;
                    /**
                     * Represents the minimum number of arguments for the function.
                     * @type {number}
                     */
                    minArgs: number;
                    /**
                     * Represents the name of the function.
                     * @type {string}
                     */
                    name: string;
                    /**
                     * Represents the type name string used for supporting serialization.
                     * @type {string}
                     */
                    typeName: string;
                    /**
                     * Determines whether the function accepts array values for the specified argument.
                     * @param {number} argIndex Index of the argument.
                     * @function
                     * @returns {boolean} `true` if the function accepts array values for the specified argument; otherwise, `false`.
                     */
                    acceptsArray(argIndex: number): boolean;
                    /**
                     * Indicates whether the function can process Error values.
                     * @param {number} argIndex Index of the argument.
                     * @returns {boolean} `true` if the function can process Error values for the specified argument; otherwise, `false`.
                     * @function
                     */
                    acceptsError(argIndex: number): boolean;
                    /**
                     * Indicates whether the Evaluate method can process missing arguments.
                     * @param {number} argIndex Index of the argument
                     * @returns {boolean} `true` if the Evaluate method can process missing arguments; otherwise, `false`.
                     */
                    acceptsMissingArgument(argIndex: number): boolean;
                    /**
                     * Determines whether the function accepts Reference values for the specified argument.
                     * @param {number} argIndex Index of the argument.
                     * @returns {boolean} `true` if the function accepts Reference values for the specified argument; otherwise, `false`.
                     * @function
                     */
                    acceptsReference(argIndex: number): boolean;
                    /**
                     * Returns the description of the function.
                     * @function
                     * @returns {Object} The description of the function.
                     */
                    description(): GC.Spread.CalcEngine.Functions.IFunctionDescription;
                    /**
                     * Returns the result of the function applied to the arguments.
                     * @param {Object} args Arguments for the function evaluation
                     * @returns {Object} The result of the function applied to the arguments.
                     */
                    evaluate(...args: any): any;
                    /**
                     * Finds the branch argument.
                     * @param {Object} test The test.
                     * @returns {number} Indicates the index of the argument that would be treated as the branch condition.
                     * @example
                     * ```javascript
                     * function EqualsFunction() {
                     *     this.name = 'Equals';
                     *     this.maxArgs = 3;
                     *     this.minArgs = 3;
                     * }
                     * EqualsFunction.prototype = new GC.Spread.CalcEngine.Functions.Function();
                     * EqualsFunction.prototype.evaluate = function(logicalTest, valueIfTrue, valueIfFalse) {
                     *     return logicalTest ? valueIfTrue : valueIfFalse;
                     * }
                     * EqualsFunction.prototype.isBranch = function() {
                     *     return true;
                     * }
                     * EqualsFunction.prototype.findTestArgument = function() {
                     *     return 0;
                     * }
                     * EqualsFunction.prototype.findBranchArgument = function(logicalTestResult) {
                     *     if (logicalTestResult === true) {
                     *         return 1;
                     *     }
                     *     return 2;
                     * }
                     * var equalsFunction = new EqualsFunction();
                     * var spread = GC.Spread.Sheets.findControl("ss") || GC.Spread.Sheets.findControl("sampleDiv");
                     * spread.addCustomFunction(equalsFunction);
                     * ```
                     */
                    findBranchArgument(test: any): number;
                    /**
                     * Finds the test argument when this function is branched.
                     * @returns {number} Indicates the index of the argument that would be treated as the test condition.
                     * @example
                     * ```javascript
                     * function EqualsFunction() {
                     *     this.name = 'Equals';
                     *     this.maxArgs = 3;
                     *     this.minArgs = 3;
                     * }
                     * EqualsFunction.prototype = new GC.Spread.CalcEngine.Functions.Function();
                     * EqualsFunction.prototype.evaluate = function(logicalTest, valueIfTrue, valueIfFalse) {
                     *     return logicalTest ? valueIfTrue : valueIfFalse;
                     * }
                     * EqualsFunction.prototype.isBranch = function() {
                     *     return true;
                     * }
                     * EqualsFunction.prototype.findTestArgument = function() {
                     *     return 0;
                     * }
                     * EqualsFunction.prototype.findBranchArgument = function(logicalTestResult) {
                     *     if (logicalTestResult === true) {
                     *         return 1;
                     *     }
                     *     return 2;
                     * }
                     * var equalsFunction = new EqualsFunction();
                     * var spread = GC.Spread.Sheets.findControl("ss") || GC.Spread.Sheets.findControl("sampleDiv");
                     * spread.addCustomFunction(equalsFunction);
                     * ```
                     */
                    findTestArgument(): number;
                    /**
                     * Gets a value that indicates whether this function is branched by arguments as conditional.
                     * Used in conjunction with the findBranchArgument and findTestArgument.
                     * @returns {boolean} `true` if this instance is branched; otherwise, `false`.
                     * @example
                     * ```javascript
                     * function EqualsFunction() {
                     *     this.name = 'Equals';
                     *     this.maxArgs = 3;
                     *     this.minArgs = 3;
                     * }
                     * EqualsFunction.prototype = new GC.Spread.CalcEngine.Functions.Function();
                     * EqualsFunction.prototype.evaluate = function(logicalTest, valueIfTrue, valueIfFalse) {
                     *     return logicalTest ? valueIfTrue : valueIfFalse;
                     * }
                     * EqualsFunction.prototype.isBranch = function() {
                     *     return true;
                     * }
                     * EqualsFunction.prototype.findTestArgument = function() {
                     *     return 0;
                     * }
                     * EqualsFunction.prototype.findBranchArgument = function(logicalTestResult) {
                     *     if (logicalTestResult === true) {
                     *         return 1;
                     *     }
                     *     return 2;
                     * }
                     * var equalsFunction = new EqualsFunction();
                     * var spread = GC.Spread.Sheets.findControl("ss") || GC.Spread.Sheets.findControl("sampleDiv");
                     * spread.addCustomFunction(equalsFunction);
                     * ```
                     */
                    isBranch(): boolean;
                    /**
                     * Determines whether the evaluation of the function is dependent on the context in which the evaluation occurs.
                     * @returns {boolean} `true` if the evaluation of the function is dependent on the context; otherwise, `false`.
                     * @example
                     * ```javascript
                     * class ContextFunction extends GC.Spread.CalcEngine.Functions.Function {
                     *     constructor () {
                     *         super('CONTEXT', 0, 0);
                     *     }
                     *     isContextSensitive () {
                     *         return true;
                     *     }
                     *     evaluate (context) {
                     *         return context.row + "/" + context.column;
                     *     }
                     * }
                     * spread.addCustomFunction(new ContextFunction());
                     * spread.getActiveSheet().setFormula(0, 0, '=CONTEXT()');
                     * ```
                     */
                    isContextSensitive(): boolean;
                    /**
                     * Determines whether the function is volatile while it is being evaluated.
                     * The volatile function is re-evaluated when any cell in the workbook is changed.
                     * @returns {boolean} `true` if the function is volatile; otherwise, `false`.
                     * @example
                     * ```javascript
                     * class RandColorFunction extends GC.Spread.CalcEngine.Functions.Function {
                     *     constructor () {
                     *         super('RANDCOLOR', 0, 0);
                     *     }
                     *     isVolatile () {
                     *         return true;
                     *     }
                     *     evaluate () {
                     *         return "#" + Math.floor(Math.random() * 16777215).toString(16);
                     *     }
                     * }
                     * spread.addCustomFunction(new RandColorFunction());
                     * spread.getActiveSheet().setFormula(0, 0, '=RANDCOLOR()');
                     * ```
                     */
                    isVolatile(): boolean;
                }
            }

            module LanguagePackages{
                /**
                 * Gets or Sets the language package for calc engine.
                 * @param {string} languageName the calc engine language name string;
                 * @returns {string|void} If no value is set, return the calc engine language name string;
                 * @example
                 * ```javascript
                 * GC.Spread.CalcEngine.LanguagePackages.languagePackages('fr');
                 * GC.Spread.CalcEngine.LanguagePackages.languagePackages();
                 * ```
                 */
                function languagePackages(languageName?: string): any;
            }

        }

        module Commands{
            /**
             * Represents the key code.
             * @enum {number}
             * @example
             * ```javascript
             * //This example creates a custom action using the enter key.
             * var activeSheet = spread.getActiveSheet();
             * spread.commandManager().register('myCmd',
             *                 function ColorAction() {
             *                     //Click on a cell and press the Enter key.
             *                     activeSheet.getCell(activeSheet.getActiveRowIndex(), activeSheet.getActiveColumnIndex()).backColor("red");
             *                 }
             *             );
             * //Map the created action to the Enter key.
             * spread.commandManager().setShortcutKey('myCmd', GC.Spread.Commands.Key.enter, false, false, false, false);
             * ```
             */
            export enum Key{
                /**
                 * Indicates the left arrow key.
                 */
                left= 37,
                /**
                 * Indicates the right arrow key.
                 */
                right= 39,
                /**
                 * Indicates the up arrow key.
                 */
                up= 38,
                /**
                 * Indicates the down arrow key.
                 */
                down= 40,
                /**
                 * Indicates the Tab key.
                 */
                tab= 9,
                /**
                 * Indicates the Enter key.
                 */
                enter= 13,
                /**
                 * Indicates the Shift key.
                 */
                shift= 16,
                /**
                 * Indicates the Ctrl key.
                 */
                ctrl= 17,
                /**
                 * Indicates the space key.
                 */
                space= 32,
                /**
                 * Indicates the Alt key.
                 */
                altkey= 18,
                /**
                 * Indicates the Home key.
                 */
                home= 36,
                /**
                 * Indicates the End key.
                 */
                end= 35,
                /**
                 * Indicates the Page Up key.
                 */
                pup= 33,
                /**
                 * Indicates the Page Down key.
                 */
                pdn= 34,
                /**
                 * Indicates the Backspace key.
                 */
                backspace= 8,
                /**
                 * Indicates the Delete key.
                 */
                del= 46,
                /**
                 * Indicates the Esc key.
                 */
                esc= 27,
                /**
                 * Indicates the A key
                 */
                a= 65,
                /**
                 * Indicates the C key.
                 */
                c= 67,
                /**
                 * Indicates the V key.
                 */
                v= 86,
                /**
                 * Indicates the X key.
                 */
                x= 88,
                /**
                 * Indicates the Z key.
                 */
                z= 90,
                /**
                 * Indicates the Y key.
                 */
                y= 89
            }


            export class CommandManager{
                /**
                 * Represents a command manager.
                 * @class
                 * @param {Object} context The execution context for all commands in the command manager.
                 * @constructor
                 */
                constructor(context: Object);
                /**
                 * Executes a command and adds the command to UndoManager.
                 * @param {Object} commandOptions The options for the command.
                 * @param {string} commandOptions.cmd The command name, the field is required.
                 * @param {Object} [commandOptions.arg1] The command argument, the name and type are decided by the command definition.
                 * @param {Object} [commandOptions.arg2] The command argument, the name and type are decided by the command definition.
                 * @param {Object} [commandOptions.argN] The command argument, the name and type are decided by the command definition.
                 * @returns {boolean} The execute command result.
                 * @example
                 * ```javascript
                 * //For example, the following code executes the autoFitColumn command.
                 * var spread = GC.Spread.Sheets.findControl(document.getElementById("ss"));
                 * spread.commandManager().execute({cmd: "autoFitColumn", sheetName: "Sheet1", columns: [{col: 1}], rowHeader: false, autoFitType: GC.Spread.Sheets.AutoFitType.cell});
                 * ```
                 */
                execute(commandOptions: Object): boolean;
                /**
                 * Registers a command with the command manager.
                 * @param {string} name - The name of the command.
                 * @param {Object} command - The object that defines the command.
                 * @param {number|GC.Spread.Commands.Key} key - The key code.
                 * @param {boolean} ctrl - `true` if the command uses the Ctrl key; otherwise, `false`.
                 * @param {boolean} shift - `true` if the command uses the Shift key; otherwise, `false`.
                 * @param {boolean} alt - `true` if the command uses the Alt key; otherwise, `false`.
                 * @param {boolean} meta - `true` if the command uses the Command key on the Macintosh or the Windows key on Microsoft Windows; otherwise, `false`.
                 * @example
                 * ```javascript
                 * //For example, the following code registers the changeBackColor command and then executes the command.
                 * var command = {
                 *   canUndo: true,
                 *   execute: function (context, options, isUndo) {
                 *     var Commands = GC.Spread.Sheets.Commands;
                 *     options.cmd = "changeBackColor";
                 *     if (isUndo) {
                 *       Commands.undoTransaction(context, options);
                 *       return true;
                 *     } else {
                 *       Commands.startTransaction(context, options);
                 *       var sheet = context.getSheetFromName(options.sheetName);
                 *       var cell = sheet.getCell(options.row, options.col);
                 *       cell.backColor(options.backColor);
                 *       Commands.endTransaction(context, options);
                 *       return true;
                 *     }
                 *   }
                 * };
                 * var spread = GC.Spread.Sheets.findControl(document.getElementById("ss"));
                 * var commandManager = spread.commandManager();
                 * commandManager.register("changeBackColor", command);
                 * commandManager.execute({cmd: "changeBackColor", sheetName: spread.getSheet(0).name(), row: 1, col: 2, backColor: "red"});
                 * ```
                 */
                register(name: string,  command: Object,  key?: number|GC.Spread.Commands.Key,  ctrl?: boolean,  shift?: boolean,  alt?: boolean,  meta?: boolean): void;
                /**
                 * Binds a shortcut key to a command.
                 * @param {string} commandName The command name, setting commandName to undefined removes the bound command of the shortcut key.
                 * @param {number|GC.Spread.Commands.Key} key The key code, setting the key code to undefined removes the shortcut key of the command.
                 * @param {boolean} ctrl `true` if the command uses the Ctrl key; otherwise, `false`.
                 * @param {boolean} shift `true` if the command uses the Shift key; otherwise, `false`.
                 * @param {boolean} alt `true` if the command uses the Alt key; otherwise, `false`.
                 * @param {boolean} meta `true` if the command uses the Command key on the Macintosh or the Windows key on Microsoft Windows; otherwise, `false`.
                 * @example
                 * ```javascript
                 * //This example changes the behavior of default keys.
                 * var activeSheet = spread.getActiveSheet();
                 * //Change the default Up arrow key action to "Page Up" for the active cell.
                 * spread.commandManager().setShortcutKey('navigationPageUp', GC.Spread.Commands.Key.up, false, false, false, false);
                 * //Change the default Down arrow key action to "Page Down" for the active cell.
                 * spread.commandManager().setShortcutKey('navigationPageDown', GC.Spread.Commands.Key.down, false, false, false, false);
                 * ```
                 */
                setShortcutKey(commandName: string,  key?: number|GC.Spread.Commands.Key,  ctrl?: boolean,  shift?: boolean,  alt?: boolean,  meta?: boolean): void;
            }

            export class UndoManager{
                /**
                 * Represents the undo manager.
                 * @constructor
                 */
                constructor();
                /**
                 * Gets whether the redo operation is allowed.
                 * @returns {boolean} `true` if the redo operation is allowed; otherwise, `false`.
                 */
                canRedo(): boolean;
                /**
                 * Gets whether the undo operation is allowed.
                 * @returns {boolean} `true` if the undo operation is allowed; otherwise, `false`.
                 */
                canUndo(): boolean;
                /**
                 * Clears all of the undo stack and the redo stack.
                 */
                clear(): void;
                /**
                 * Get the redo stack.
                 * @returns {any[]} It returns an object. This  must include sheetName and cmd. the type of sheetName and cmd both is string.
                 */
                getRedoStack():any[];
                /**
                 * Get the undo stack.
                 * @returns {any[]} It returns is an object. This object must include sheetName and cmd. the type of sheetName and cmd both is string.
                 */
                getUndoStack():any[];
                /**
                 * Gets or sets the the undo/redo stack max size.
                 * @param {number} value this value should be greater or equal to 0.
                 * @returns {number | UndoManager} If no value is set. return the max size of undo/redo stack. otherwise return UndoManager.
                 */
                maxSize(value?: number):any;
                /**
                 * Redoes the last command.
                 * @returns {boolean} `true` if the redo operation is successful; otherwise, `false`.
                 */
                redo(): boolean;
                /**
                 * Undoes the last command.
                 * @returns {boolean} `true` if the undo operation is successful; otherwise, `false`.
                 */
                undo(): boolean;
            }
        }

        module Common{

            export interface ICurrentUserChangedEventArgs{
                oldCurrentUser?: GC.Spread.Common.IUser;
                newCurrentUser?: GC.Spread.Common.IUser;
            }


            export interface IDateTimeFormat{
                /**
                 * Specifies the day formatter for "ddd".
                 */
                abbreviatedDayNames?: string[];
                /**
                 * Specifies the month formatter for "MMM".
                 */
                abbreviatedMonthGenitiveNames?: string[];
                /**
                 * Specifies the month formatter for "MMM".
                 */
                abbreviatedMonthNames?: string[];
                /**
                 * Indicates the AM designator.
                 */
                amDesignator?: string;
                /**
                 * Specifies the day formatter for "dddd".
                 */
                dayNames?: string[];
                /**
                 * Specifies the standard date formatter for "F".
                 */
                fullDateTimePattern?: string;
                /**
                 * Specifies the standard date formatter for "D".
                 */
                longDatePattern?: string;
                /**
                 * Specifies the standard date formatter for "T" and "U".
                 */
                longTimePattern?: string;
                /**
                 * Specifies the standard date formatter for "M" and "m".
                 */
                monthDayPattern?: string;
                /**
                 * Specifies the formatter for "MMMM".
                 */
                monthGenitiveNames?: string[];
                /**
                 * Specifies the formatter for "M" or "MM".
                 */
                monthNames?: string[];
                /**
                 * Indicates the PM designator.
                 */
                pmDesignator?: string;
                /**
                 * Specifies the standard date formatter for "d".
                 */
                shortDatePattern?: string;
                /**
                 * Specifies the standard date formatter for "t".
                 */
                shortTimePattern?: string;
                /**
                 * Specifies the standard date formatter for "y" and "Y".
                 */
                yearMonthPattern?: string;
                /**
                 * Specifies the default date formatter for the case that cell formatter is null or "General" and cell value is a date with time.
                 */
                defaultDatePattern?: string;
            }


            export interface ILineBreakingStrategy{
                (text: string, option: GC.Spread.Common.ITextFormatOption) : string[];
            }


            export interface ILocalNumberFormat{
                /**
                 * this property key is number, this property value is format string.
                 */
                [K: number]: string;
            }


            export interface INumberFormat{
                /**
                 * Indicates the currency decimal point.
                 */
                currencyDecimalSeparator?: string;
                /**
                 * Indicates the currency thousand separator.
                 */
                currencyGroupSeparator?: string;
                /**
                 * Indicates the currency symbol.
                 */
                currencySymbol?: string;
                /**
                 * Indicates the decimal point.
                 */
                numberDecimalSeparator?: string;
                /**
                 * Indicates the thousand separator.
                 */
                numberGroupSeparator?: string;
                /**
                 * Indicates the separator for function arguments in a formula.
                 */
                listSeparator?: string;
                /**
                 * Indicates the separator for the constants in one row of an array constant in a formula.
                 */
                arrayListSeparator?: string;
                /**
                 * Indicates the separator for the array rows of an array constant in a formula.
                 */
                arrayGroupSeparator?: string;
                /**
                 * Specifies the DBNumber characters.
                 */
                dbNumber?: Object;
            }


            export interface IPredefinedFormats{
                /**
                 * The formats displayed in currency category.
                 */
                Currency?: Array<string>;
                /**
                 * The format displayed in accounting category.
                 */
                Accounting?: string;
                /**
                 * The format displayed in accounting category has no symbol.
                 */
                Comma?: string;
                /**
                 * The formats in date category.
                 */
                Date?: Array<string>;
                /**
                 * The formats in time category.
                 */
                Time?: Array<string>;
                Special?: {
                [K: string]: string;
                /**
                 * The formats in special category.
                 */
                };
                shortcut?: {
                Number?: string;
                Currency?: string;
                Accounting?: string;
                ShortDate?: string;
                LongDate?: string;
                Time?: string;
                Percentage?: string;
                Fraction?: string;
                Scientific?: string;
                Text?: string;
                Comma?: string;
                };
            }


            export interface ITextFormat{
                /**
                 * Specifies the characters can split up two lines. Default is [" ", "-"].
                 */
                lineBreakingChar?: string[];
                /**
                 * Specifies the characters that are not allowed at the start of a breaking line.
                 */
                lineBreakingForbidStart?: string[];
                /**
                 * Specifies the characters that are not allowed at the end of a breaking line.
                 */
                lineBreakingForbidEnd?: string[];
                /**
                 * Specifies the function for word split.
                 */
                lineBreakingStrategy?: GC.Spread.Common.ILineBreakingStrategy;
            }


            export interface ITextFormatOption{
                lineBreakingChar?: string[];
                lineBreakingForbidStart?: string[];
                lineBreakingForbidEnd?: string[];
            }


            export interface IUser{
                id: string;
                name: string;
                email?: string;
                avatar?: GC.Spread.Common.IImageSource;
            }


            export interface IUserManagerOptions{
                get: (userId: string) => Promise<GC.Spread.Common.IUser | undefined>;
                search: (query: string) => Promise<GC.Spread.Common.IUser[] | undefined>;
            }


            /**
             * @typedef GC.Spread.Common.IImageSource
             * @type {{kind: 'url';src: string;} | {kind: 'data';dataUrl: string;} | {kind: 'blobUrl';blobUrl: string;} | {kind: 'blob';blob: Blob;} | {kind: 'file';file: File;}
             */
            export type IImageSource = 
                {
                    kind: 'url';
                    src: string;
                    } | {
                        kind: 'data';
                        dataUrl: string;
                        } | {
                            kind: 'blobUrl';
                            blobUrl: string;
                            } | {
                                kind: 'blob';
                                blob: Blob;
                                } | {
                                    kind: 'file';
                                    file: File;
                                }
            

            export class CultureInfo{
                /**
                 * Represents the custom culture class. The member variable can be overwritten.
                 * @class
                 */
                constructor();
                /**
                 * Indicates the date time format fields.
                 * @type {Object}
                 * @example
                 * ```javascript
                 * // This example creates a custom culture.
                 * var myCulture = new GC.Spread.Common.CultureInfo();
                 * myCulture.NumberFormat.currencySymbol = "\u20ac"
                 * myCulture.NumberFormat.numberDecimalSeparator = ",";
                 * myCulture.NumberFormat.numberGroupSeparator = ".";
                 * myCulture.NumberFormat.arrayGroupSeparator = ";";
                 * myCulture.NumberFormat.arrayListSeparator = "\\";
                 * myCulture.NumberFormat.listSeparator = ";";
                 * myCulture.DateTimeFormat.amDesignator = "";
                 * myCulture.DateTimeFormat.pmDesignator = "";
                 * myCulture.DateTimeFormat.abbreviatedMonthNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.abbreviatedDayNames = ["So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"];
                 * myCulture.DateTimeFormat.abbreviatedMonthGenitiveNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.dayNames = ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"];
                 * myCulture.DateTimeFormat.fullDateTimePattern = "dddd, d. MMMM yyyy HH:mm:ss";
                 * myCulture.DateTimeFormat.longDatePattern = "dddd, d. MMMM yyyy";
                 * myCulture.DateTimeFormat.longTimePattern = "HH:mm:ss";
                 * myCulture.DateTimeFormat.monthDayPattern = "dd MMMM";
                 * myCulture.DateTimeFormat.monthNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.monthGenitiveNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.shortDatePattern = "dd.MM.yyyy";
                 * myCulture.DateTimeFormat.shortTimePattern = "HH:mm";
                 * myCulture.DateTimeFormat.yearMonthPattern = "MMMM yyyy";
                 * myCulture.DateTimeFormat.defaultDatePattern = "MM/dd/yyyy HH:mm:ss";
                 * //add one culture
                 * GC.Spread.Common.CultureManager.addCultureInfo("de-DE", myCulture);
                 * //switch to "de-DE" culture
                 * GC.Spread.Common.CultureManager.culture("de-DE");
                 * var d = new Date();
                 * //With culture
                 * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate() + 1)));
                 * activeSheet.getCell(1, 0).formatter("mmm");
                 * var dvalue = 12345.6789;
                 * activeSheet.setColumnWidth(0, 200);
                 * activeSheet.setColumnWidth(1, 200);
                 * activeSheet.setColumnWidth(2, 200);
                 * activeSheet.setValue(0, 0, dvalue);
                 * activeSheet.getCell(0, 0).formatter("###,###.00");
                 * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate() + 1)));
                 * //With culture
                 * activeSheet.getCell(3, 0).formatter("yyyy/mmm/dddd");
                 * activeSheet.setValue(3, 0, new Date());
                 * ```
                 */
                DateTimeFormat: GC.Spread.Common.IDateTimeFormat;
                /**
                 * The name displayed in the location dropdown or symbol dropdown in format dialog.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example set the displayName of CultureInfo:
                 * var culture = new GC.Spread.Common.CultureInfo();
                 * culture.displayName = "English(U.S.)"
                 * ```
                 */
                displayName: string;
                /*
                 * A list that defines which characters are considered specially equal in the calculation.
                 * @example
                 * ```javascript
                 * //This example sets the EqualCharacters of some characters:
                 * var culture = new GC.Spread.Common.CultureInfo();
                 * culture.equalCharacters = ["\u3002.", "\uff0c,", "\u24601\u4e00\u58f9"];
                 * sheet.setFormula(0, 0, '"1.hello"="\u4e00\u3002Hello"'); // formula result will be TRUE after set culture.
                 * ```
                 */
                equalCharacters: string[];
                /**
                 * The font script code be used by theme to find the typeface.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example set the fontScriptCode of CultureInfo:
                 * var culture = new GC.Spread.Common.CultureInfo();
                 * culture.fontScriptCode = "Hans";
                 * ```
                 */
                fontScriptCode?: string;
                /**
                 * The id can be used to specify CultureInfo in number format.
                 * @type {number}
                 * @example
                 * ```javascript
                 * //This example adds a new CultureInfo and affects the number format.
                 * let culture = new GC.Spread.Common.CultureInfo();
                 * culture.id = 0x407;
                 * culture.NumberFormat.numberDecimalSeparator = ",";
                 * culture.NumberFormat.numberGroupSeparator = ".";
                 * culture.NumberFormat.listSeparator = "";
                 * culture.name = function () { return "de-DE"; };
                 * GC.Spread.Common.CultureManager.addCultureInfo(culture.name(), culture);
                 * let formatter = new GC.Spread.Formatter.GeneralFormatter("[$-407]0,000.000");
                 * let result = formatter.format(100000); //result: '100.000,000'
                 * ```
                 */
                id: number;
                /**
                 * indicates the local number format built. It's a map whose keys are number and values are formatString.
                 * @type {Object}
                 * @example
                 * ```javascript
                 * //this is an example for LocalNumberFormat.
                 * var cultureInfo = new GC.Spread.Common.CultureInfo().
                 * cultureInfo.LocalNumberFormat = {
                 *  14:"yyyy-mm-dd",
                 *  15:"yyyy/mm/dd"
                 * }
                 * ```
                 */
                LocalNumberFormat: GC.Spread.Common.ILocalNumberFormat;
                /**
                 * Indicates all the number format fields.
                 * @type {Object}
                 * @example
                 * ```javascript
                 * // This example creates a custom culture.
                 * var myCulture = new GC.Spread.Common.CultureInfo();
                 * myCulture.NumberFormat.currencySymbol = "\u20ac"
                 * myCulture.NumberFormat.numberDecimalSeparator = ",";
                 * myCulture.NumberFormat.numberGroupSeparator = ".";
                 * myCulture.NumberFormat.arrayGroupSeparator = ";";
                 * myCulture.NumberFormat.arrayListSeparator = "\\";
                 * myCulture.NumberFormat.dbNumber = {
                 *    1: {letters: ['\u5146', '\u5343', '\u767e', '\u5341', '\u4ebf', '\u5343', '\u767e', '\u5341', '\u4e07', '\u5343', '\u767e', '\u5341', ''], // \u5146\u5343\u767e\u5341\u4ebf\u5343\u767e\u5341\u4e07\u5343\u767e\u5341
                 *        numbers: ['\u25cb', '\u4e00', '\u4e8c', '\u4e09', '\u56db', '\u4e94', '\u516d', '\u4e03', '\u516b', '\u4e5d'] }, // \u25cb\u4e00\u4e8c\u4e09\u56db\u4e94\u516d\u4e03\u516b\u4e5d
                 *    2: {letters: ['\u5146', '\u4edf', '\u4f70', '\u62fe', '\u4ebf', '\u4edf', '\u4f70', '\u62fe', '\u4e07', '\u4edf', '\u4f70', '\u62fe', ''], // \u5146\u4edf\u4f70\u62fe\u4ebf\u4edf\u4f70\u62fe\u4e07\u4edf\u4f70\u62fe
                 *        numbers: ['\u96f6', '\u58f9', '\u8d30', '\u53c1', '\u8086', '\u4f0d', '\u9646', '\u67d2', '\u634c', '\u7396']}, // \u96f6\u58f9\u8d30\u53c1\u8086\u4f0d\u9646\u67d2\u634c\u7396
                 *    3: {letters: null,
                 *        numbers: ['\uff10', '\uff11', '\uff12', '\uff13', '\uff14', '\uff15', '\uff16', '\uff17', '\uff18', '\uff19']} // \uff10\uff11\uff12\uff13\uff14\uff15\uff16\uff17\uff18\uff19
                 * };
                 * myCulture.NumberFormat.listSeparator = ";";
                 * myCulture.DateTimeFormat.amDesignator = "";
                 * myCulture.DateTimeFormat.pmDesignator = "";
                 * myCulture.DateTimeFormat.abbreviatedMonthNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.abbreviatedDayNames = ["So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"];
                 * myCulture.DateTimeFormat.abbreviatedMonthGenitiveNames = ["Jan", "Feb", "Mrz", "Apr", "Mai", "Jun", "Jul", "Aug", "Sep", "Okt", "Nov", "Dez", ""];
                 * myCulture.DateTimeFormat.dayNames = ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"];
                 * myCulture.DateTimeFormat.fullDateTimePattern = "dddd, d. MMMM yyyy HH:mm:ss";
                 * myCulture.DateTimeFormat.longDatePattern = "dddd, d. MMMM yyyy";
                 * myCulture.DateTimeFormat.longTimePattern = "HH:mm:ss";
                 * myCulture.DateTimeFormat.monthDayPattern = "dd MMMM";
                 * myCulture.DateTimeFormat.monthNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.monthGenitiveNames = ["Januar", "Februar", "M\xe4rz", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Dezember", ""];
                 * myCulture.DateTimeFormat.shortDatePattern = "dd.MM.yyyy";
                 * myCulture.DateTimeFormat.shortTimePattern = "HH:mm";
                 * myCulture.DateTimeFormat.yearMonthPattern = "MMMM yyyy";
                 * //add one culture
                 * GC.Spread.Common.CultureManager.addCultureInfo("de-DE", myCulture);
                 * //switch to "de-DE" culture
                 * GC.Spread.Common.CultureManager.culture("de-DE");
                 * var d = new Date();
                 * //With culture
                 * activeSheet.setValue(1, 0, new Date(d.setDate(d.getDate() + 1)));
                 * activeSheet.getCell(1, 0).formatter("mmm");
                 * var dvalue = 12345.6789;
                 * activeSheet.setColumnWidth(0, 200);
                 * activeSheet.setColumnWidth(1, 200);
                 * activeSheet.setColumnWidth(2, 200);
                 * activeSheet.setValue(0, 0, dvalue);
                 * activeSheet.getCell(0, 0).formatter("###,###.00");
                 * activeSheet.setValue(2, 0, new Date(d.setDate(d.getDate() + 1)));
                 * //With culture
                 * activeSheet.getCell(3, 0).formatter("yyyy/mmm/dddd");
                 * activeSheet.setValue(3, 0, new Date());
                 * ```
                 */
                NumberFormat: GC.Spread.Common.INumberFormat;
                /**
                 * The predefinedFormats is an object that describes part of the number format in format dialog.
                 * When opening the format dialog, designer read predefinedFormats in all CultureInfos to display culture related formats.
                 * @type {Object}
                 * @example
                 * ```javascript
                 * //This is an example for predefinedFormats:
                 * var culture = new GC.Spread.Common.CultureInfo()
                 * culture.predefinedFormats = {
                 *     Accounting: '_($* #,##0._);_($* (#,##0.);_($* "-"._);_(@_)',
                 *     Currency: [
                 *         "[$$-409]#,##0.",
                 *         "[$$-409]#,##0.;[Red][$$-409]#,##0.",
                 *         "[$$-409]#,##0._);([$$-409]#,##0.)",
                 *         "[$$-409]#,##0._);[Red]([$$-409]#,##0.)"
                 *     ],
                 *     Date: [
                 *         "m/d/yyyy",
                 *         "[$-409]dddd, mmmm dd, yyyy",
                 *         "m/d;@",
                 *         "m/d/yy;@",
                 *         "mm/dd/yy;@",
                 *         "[$-409]d-mmm;@",
                 *         "[$-409]d-mmm-yy;@",
                 *         "[$-409]mmm-yy;@",
                 *         "[$-409]mmmm-yy;@",
                 *         "[$-409]mmmm d, yyyy;@",
                 *         "[$-409]m/d/yy h:mm AM/PM;@",
                 *         "m/d/yy h:mm;@",
                 *         "[$-409]mmmmm;@",
                 *         "[$-409]mmmmm-yy;@",
                 *         "m/d/yyyy;@",
                 *         "[$-409]d-mmm-yyyy;@",
                 *     ],
                 *     Time: [
                 *         "[$-409]h:mm:ss AM/PM",
                 *         "h:mm;@",
                 *         "[$-409]h:mm AM/PM;@",
                 *         "h:mm:ss;@",
                 *         "[$-409]h:mm:ss AM/PM;@",
                 *         "mm:ss.0;@",
                 *         "[h]:mm:ss;@",
                 *         "[$-409]m/d/yy h:mm AM/PM;@",
                 *         "m/d/yy h:mm;@"
                 *     ],
                 *     Special: {
                 *         "Zip Code": "00000",
                 *         "Zip Code + 4": "00000-0000",
                 *         "Phone Number": "[<=9999999]###-####;(###) ###-####",
                 *         "Social Security Number": "000-00-0000"
                 *     }
                 * };
                 * ```
                 */
                predefinedFormats: GC.Spread.Common.IPredefinedFormats;
                /**
                 * Indicates the text format fields.
                 * @type {Object}
                 * @example
                 * ```javascript
                 * // This example modify culture line breaking strategy.
                 * var myCulture = GC.Spread.Common.CultureManager.getCultureInfo();
                 * myCulture.TextFormat.lineBreakingChar = [" ", "+"]; // can line breaking when "+"
                 * myCulture.TextFormat.lineBreakingForbidStart = ["\u300b"]; // the \u300bwon't be start of the breaking line
                 * myCulture.TextFormat.lineBreakingForbidEnd = ["\u300a"]; // the \u300a won't be end of the breaking line
                 * activeSheet.setValue(0, 0, "1. 1+2+3+4+5+6+7");
                 * activeSheet.getCell(0, 0).wordWrap(true);  // the "1. " won't be in a single line
                 * activeSheet.setValue(1, 0, "\u300aabc\u300b\u300adef\u300b\u300aghk\u300b");
                 * activeSheet.getCell(1, 0).wordWrap(true);  // the\u300a \u300bwill looks better
                 * ```
                 */
                TextFormat: GC.Spread.Common.ITextFormat;
                /**
                 * Get the name of the CultureInfo, such as 'de-DE'.
                 * @returns {string} The name of the CultureInfo
                 * @example
                 * ```javascript
                 * //This example adds and gets a new CultureInfo
                 * let cultureInfo = new GC.Spread.Common.CultureInfo();
                 * cultureInfo.id = 0x407;
                 * cultureInfo.NumberFormat.numberDecimalSeparator = ",";
                 * cultureInfo.NumberFormat.numberGroupSeparator = ".";
                 * cultureInfo.NumberFormat.listSeparator = "";
                 * cultureInfo.name = function () { return "de-DE"; };
                 * GC.Spread.Common.CultureManager.addCultureInfo(cultureInfo.name(), cultureInfo);
                 * console.log(cultureInfo === GC.Spread.Common.CultureManager.getCultureInfo("de-DE"));//output: true
                 * ```
                 */
                name(): string;
            }

            export class CultureManager{
                /**
                 * Represent a culture manager.
                 * @constructor
                 */
                constructor();
                /**
                 * Adds the cultureInfo or custom language into the culture manager.
                 * @static
                 * @param {string} cultureName The culture name to set.
                 * @param {GC.Spread.Common.CultureInfo} cultureInfo The cultureInfo set to the culture.
                 * @param {object} language The custom language set to the culture. If already set, it will overwrite the old language.
                 * @example
                 * ```javascript
                 * var myCulture = new GC.Spread.Common.CultureInfo();
                 * myCulture.NumberFormat.currencySymbol = "\u20ac"
                 * myCulture.NumberFormat.numberDecimalSeparator = ",";
                 * myCulture.NumberFormat.numberGroupSeparator = ".";
                 * myCulture.NumberFormat.arrayGroupSeparator = ";";
                 * myCulture.NumberFormat.arrayListSeparator = "\\";
                 * myCulture.NumberFormat.listSeparator = ";";
                 * //add one culture
                 * GC.Spread.Common.CultureManager.addCultureInfo("de-DE", myCulture);
                 * ```
                 */
                static addCultureInfo(cultureName: string,  culture: GC.Spread.Common.CultureInfo,  language?: object): void;
                /**
                 * Get or set the Sheets culture.
                 * @static
                 * @param {string} cultureName The culture name to get.
                 * @param {boolean} force Optional. If `true`, forces reapplication of the culture settings.
                 * @returns {string}. The current culture name of Sheets.
                 */
                static culture(cultureName?: string,  force?: boolean): string;
                /**
                 * Gets the specified cultureInfo. If no culture name, get current cultureInfo.
                 * @static
                 * @param {string | number} cultureName Culture name or culture ID
                 * @returns {GC.Spread.Common.CultureInfo} The specified cultureInfo object.
                 * @example
                 * ```javascript
                 * GC.Spread.Common.CultureManager.getCultureInfo(); // return the current culture info.
                 * GC.Spread.Common.CultureManager.getCultureInfo(1033); // return the culture info of culture id 1033, it's en culture.
                 * GC.Spread.Common.CultureManager.getCultureInfo('en-us'); // return the culture info of en.
                 * ```
                 */
                static getCultureInfo(cultureName?: string | number): GC.Spread.Common.CultureInfo;
                /**
                 * Gets the specified custom language.
                 * @static
                 * @param {string} cultureName The culture name to get.
                 * @returns {object} The specified object. Null if not define the language.
                 */
                static getLanguage(cultureName: string): object;
                /**
                 * Gets the specified or current working resources.
                 * @static
                 * @param {string | null} [cultureName] The culture name to get. If the cultureName is null, will return the current working resources.
                 * @returns {object} The specified or current working resources. Null if not define the language.
                 */
                static getResources(cultureName?: string): object;
            }

            export class Events{
                /**
                 * Define the events supported by UserManager.
                 * @class
                 */
                constructor();
                /**
                 * Indicates the event when the current user is changed.
                 * @type {string}
                 * @event
                 * @param eventParam *{@link GC.Spread.Common.IUser}* `oldCurrentUser` The old current user.
                 * @param eventParam *{@link GC.Spread.Common.IUser}* `newCurrentUser` The new current user.
                 * @example
                 * ```javascript
                 * GC.Spread.Common.UserManager.bind(GC.Spread.Common.Events.CurrentUserChanged, (event, args) => {
                 *     console.log(event);
                 *     console.log(args.oldCurrentUser);
                 *     console.log(args.newCurrentUser);
                 * });
                 * ```
                 */
                static CurrentUserChanged: string;
            }

            export class UserManager{
                /**
                 * Represent a user manager.
                 * @constructor
                 */
                constructor();
                /**
                 * Bind an event handler.
                 * @static
                 * @param {GC.Spread.Common.Events} event Event type.
                 * @param {Function} handler Event handler.
                 * @returns {void}
                 * @example
                 * ```javascript
                 * const users = spread.users;
                 * users.bind(GC.Spread.Common.Events.CurrentUserChanged, (event, args) => {
                 *     console.log(event);
                 *     console.log(args.oldCurrentUser);
                 *     console.log(args.newCurrentUser);
                 * });
                 * ```
                 */
                static bind(event: GC.Spread.Common.Events,  handler: Function): void;
                /**
                 * Configure the user manager options.
                 * @static
                 * @param {GC.Spread.Common.IUserManagerOptions} options The user manager options.
                 * @returns {void}
                 */
                static configure(options: GC.Spread.Common.IUserManagerOptions): void;
                /**
                 * Get or set the current user.
                 * @static
                 * @param {string} [userId] The current user id.
                 * @returns {string | undefined} The current user id.
                 * @example
                 * ```javascript
                 * // navigate from the login page, and get the login user id
                 * spread.users.current('715CFB19-2BB9-4B94-BE0C-08D0F9A019D4'); // { id: "715CFB19-2BB9-4B94-BE0C-08D0F9A019D4", name: "John Doe" }
                 * console.log(spread.users.getSync(spread.users.current()).name); // "John Doe"
                 * ```
                 */
                static current(userId?: string): string | undefined;
                /**
                 * Get a user by user id.
                 * @static
                 * @param {string} userId The user id.
                 * @returns {Promise<GC.Spread.Common.IUser | undefined>} The user.
                 */
                static get(userId: string): Promise<GC.Spread.Common.IUser | undefined>;
                /**
                 * Unbind an event handler.
                 * @static
                 * @param {GC.Spread.Common.Events} event Event type.
                 * @param {Function} handler Event handler.
                 * @returns {void}
                 */
                static unbind(event: GC.Spread.Common.Events,  handler: Function): void;
            }
        }

        module Formatter{

            export class FormatterBase{
                /**
                 * Represents a custom formatter with the specified format string.
                 * @class
                 * @param {string} format The format.
                 * @param {string} cultureName The culture name.
                 */
                constructor(format: string,  cultureName: string);
                /** Represents the type name string used for supporting serialization.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example creates a custom formatter.
                 * function MyFormatter(format, cultureName) {
                 *   GC.Spread.Formatter.FormatterBase.apply(this, arguments);
                 *   this.typeName = "MyFormatter";
                 *   this.formatter = format;
                 * }
                 * MyFormatter.prototype = new GC.Spread.Formatter.FormatterBase();
                 * MyFormatter.prototype.format = function (obj, options) {
                 *     if (typeof obj === "number") {
                 *         var colors = this.formatter.split(";");
                 *         if (obj >= 0) {
                 *             options && options.conditionalForeColor = colors[0];
                 *             return "PositiveOrZero: " + obj;
                 *         } else {
                 *             options && options.conditionalForeColor = colors[1];
                 *             return "Negative: " + obj;
                 *         }
                 *     }
                 *     return obj ? obj.toString() : "";
                 * };
                 * MyFormatter.prototype.parse = function (str) {
                 *     var index = str.indexOf(": ");
                 *     if (index >= 0) {
                 *         return +str.substr(index + 2);
                 *     } else {
                 *         return +str;
                 *     }
                 * };
                 * MyFormatter.prototype.toJSON = function () {
                 *     return {
                 *         typeName: this.typeName,
                 *         formatter: this.formatter
                 *     };
                 * };
                 * MyFormatter.prototype.fromJSON = function (settings) {
                 *     if (settings) {
                 *         this.formatter = settings.formatter;
                 *     }
                 * };
                 * var formatter = new MyFormatter("red;green");
                 * formatter.format(12345); // "PositiveOrZero: 12345"
                 * formatter.parse("PositiveOrZero: 12345"); // 12345
                 * ```
                 */
                typeName: string;
                /**
                 * Formats the specified object as a string with a conditional color. This function should be overwritten.
                 * @param {Object} obj - The object with cell data to format.
                 * @param {Object} [options] - The additional format data.
                 * @param {string} [options.conditionalForeColor] - The conditional foreColor when format pattern contains color patter, such as [red]###.##
                 * @returns {string} The formatted string.
                 * @example
                 * ```javascript
                 * //This example creates a custom formatter.
                 * function CustomFormatterTest() {
                 * }
                 * CustomFormatterTest.prototype = new GC.Spread.Formatter.FormatterBase();
                 * CustomFormatterTest.prototype.format = function (obj, options) {
                 *     //if the obj is color string, exp: red, blue.the text will render with obj color.
                 *     if (obj) {
                 *         options.conditionalForeColor = obj.toString()
                 *         return "My format result : " + obj.toString();
                 *     }
                 *     return "";
                 * };
                 * CustomFormatterTest.prototype.parse = function (str) {
                 *     if (!str) {
                 *         return "";
                 *     }
                 *     return str;
                 * }
                 * var sheet = spread.getActiveSheet();
                 * sheet.getCell(1, 0).formatter(new CustomFormatterTest());
                 * sheet.getCell(1, 0).value("red");
                 * ```
                 */
                format(obj: Object,  options?: Object): string;
                /**
                 * Loads the object state from the specified JSON string.
                 * @param {Object} settings The custom formatter data from deserialization.
                 * @example
                 * ```javascript
                 * //This example creates a custom formatter.
                 * function MyFormatter(format, cultureName) {
                 *   GC.Spread.Formatter.FormatterBase.apply(this, arguments);
                 *   this.typeName = "MyFormatter";
                 *   this.formatter = format;
                 * }
                 * MyFormatter.prototype = new GC.Spread.Formatter.FormatterBase();
                 * MyFormatter.prototype.format = function (obj, options) {
                 *     if (typeof obj === "number") {
                 *         var colors = this.formatter.split(";");
                 *         if (obj >= 0) {
                 *             options && options.conditionalForeColor = colors[0];
                 *             return "PositiveOrZero: " + obj;
                 *         } else {
                 *             options && options.conditionalForeColor = colors[1];
                 *             return "Negative: " + obj;
                 *         }
                 *     }
                 *     return obj ? obj.toString() : "";
                 * };
                 * MyFormatter.prototype.parse = function (str) {
                 *     var index = str.indexOf(": ");
                 *     if (index >= 0) {
                 *         return +str.substr(index + 2);
                 *     } else {
                 *         return +str;
                 *     }
                 * };
                 * MyFormatter.prototype.toJSON = function () {
                 *     return {
                 *         typeName: this.typeName,
                 *         formatter: this.formatter
                 *     };
                 * };
                 * MyFormatter.prototype.fromJSON = function (settings) {
                 *     if (settings) {
                 *         this.formatter = settings.formatter;
                 *     }
                 * };
                 * var formatter = new MyFormatter("red;green");
                 * formatter.format(12345); // "PositiveOrZero: 12345"
                 * formatter.parse("PositiveOrZero: 12345"); // 12345
                 * ```
                 */
                fromJSON(settings: Object): void;
                /**
                 * Parses the specified text. This function should be overwritten.
                 * @param {string} text The text.
                 * @returns {Object} The parsed object.
                 * @example
                 * ```javascript
                 * //This example creates a custom formatter.
                 * function CustomFormatterTest() {
                 * }
                 * CustomFormatterTest.prototype = new GC.Spread.Formatter.FormatterBase();
                 * CustomFormatterTest.prototype.format = function (obj, options) {
                 *     //if the obj is color string, exp: red, blue.the text will render with obj color.
                 *     if (obj) {
                 *         options.conditionalForeColor = obj.toString()
                 *         return "My format result : " + obj.toString();
                 *     }
                 *     return "";
                 * };
                 * CustomFormatterTest.prototype.parse = function (str) {
                 *     if (!str) {
                 *         return "";
                 *     }
                 *     return str;
                 * }
                 * var sheet = spread.getActiveSheet();
                 * sheet.getCell(1, 0).formatter(new CustomFormatterTest());
                 * sheet.getCell(1, 0).value("red");
                 * ```
                 */
                parse(str: string): Object;
                /**
                 * Saves the object state to a JSON string.
                 * @returns {Object} The custom formatter data.
                 * @example
                 * ```javascript
                 * //This example creates a custom formatter.
                 * function MyFormatter(format, cultureName) {
                 *   GC.Spread.Formatter.FormatterBase.apply(this, arguments);
                 *   this.typeName = "MyFormatter";
                 *   this.formatter = format;
                 * }
                 * MyFormatter.prototype = new GC.Spread.Formatter.FormatterBase();
                 * MyFormatter.prototype.format = function (obj, options) {
                 *     if (typeof obj === "number") {
                 *         var colors = this.formatter.split(";");
                 *         if (obj >= 0) {
                 *             options && options.conditionalForeColor = colors[0];
                 *             return "PositiveOrZero: " + obj;
                 *         } else {
                 *             options && options.conditionalForeColor = colors[1];
                 *             return "Negative: " + obj;
                 *         }
                 *     }
                 *     return obj ? obj.toString() : "";
                 * };
                 * MyFormatter.prototype.parse = function (str) {
                 *     var index = str.indexOf(": ");
                 *     if (index >= 0) {
                 *         return +str.substr(index + 2);
                 *     } else {
                 *         return +str;
                 *     }
                 * };
                 * MyFormatter.prototype.toJSON = function () {
                 *     return {
                 *         typeName: this.typeName,
                 *         formatter: this.formatter
                 *     };
                 * };
                 * MyFormatter.prototype.fromJSON = function (settings) {
                 *     if (settings) {
                 *         this.formatter = settings.formatter;
                 *     }
                 * };
                 * var formatter = new MyFormatter("red;green");
                 * formatter.format(12345); // "PositiveOrZero: 12345"
                 * formatter.parse("PositiveOrZero: 12345"); // 12345
                 * ```
                 */
                toJSON(): Object;
            }

            export class GeneralFormatter{
                /**
                 * Represents a formatter with the specified format mode and format string.
                 * @class
                 * @param {string} [format] The format.
                 * @param {string} [cultureName] The culture name.
                 */
                constructor(format?: string,  cultureName?: string);
                /**
                 * Formats the specified object as a string with a formatted data Object.
                 * @param {Object} obj The object with cell data to format.
                 * @param {Object} [formattedData] The object with formatted data.
                 * @param {Array} [formattedData.content] - The formatted data array, each item is an object that has two properties type and value, And it may contain these types: 'number', 'text', 'fillingChar', 'placeholder', 'exponent', 'decimalSeparator', 'groupSeparator', 'numberPlaceholder', 'percent', 'permille' and 'currency'. For example: {type: 'number', value: '123'}.
                 * @param {string} [formattedData.conditionalForeColor] - The conditional foreground color.
                 * @returns {string} The formatted string.
                 * @example
                 * ```javascript
                 * //This example uses the format method.
                 * var formatter = new GC.Spread.Formatter.GeneralFormatter("#,##0.00");
                 * var result = formatter.format(123456.789);
                 * console.log(result); // '123,456.79'
                 * ```
                 */
                format(obj: Object,  formattedData?: Object): string;
                /**
                 * Gets or sets the format string for this formatter.
                 * @param {string} value The format string for this formatter.
                 * @returns {string|GC.Spread.Formatter.GeneralFormatter} If no value is set, returns the formatter string for this formatter; otherwise, returns the formatter.
                 * @example
                 * ```javascript
                 * //This example gets the format string.
                 * var formatter = new GC.Spread.Formatter.GeneralFormatter("#,##0.00");
                 * var result = formatter.formatString();
                 * console.log(result); // '#,##0.00'
                 * ```
                 */
                formatString(value?: string): string | GC.Spread.Formatter.GeneralFormatter;
                /**
                 * Parses the specified text.
                 * @param {string} text The text.
                 * @returns {Object} The parsed object.
                 * @example
                 * ```javascript
                 * //This example uses the parse method.
                 * var formatter = new GC.Spread.Formatter.GeneralFormatter("#,##0.00");
                 * var result = formatter.parse("123,456.78");
                 * console.log(result); // 123456.78
                 * ```
                 */
                parse(str: string): Object;
            }
        }

        module Pivot{

            export interface IPivotItemInfo{
                fieldName: string;
                fieldItem: string;
            }


            export interface IPivotTableViewManager{
                add: (view: IPivotTableView) => boolean;
                save: (name: string) => boolean;
                remove: (name: string) => void;
                get: (name: string) => GC.Spread.Pivot.IPivotTableView;
                all: () => IPivotTableView[];
                apply: (name: string) => void;
            }


            /**
             * @typedef {object} GC.Spread.Pivot.ICalcFieldInfo
             * @property {string} fieldName  the field name of calc field
             * @property {string} formula  the formula of calc field
             */
            export type ICalcFieldInfo = 
                {
                    fieldName: string;
                    formula: string;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ICalcItemInfo
             * @property {string} sourceName The name of PivotTable sourceField name
             * @property {string} calcItemName The name of PivotTable sourceField calcItem name
             * @property {string} formula The formula of PivotTable sourceField calcItem formula
             * @property {number} priority The priority of PivotTable sourceField calcItem priority
             */
            export type ICalcItemInfo = 
                {
                    sourceName: string;
                    calcItemName: string;
                    formula: string;
                    priority: number;
                }
            

            /**
             * @typedef GC.Spread.Pivot.IDateGroupInfo
             * @property {GC.Pivot.DateGroupType} by The date unit of date group info.
             * @property {Date} [start]  The start date of group.
             * @property {Date} [end] The end date of group.
             * @property {number} [groupInterval] Step size grouped by days.
             */
            export type IDateGroupInfo = 
                {
                    start?: Date;
                    end?: Date;
                    by: GC.Pivot.DateGroupType;
                    groupInterval?: number;
                }
            

            /**
             * @typedef GC.Spread.Pivot.IDateGroupsInfo
             * @property {string} originFieldName The source field name.
             * @property {GC.Spread.Pivot.IDateGroupInfo[]} dateGroups The date group info.
             */
            export type IDateGroupsInfo = 
                {
                    originFieldName: string;
                    dateGroups: GC.Spread.Pivot.IDateGroupInfo[];
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IFieldInfo - the field info of a pivot table field
             * @property {string} fieldName the field name
             * @property {string} sourceName the source name
             * @property {GC.Pivot.PivotDataType} dataType  the field correspondence source data type
             * @property {GC.Spread.Pivot.PivotTableFieldType} pivotArea the field type, which effect the field in different area
             * @property {number} pivotIndex the index in the field area
             * @property {Date|number} [start] the min value of number type data field or the oldest of date type field
             * @property {Date|number} [end]  the max value of number type data field or the newest of date type field
             */
            export type IFieldInfo = 
                {
                    fieldName: string;
                    sourceName: string;
                    dataType: GC.Pivot.PivotDataType;
                    pivotArea: GC.Spread.Pivot.PivotTableFieldType;
                    pivotIndex: number;
                    start?: Date|number;
                    end?: Date|number;
                }
            

            /**
             * @typedef GC.Spread.Pivot.INumberGroupInfo
             * @property {string} originFieldName The source field name.
             * @property {object}  numberGroup The number group info
             * @property {number}  [numberGroup.start]  The start number of number group info.
             * @property {number}  [numberGroup.end]  The end number of number group info.
             * @property {number}  numberGroup.by   The step of number group info.
             */
            export type INumberGroupInfo = 
                {
                    originFieldName: string;
                    numberGroup: {
                        start?: number;
                        end?: number;
                        by: number;
                    };
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotArea
             * @property {GC.Spread.Pivot.PivotAreaType} [type]
             * @property {string} [fieldName]
             * @property {boolean} [labelOnly]
             * @property {boolean} [dataOnly]
             * @property {boolean} [grandRow]
             * @property {boolean} [grandCol]
             * @property {GC.Spread.Pivot.IPivotReference[]} [references]
             * @property {GC.Spread.Pivot.IPivotAreaOffset} [offset]
             */
            export type IPivotArea = 
                {
                    type?: GC.Spread.Pivot.PivotAreaType;
                    fieldName?: string;
                    labelOnly?: boolean;
                    dataOnly?: boolean;
                    grandRow?: boolean;
                    grandCol?: boolean;
                    references?: GC.Spread.Pivot.IPivotReference[];
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotAreaOffset
             * @property {number} row
             * @property {number} col
             * @property {number} [rowCount]
             * @property {number} [colCount]
             */
            export type IPivotAreaOffset = 
                {
                    row: number;
                    col:number;
                    rowCount?: number;
                    colCount?: number
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotAreasCollection
             * @property {Object.<string, GC.Spread.Pivot.IPivotArea[]>} - key indicates pivot table name, value indicates selected pivot areas.
             */
            export type IPivotAreasCollection = 
                {
                    [pivotTableName: string]: IPivotArea[];
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotConditionFilterInfo
             * @property {string} conditionByName
             * @property {GC.Pivot.IPivotCaptionConditionFilterInfo | GC.Pivot.IPivotDateConditionFilterInfo | GC.Pivot.IPivotTop10ConditionFilterInfo | GC.Pivot.IPivotValueConditionInfo} condition
             */
            export type IPivotConditionFilterInfo = 
                {
                    conditionByName: string;
                    condition: GC.Pivot.IPivotCaptionConditionFilterInfo | GC.Pivot.IPivotDateConditionFilterInfo | GC.Pivot.IPivotTop10ConditionFilterInfo | GC.Pivot.IPivotValueConditionInfo
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotNodeInfo
             * @property {GC.Spread.Pivot.IPivotItemInfo[]} fieldInfos the label field infos of node info.
             * @property {GC.Pivot.IValueInfo} valueInfo the value field info of node info.
             */
            export type IPivotNodeInfo = 
                {
                    fieldInfos: GC.Spread.Pivot.IPivotItemInfo[];
                    valueInfo: GC.Pivot.IValueInfo;
                }
            

            /**
             * @typedef {object} GC.Pivot.IOverwriteNodeInfo
             * @property {GC.Spread.Pivot.IPivotItemInfo[]} fieldInfos the label field infos of node info.
             * @property {GC.Pivot.IValueInfo} valueInfo the value field info of node info.
             * @property {number} value the value of node info.
             */
            export type IPivotOverwriteNodeInfo = 
                {
                    fieldInfos: GC.Spread.Pivot.IPivotItemInfo[];
                    valueInfo: GC.Pivot.IValueInfo;
                    value: number;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotReference
             * @property {string} fieldName
             * @property {boolean} [subtotal]
             * @property {string[]} [items]
             */
            export type IPivotReference = 
                {
                    fieldName: string;
                    subtotal?: boolean;
                    items?: string[];
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotShowDataAsInfo
             * @property {string} baseFieldName
             * @property {GC.Pivot.PivotShowDataAsBaseItemType} baseFieldItemType
             * @property {string} baseFieldItem
             */
            export type IPivotShowDataAsInfo = 
                {
                    showDataAs: GC.Pivot.PivotShowDataAs;
                    baseFieldName?:string;
                    baseFieldItemType?:GC.Pivot.PivotShowDataAsBaseItemType;
                    baseFieldItem?:string;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotStyle
             * @property {GC.Spread.Pivot.IPivotArea} pivotArea The pivotArea in pivot table.
             * @property {GC.Spread.Sheets.Style}  style The style use for pivotArea.
             */
            export type IPivotStyle = 
                {
                    pivotArea?: GC.Spread.Pivot.IPivotArea;
                    style?: GC.Spread.Sheets.Style;
                }
            

            /**
             * @property {boolean} [allowMultipleFiltersPerField]
             * @property {boolean} [fillDownLabels]
             * @property {boolean} [insertBlankLineAfterEachItem]
             * @property {GC.Spread.Pivot.GrandTotalPosition} grandTotalPosition
             * @property {GC.Spread.Pivot.SubtotalsPosition} subtotalsPosition
             * @property {GC.Spread.Pivot.DisplayFields} displayFieldsInPageFilterArea
             * @property {number} reportFilterFieldsPerColumn
             * @property {boolean} [bandRows]
             * @property {boolean} [bandColumns]
             * @property {boolean} [showRowHeader]
             * @property {boolean} [showColumnHeader]
             * @property {boolean} [showDrill]
             * @property {boolean} [showMissing]
             * @property {string | number} missingCaption
             * @property {number} rowLabelIndent
             * @property {boolean} [printDrill]
             * @property {boolean} [itemPrintTitles]
             * @property {boolean} [fieldPrintTitles]
             * @property {boolean} [showFilter]
             * @property {boolean} [showToolTip]
             * @property {boolean} [mergeItem]
             * @property {boolean} [isShowErrorValue]
             * @property {string} [errorValueInfo]
             * @property {string} [rowHeaderCaption]
             * @property {string} [colHeaderCaption]
             * @property {string} [showHeaders]
             * @property {GC.Spread.Pivot.CalcItemAggregation} [calcItemAggregation]
             * @property {boolean} [enableDataValueEditing]
             */
            export type IPivotTableOption = 
                {
                    allowMultipleFiltersPerField?: boolean;
                    fillDownLabels?: boolean;
                    insertBlankLineAfterEachItem?: boolean;
                    grandTotalPosition?: GC.Spread.Pivot.GrandTotalPosition;
                    subtotalsPosition?: GC.Spread.Pivot.SubtotalsPosition;
                    displayFieldsInPageFilterArea?: GC.Spread.Pivot.DisplayFields;
                    reportFilterFieldsPerColumn?: number;
                    bandRows?: boolean;
                    bandColumns?: boolean;
                    showRowHeader?: boolean;
                    showColumnHeader?: boolean;
                    showDrill?: boolean;
                    showMissing?: boolean;
                    missingCaption?: string | number;
                    rowLabelIndent?: number;
                    printDrill?: boolean;
                    itemPrintTitles?: boolean;
                    fieldPrintTitles?: boolean;
                    showFilter?: boolean;
                    showToolTip?: boolean;
                    mergeItem?: boolean;
                    isShowErrorValue?: boolean;
                    errorValueInfo?: string;
                    rowHeaderCaption?: string;
                    colHeaderCaption?: string;
                    showHeaders?: boolean;
                    calcItemAggregation?: CalcItemAggregation;
                    enableDataValueEditing?: boolean;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotTablePosition
             * @property {number} row indicates pivot table start row
             * @property {number} col indicates pivot table start col
             * @property {number} [sheetName] which sheet is pivot table on
             */
            export type IPivotTablePosition = 
                {
                    row: number;
                    col: number;
                    sheetName?: string;
                }
            

            /**
             * @typedef GC.Spread.Pivot.IPivotTableRange
             * @property {GC.Spread.Sheets.Range} page the filter area range
             * @property {GC.Spread.Sheets.Range} content the content area range
             */
            export type IPivotTableRange = 
                {
                    page: GC.Spread.Sheets.Range;
                    content: GC.Spread.Sheets.Range;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotTableView
             * @property {string} name
             * @property {GC.Spread.Pivot.ISerializeInfo} config
             */
            export type IPivotTableView = 
                {
                    name: string;
                    config: GC.Spread.Pivot.ISerializeInfo;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.IPivotTextFilterInfo
             * @property {GC.Spread.Pivot.ITextCollectionCondition} textItem
             */
            export type IPivotTextFilterInfo = 
                {
                    textItem: GC.Spread.Pivot.ITextCollectionCondition;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ISerializeFieldInfo
             * @property {string} conditionBySourceName
             * @property {string} displayName
             * @property {GC.Pivot.SubtotalType} [subtotal]
             * @property {GC.Spread.Pivot.IPivotTextFilterInfo | GC.Spread.Pivot.IPivotConditionFilterInfo} [labelFilter]
             * @property {GC.Spread.Pivot.IPivotConditionFilterInfo} [valueFilter]
             * @property {GC.Pivot.IPivotViewSortInfo} [sortInfo]
             */
            export type ISerializeFieldInfo = 
                {
                    sourceName: string;
                    displayName: string;
                    subtotal?:  GC.Pivot.SubtotalType;
                    labelFilter?:  GC.Spread.Pivot.IPivotTextFilterInfo | GC.Spread.Pivot.IPivotConditionFilterInfo;
                    valueFilter?:  GC.Spread.Pivot.IPivotConditionFilterInfo;
                    sortInfo?:  GC.Pivot.IPivotViewSortInfo;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ISerializeInfo
             * @property {GC.Spread.Pivot.PivotTableLayoutType} [layoutType]
             * @property {object} [options] the options of pivot table
             * @property {string} [theme] the theme of pivot table
             * @property {GC.Pivot.IDataPosition} [valuePosition] the valuePosition info of pivot table
             * @property {number[]} [pivotTablePosition] the position  of pivot table
             * @property {GC.Spread.Pivot.ISerializeFieldInfo[][]} [fieldsInfo] the fields info of pivot table
             * @property {GC.Spread.Pivot.IPivotStyle} [styles] the pivot area styles of pivot table
             * @property {object} [collapseItems] the collapse info of pivot table
             * @property {GC.Spread.Pivot.ISerializeShowDataAsInfo[]} [showDataAsList] the show data as info of pivot table
             */
            export type ISerializeInfo = 
                {
                    layoutType?:  GC.Spread.Pivot.PivotTableLayoutType;
                    options?: object;
                    theme?: string;
                    valuePosition?: GC.Pivot.IDataPosition;
                    pivotTablePosition?: number[];
                    fieldsInfo?:  GC.Spread.Pivot.ISerializeFieldInfo[][];
                    styles?: GC.Spread.Pivot.IPivotStyle[];
                    collapseItems?: object;
                    showDataAsList?: GC.Spread.Pivot.ISerializeShowDataAsInfo[];
                    showNoDataInfo?: object
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ISerializeShowDataAsInfo
             * @property {string} valueFieldName
             * @property {GC.Spread.Pivot.IPivotShowDataAsInfo} showDataAsInfo
             */
            export type ISerializeShowDataAsInfo = 
                {
                    valueFieldName: string;
                    showDataAsInfo: GC.Spread.Pivot.IPivotShowDataAsInfo;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ISourceFieldInfo
             * @property {string} name the name of source field
             * @property {GC.Pivot.PivotSourceFieldType} [fieldType] the pivot field type is calc field or group field
             */
            export type ISourceFieldInfo = 
                {
                    name: string;
                    fieldType?: GC.Pivot.PivotSourceFieldType;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ITextCollectionCondition
             * @property {string[]} list
             * @property {boolean} isAll
             */
            export type ITextCollectionCondition = 
                {
                    list: string[];
                    isAll: boolean;
                }
            

            /**
             * @typedef {object} GC.Spread.Pivot.ITextGroupInfo The text group info.
             * @property {string} originFieldName  The source field name.
             * @property {object} textGroup  The text group info
             * @property {string} textGroup.fieldName  The group field name.
             * @property {GC.Pivot.IStringGroupItems} textGroup.groupItems  The text group items info.
             */
            export type ITextGroupInfo = 
                {
                    originFieldName: string;
                    textGroup: {
                        fieldName: string;
                        groupItems: GC.Pivot.IStringGroupItems;
                    }
                }
            
            /**
             * Indicates whether PivotTable total contains the value of calcItem.
             */
            export enum CalcItemAggregation{
                /**
                 * PivotTable total include calcItem value.
                 */
                include= 0,
                /**
                 *  PivotTable total exclude calcItem value.
                 */
                exclude= 1
            }

            /**
             * Indicates the display direction of the pivot table's page area.
             * @enum {number}
             */
            export enum DisplayFields{
                /**
                 * Indicates the first vertical direction, then horizontal direction.
                 */
                downThenOver= 0,
                /**
                 * Indicates the first horizontal direction, then vertical direction.
                 */
                overThenDown= 1
            }

            /**
             * Indicates the direction of grand total in the pivot table.
             * @enum {number}
             */
            export enum GrandTotalPosition{
                /**
                 * Don't show grand total
                 */
                none= 0,
                /**
                 * Show grand total in row direction for pivot table.
                 */
                row= 1,
                /**
                 * Show grand total in column direction for pivot table.
                 */
                col= 2,
                /**
                 * Show grand total in row and column direction for pivot table.
                 */
                both= 3
            }

            /**
             * Indicates the pivot area axis type of pivot table.
             * @enum {number}
             */
            export enum PivotAreaAxisType{
                /**
                 * indicate grandTotal is on the row axis
                 */
                row= 0,
                /**
                 * indicate grandTotal is on the col axis
                 */
                col= 1
            }

            /**
             * Specifies the whether automatically generate the getPivotData formula or cell reference when choose pivot table data area. Default is getPivotData.
             * @enum {number}
             */
            export enum PivotAreaReference{
                /**
                 * Generate normal cell reference.
                 */
                cellReference= 0,
                /**
                 * Generate the getPivotData formula.
                 */
                getPivotData= 1
            }

            /**
             * Indicates the pivot area type of pivot table.
             * @enum {number}
             */
            export enum PivotAreaType{
                /**
                 * indicate the whole pivot table
                 */
                all= 0,
                /**
                 * indicate the filter button of the pivot table
                 */
                button= 1,
                /**
                 * indicate the corner of the pivot table
                 */
                corner= 5,
                /**
                 * indicate the top right area of the pivot table
                 */
                topRight= 7
            }

            /**
             * Indicates the pivot panel layout type.
             * @enum {number}
             */
            export enum PivotPanelLayoutType{
                /**
                 * stack
                 * @type {number}
                 */
                stack= 0,
                /**
                 * flow
                 * @type {number}
                 */
                flow= 1
            }

            /**
             * Indicates the pivot panel section.
             * @enum {number}
             */
            export enum PivotPanelSection{
                /**
                 * fields
                 * @type {number}
                 */
                fields= 1,
                /**
                 * area
                 * @type {number}
                 */
                area= 2,
                /**
                 * viewList
                 * @type {number}
                 */
                viewList= 4
            }

            /**
             * Indicates the pivot table field area.
             * @enum {number}
             */
            export enum PivotTableFieldType{
                /**
                 * Filter area
                 * @type {number}
                 */
                filterField= 0,
                /**
                 * Row area
                 * @type {number}
                 */
                rowField= 1,
                /**
                 * Column area
                 * @type {number}
                 */
                columnField= 2,
                /**
                 * Value area
                 * @type {number}
                 */
                valueField= 3
            }

            /**
             * Indicates the view layout type of pivot table.
             * @enum {number}
             */
            export enum PivotTableLayoutType{
                /**
                 * Compact layout
                 */
                compact= 0,
                /**
                 * Outline layout
                 */
                outline= 1,
                /**
                 * Tabular layout
                 */
                tabular= 2
            }

            /**
             * Indicates the position of subtotal in the pivot table.
             * @enum {number}
             */
            export enum SubtotalsPosition{
                /**
                 * Don't show subtotal.
                 */
                none= 0,
                /**
                 * Show subtotal at the top of the pivot table.
                 */
                top= 1,
                /**
                 * Show subtotal at the bottom of the pivot table.
                 */
                bottom= 2
            }


            export class CustomPivotTableThemeManager extends GC.Spread.Sheets.CustomThemeManagerBase{
                /**
                 * Represents a custom pivot table theme manager that can manage all custom pivot table themes.
                 * @class
                 * @param {GC.Spread.Sheets.Workbook} workbook The workbook.
                 * @extends GC.Spread.Sheets.CustomThemeManagerBase
                 */
                constructor(workbook: GC.Spread.Sheets.Workbook);
                /**
                 * add a new pivot table theme.
                 * @param {string | GC.Spread.Pivot.PivotTableTheme} theme the new pivot table theme or just a new pivot table theme name you want to add
                 * @returns {GC.Spread.Pivot.PivotTableTheme | undefined} return the newly added pivot table theme, if the named pivot table theme is existed, failed to add pivot table theme and return undefined
                 * @example
                 * ```javascript
                 * // add a new pivot table theme named "custom0"
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * let pivotTableStyle = spread.customPivotTableThemes.add("custom0");
                 * let wholeTableStyle = new GC.Spread.Pivot.PivotTableStyle();
                 * wholeTableStyle.backColor = "#0C66E4";
                 * pivotTableStyle.wholeTableStyle(wholeTableStyle);
                 * ```
                 */
                add(theme: string | GC.Spread.Pivot.PivotTableTheme): GC.Spread.Pivot.PivotTableTheme | undefined;
                /**
                 * get the pivot table themes collection.
                 * @returns Array<GC.Spread.Pivot.PivotTableTheme>
                 * @example
                 * ```javascript
                 * // get all pivot table themes
                 * let tableStylesCollection = spread.customPivotTableThemes.all();
                 * ```
                 */
                all(): Array<GC.Spread.Pivot.PivotTableTheme>;
                /**
                 * get the pivot table theme by name.
                 * @param {string} name the specific name of the pivot table theme to get
                 * @returns {GC.Spread.Pivot.PivotTableTheme | undefined} If the corresponding pivot table theme with the spefic name is found, return the theme; otherwise, return undefined.
                 * @example
                 * ```javascript
                 * // get pivot table theme
                 * pivotTableStyle = spread.customPivotTableThemes.get("custom0");
                 * ```
                 */
                get(name: string): GC.Spread.Pivot.PivotTableTheme | undefined;
                /**
                 * remove the pivot table theme by name.
                 * @param {string} name the specific name of the pivot table theme to remove
                 * @returns {void}
                 * @example
                 * ```javascript
                 * // delete pivot table theme
                 * spread.customPivotTableThemes.remove("custom0");
                 * ```
                 */
                remove(name: string): void;
                /**
                 * update the pivot table theme.
                 * @param {string} oldThemeName the specific name of the pivot table theme to update
                 * @param {GC.Spread.Pivot.PivotTableTheme} newTheme the specific name of the pivot table theme to update
                 * @returns {void}
                 * @example
                 * ```javascript
                 * // update pivot table theme
                 * tableStyle = spread.customPivotTableThemes.update("custom0", new GC.Spread.Pivot.PivotTableTheme());
                 * ```
                 */
                update(oldThemeName: string,  newTheme: GC.Spread.Pivot.PivotTableTheme): void;
            }

            export class PivotPanel{
                /**
                 * Represents the pivot panel of pivot table.
                 * @class
                 * @param {string} name Indicates the Pivot panel name.
                 * @param {GC.Spread.Pivot.PivotTable} pivotTable  Indicates the pivot table that is related pivot panel.
                 * @param {HTMLDivElement} host Indicates the container html element of the pivot panel.
                 * @example
                 * ```javascript
                 * //This example creates a pivot panel.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.suspendLayout();
                 * pivotTable.options.showRowHeader =true;
                 * pivotTable.options.showColumnHeader =true;
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Amount", "Sum of Amount",   GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var panel = new GC.Spread.Pivot.PivotPanel("myPivotPanel", pivotTable, document.getElementById("panel"));
                 * pivotTable.resumeLayout();
                 * ```
                 */
                constructor(name: string,  pivotTable: GC.Spread.Pivot.PivotTable,  host: HTMLDivElement);
                /**
                 * @description attach to a pivot table for pivot panel, then the pivot panel can control the bound pivot table.
                 * @param {GC.Spread.Pivot.PivotTable} pivotTable  Indicates the pivot table which is attached.
                 * @returns void
                 */
                attach(pivotTable: GC.Spread.Pivot.PivotTable): any;
                /**
                 * @description destroy the PivotPanel.
                 */
                destroy(): any;
                /**
                 * @description detach the pivot table for pivot panel, stop the pivot panel control bound pivot table.
                 * @returns void
                 */
                detach(): void;
                /**
                 * Gets the PivotPanel instance by the host element.
                 * @param {HTMLElement|string} host The host element or the host element id.
                 * @returns {GC.Spread.Pivot.PivotPanel} The PivotPanel instance.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * var panel = new GC.Spread.Pivot.PivotPanel("sourceData", pivotTable, document.getElementById("pivotPanel"));
                 * var pivotPanel = GC.Spread.Pivot.PivotPanel.findControl("pivotPanel");
                 * ```
                 */
                static findControl(host: HTMLElement|string): GC.Spread.Pivot.PivotPanel;
                /**
                 * @description set or get the panelLayout type.
                 * @param {GC.Spread.Pivot.PivotPanelLayoutType} [value] change the panelLayout option to stack or flow.
                 * @returns GC.Spread.Pivot.PivotPanelLayoutType | void
                 */
                panelLayout(value?: GC.Spread.Pivot.PivotPanelLayoutType): GC.Spread.Pivot.PivotPanelLayoutType | void;
                /**
                 * @description  get or set which sections are visible.
                 * @param {number} [value] get or set which sections are visible by the sum of GC.Spread.Pivot.PivotPanelSection enum.
                 * @returns {number|void} The visibility value.
                 * @example
                 * ```javascript
                 *  var visibility = pivotPanel.sectionVisibility() // 7;
                 *  visibility = visibility & ~GC.Spread.Pivot.PivotPanelSection.viewList; // 3
                 *  pivotPanel.sectionVisibility(visibility);
                 * ```
                 */
                sectionVisibility(value?: number): number | void;
            }

            export class PivotTable{
                /**
                 * Represents a PivotTable.
                 * @class
                 * @param {string} name Indicates the name of pivot table.
                 * @param {GC.Spread.Sheets.Worksheet} sheet Indicates the owner worksheet.
                 * @param {number} row Indicates the pivot table start row.
                 * @param {number} col Indicates the pivot table start column.
                 * @param {GC.Spread.Pivot.PivotTableLayoutType} layout Indicates the pivot table layout type.
                 * @param {string | GC.Spread.Pivot.PivotTableTheme} style Indicates the pivot table theme style or style name.
                 */
                constructor(name: string,  sheet?: GC.Spread.Sheets.Worksheet,  row?: number,  col?: number,  layout?: GC.Spread.Pivot.PivotTableLayoutType,  style?: string | GC.Spread.Pivot.PivotTableTheme,  options?: GC.Spread.Pivot.IPivotTableOption,  layoutModel?: any);
                /**
                 * Indicates the options of the PivotTable.
                 * @type {Object}
                 * @property {boolean} [allowMultipleFiltersPerField] Indicates whether use multiple filter in one field.
                 * @property {boolean} [fillDownLabels] Indicates show repeat label items or not.
                 * @property {boolean} [insertBlankLineAfterEachItem] Indicates whether insert a blank row at end of each item.
                 * @property {GC.Spread.Pivot.GrandTotalPosition} [grandTotalPosition] Indicates whether show grandtotal in row, column or both.
                 * @property {GC.Spread.Pivot.SubtotalsPosition} [subtotalsPosition] Indicates show subtotal top or bottom or not show.
                 * @property {GC.Spread.Pivot.DisplayFields} [displayFieldsInPageFilterArea] Indicates the field display in page area show first over then down or first down then over.
                 * @property {number} [reportFilterFieldsPerColumn] Indicates the number of report filer field per column.
                 * @property {boolean} [bandRows] Indicates show band row or not.
                 * @property {boolean} [bandColumns] Indicates show band column or not.
                 * @property {boolean} [showRowHeader] Indicates show row header style or not.
                 * @property {boolean} [showColumnHeader] Indicates show column header style or not.
                 * @property {boolean} [showDrill] Indicates show expand/collapse button or not.
                 * @property {boolean} [showMissing] Indicates whether the missingCaption option is effected.
                 * @property {boolean} [missingCaption] Indicates what value should be shown when the actual value is empty
                 * @property {boolean} [rowLabelIndent] Indicates the indent of the title of each level.
                 * @property {boolean} [printDrill] Print expand/collapse buttons when displayed on PivotTable.
                 * @property {boolean} [itemPrintTitles] Repeat row labels on each PivotTable.
                 * @property {boolean} [fieldPrintTitles] Set Print titles.
                 * @property {boolean} [showFilter] Indicates show filter button or not.
                 * @property {boolean} [showToolTip] Indicates show tooltip or not.
                 * @property {boolean} [mergeItem] Indicates whether merge and center the cells with labels.
                 * @property {boolean} [isShowErrorValue] Indicates whether the errorValueInfo option is effected.
                 * @property {boolean} [errorValueInfo] Indicates what value should be shown when the actual value is a error.
                 * @property {string} [rowHeaderCaption] Indicates what value should be shown in compact layout to replace Row Label.
                 * @property {string} [colHeaderCaption] Indicates what value should be shown in compact layout to replace Column Label.
                 * @property {boolean} [showHeaders] Indicates show field headers.
                 * @property {GC.Spread.Pivot.CalcItemAggregation} [calcItemAggregation] Indicates whether PivotTable total contains the value of calcItem.
                 * @property {boolean} [enableDataValueEditing] Indicates whether allow edit cell values of data area of pivot table.
                 */
                options: GC.Spread.Pivot.IPivotTableOption;
                /**
                 * Pivot table view manager for the pivot table.
                 * @type {GC.Spread.Pivot.IPivotTableViewManager}
                 */
                views: GC.Spread.Pivot.IPivotTableViewManager;
                /**
                 * @description Add a field to pivot table.
                 * @param {string} sourceName Indicates the source name of  the field.
                 * @param {string} displayName Indicates the display name of the field.
                 * @param {number} area Indicates which area will be added to.
                 * @param {GC.Pivot.SubtotalType} [subtotal] Indicates the summary type when the field add to value area.
                 * @param {number} [index] Indicates the field index of pivot table's field area.
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var subtotal = GC.Pivot.SubtotalType.count;
                 * pivotTable.add("Buyer", "Buyer", 1, subtotal, 0) //add a field to pivot table, and field name is displayName, field in row area
                 * ```
                 */
                add(sourceName: string,  displayName: string,  area: number,  subtotal?: GC.Pivot.SubtotalType,  index?: number): void;
                /**
                 * @description Add a calculated field, the calculated field can only add into value area of pivot table.
                 * @param {string} fieldName Indicates the calculated field name.
                 * @param {string} formula Indicates the calculated formula.
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("myPivotTable", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.addCalcField("PercentOfEach", "=Amount/454");
                 * ```
                 */
                addCalcField(fieldName: string,  formula: string): void;
                /**
                 * @description add a calcItem
                 * @param {string} sourceName The name of sourceField name
                 * @param {string} calcItemName The name of sourceField calcItem name
                 * @param {string} formula The formula of sourceField calcItem formula
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.addCalcItem("Buyer", "formula1", "=Buyer[Mom]+Buyer[Dad]");
                 * ```
                 */
                addCalcItem(sourceName: string,  calcItemName: string,  formula: string): void;
                /**
                 * Sets the rules using the pivot areas.
                 * @param {GC.Spread.Pivot.IPivotArea[]} pivotArea The pivotArea in pivot table.
                 * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} conditionalRule The rules set to the pivot area.
                 * @example
                 * ```javascript
                 * //This example uses the getRule method.
                 * var pivotTable = activeSheet.pivotTables.all()[0];
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.ScaleRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.threeScaleRule);
                 * rule.midColor("#12ff34");
                 * rule.midType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.midValue(50000);
                 * rule.maxColor("#EE3344");
                 * rule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.maxValue(400000);
                 * rule.minColor("#AAff34");
                 * rule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.minValue(5000);
                 * var AmericaPivotArea = {
                 *        dataOnly: true,
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["America"]
                 *        }]
                 *    }
                 * var BritainPivotArea = {
                 *        dataOnly: true,
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["Britain"]
                 *        }]
                 *    }
                 * pivotTable.addConditionalRule([AmericaPivotArea, BritainPivotArea], rule);
                 * ```
                 */
                addConditionalRule(pivotArea: GC.Spread.Pivot.IPivotArea[],  conditionalRule: GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase): void;
                /**
                 * @description Set the minimum visual column width for each Field item.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Date", "Date", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.autoFitColumn();
                 * ```
                 */
                autoFitColumn(): void;
                /**
                 * @description get or set sortOption for a field of pivot table.
                 * @param {string} fieldName Indicates the target field name.
                 * @param {boolean} enabled Indicates whether the field is in auto sort state.
                 * @returns {boolean | void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.workbook(document.getElementById("ss"));
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = spread.getSheet(0).pivotTables.add("pivotTable_1",sourceData,1,1,layout,theme,options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.sort("Type", { sortType: GC.Pivot.SortType.asc, sortValueFieldName: "Sum of Amount"});
                 * pivotTable.autoSortState("Buyer", false);
                 * pivotTable.sort("Buyer", { sortType: GC.Pivot.SortType.asc });
                 * pivotTable.sort("Buyer", { customSortCallback: function(fieldItemNameArray) {
                 *     return fieldItemNameArray.sort((a, b) => a.length - b.length);
                 *     }
                 * });
                 * ```
                 */
                autoSortState(fieldName: string,  enabled?: boolean): boolean | void;
                /**
                 * clear overwrite info list.
                 */
                clearOverwriteList(): void;
                /**
                 * @description Get or set collapse info for a field of pivot table.
                 * @param {string} fieldName Indicates the target field name.
                 * @param {string} item Indicates the collapse item name.
                 * @param {boolean} [isCollapse] Indicates the status of the collapse.
                 * @returns {boolean|void} whether is collapsed.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var subtotal = GC.Pivot.SubtotalType.count;
                 * pivotTable.add("Buyer", "Buyer", 1, subtotal, 0);
                 * pivotTable.add("Type", "Type", 1, subtotal, 1);
                 * var collapseValue = pivotTable.collapse("Buyer","Mom");
                 * pivotTable.collapse("Buyer","Mom", !collapseValue);
                 * ```
                 */
                collapse(fieldName: string,  item: string,  isCollapse?: boolean): boolean | void;
                /**
                 * Connect slicer with the PivotTable
                 * @param {string} name name of slicer
                 */
                connectSlicer(name: string): void;
                /**
                 * @description Get or set the values position.
                 * @param {GC.Pivot.DataPosition} [positionType] Indicates the position which is in row or column area.
                 * @param {number} [positionIndex] Indicates the order in field row or column list.
                 * @returns {GC.Pivot.IDataPosition|void} return the data position info of pivot table or void
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.dataPosition(1,0) //move values to row area and index is 0
                 * pivotTable.dataPosition();//{positionType:1,positionIndex:0}
                 * ```
                 */
                dataPosition(positionType?: GC.Pivot.DataPosition,  positionIndex?: number): GC.Pivot.IDataPosition | void;
                /**
                 * @description restore serialized pivot table data to a existed pivot table
                 * @param {object} serializeInfo serialized pivot table data.
                 * @param {GC.Spread.Pivot.PivotTableLayoutType} [serializeInfo.layoutType] the layoutType of pivot table
                 * @param {object} [serializeInfo.options] the options of pivot table
                 * @param {string} [serializeInfo.theme] the theme of pivot table
                 * @param {object} [serializeInfo.valuePosition] the valuePosition info of pivot table
                 * @param {array} [serializeInfo.pivotTablePosition] the position  of pivot table
                 * @param {array} [serializeInfo.fieldsInfo] the fields info of pivot table
                 * @param {array} [serializeInfo.styles] the pivot area styles of pivot table
                 * @param {object} [serializeInfo.collapseItems] the collapse info of pivot table
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var serialization = pivotTable.serialize();
                 * pivotTable.remove('Type');
                 * pivotTable.deserialize(serialization);
                 * ```
                 */
                deserialize(serializeInfo: GC.Spread.Pivot.ISerializeInfo): void;
                /**
                 * Disconnect slicer with PivotTable
                 * @param {string} ptName name of slicer
                 */
                disconnectSlicer(name: string): void;
                /**
                 * Get All Slicers connect with the PivotTable
                 * @returns {GC.Spread.Pivot.PivotTableItemSlicer[]} slicers connect with the PivotTable
                 */
                getAllSlicers(): GC.Spread.Pivot.PivotTableItemSlicer[];
                /**
                 * @description get all calculated fields's info.
                 * @return {GC.Spread.Pivot.ICalcFieldInfo[]} return all calculated fields's info.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.addCalcField("PercentOfEach", "=Amount/454");
                 * pivotTable.getCalcFields();
                 * ```
                 */
                getCalcFields(): GC.Spread.Pivot.ICalcFieldInfo[];
                /**
                 * @description get calcItems information
                 * @param {string} sourceName The name of sourceField name
                 * @returns {GC.Spread.Pivot.ICalcItemInfo[]}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.addCalcItem("Buyer", "formula1", "=Buyer[Mom]+Buyer[Dad]");
                 * pivotTable.getCalcItems("Buyer");
                 * ```
                 */
                getCalcItems(sourceName?: string): GC.Spread.Pivot.ICalcItemInfo[];
                /**
                 * Gets the rules using the pivot area.
                 * @param {GC.Spread.Pivot.IPivotArea} pivotArea The pivotArea in pivot table.
                 * @returns {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase[]} The rules from the pivot area.
                 * @example
                 * ```javascript
                 * //This example uses the getRule method.
                 * var pivotTable = activeSheet.pivotTables.all()[0];
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.ScaleRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.threeScaleRule);
                 * rule.midColor("#12ff34");
                 * rule.midType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.midValue(50000);
                 * rule.maxColor("#EE3344");
                 * rule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.maxValue(400000);
                 * rule.minColor("#AAff34");
                 * rule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.minValue(5000);
                 * var AmericaPivotArea = {
                 *        dataOnly: true
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["America"]
                 *        }]
                 *    }
                 * var BritainPivotArea = {
                 *        dataOnly: true
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["Britain"]
                 *        }]
                 *    }
                 * pivotTable.addConditionalRule([AmericaPivotArea, BritainPivotArea], rule);
                 * var ruleTest = pivotTable.getConditionalRules(BritainPivotArea);
                 * alert(ruleTest[0].midValue());
                 * ```
                 */
                getConditionalRules(pivotArea: GC.Spread.Pivot.IPivotArea): GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase[];
                /**
                 * @description Get field information from pivot table by field name
                 * @param {string} fieldName the field's name
                 * @returns {GC.Spread.Pivot.IFieldInfo} return a field information
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.getField("Type");
                 * ```
                 */
                getField(fieldName: string): GC.Spread.Pivot.IFieldInfo;
                /**
                 * @description Get all field information from pivot table or one of pivot table area
                 * @param {GC.Spread.Pivot.PivotTableFieldType} [area] area index
                 * @returns {GC.Spread.Pivot.IFieldInfo[]} return all field information in the pivot table area
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.getFieldsByArea(GC.Spread.PivotTableFieldType.columnField);
                 * ```
                 */
                getFieldsByArea(area?: GC.Spread.Pivot.PivotTableFieldType): GC.Spread.Pivot.IFieldInfo[];
                /**
                 * @description get all items from pivot table by field name
                 * @param {string} fieldName the field name of pivot table field
                 * @param {boolean} sorted whether to return the sorted items
                 * @returns {string[]}return all items of pivot table field
                 */
                getItemsByField(fieldName: string,  sorted?: boolean): any;
                /**
                 * compose overwrite info from sheet row and column.
                 * @param {number} row sheet row of the cell.
                 * @param {number} col sheet column of the cell.
                 * @returns {GC.Spread.Pivot.IPivotNodeInfo} The node info want to get.
                 */
                getNodeInfo(row: number,  col: number): GC.Spread.Pivot.IPivotNodeInfo;
                /**
                 * get value by node info.
                 * @param {GC.Spread.Pivot.IPivotNodeInfo} nodeInfo The node info want to get.
                 * @returns {number} value of node info.
                 */
                getNodeValue(nodeInfo: GC.Spread.Pivot.IPivotNodeInfo): number;
                /**
                 * get all overwrite info of pivot cache.
                 * @returns {GC.Spread.Pivot.IPivotOverwriteNodeInfo[]} overwrite info list.
                 */
                getOverwriteList(): GC.Spread.Pivot.IPivotOverwriteNodeInfo[];
                /**
                 * @description get the sheet ranges that corresponding to the specific pivotArea.
                 * @param {GC.Spread.Pivot.IPivotArea} pivotArea the specific pivotArea
                 * @returns {GC.Spread.Sheets.Range[]} ranges the sheet ranges that corresponding to the specific pivotArea.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Date", "Date", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var pivotArea = {
                 *     dataOnly: false,
                 *     references: [
                 *         {
                 *             fieldName: "Buyer",
                 *             items: ["Mom", "Dad"]
                 *         }
                 *     ]
                 * };
                 * let ranges = pivotTable.getPivotAreaRanges(pivotArea);
                 * ```
                 */
                getPivotAreaRanges(pivotArea: GC.Spread.Pivot.IPivotArea): GC.Spread.Sheets.Range[];
                /**
                 * @description get PivotTable Details
                 * @param {Array} pivotItemInfo The pivot details information list
                 * @param {string} pivotItemInfo.fieldName The name of field
                 * @param {string} pivotItemInfo.fieldItem The name of field item
                 * @returns {array|void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.getPivotDetails([{fieldName:"Buyer", fieldItem:"Kelly"}]);
                 * ```
                 */
                getPivotDetails(pivotItemInfo: GC.Spread.Pivot.IPivotItemInfo[]): any[][] | void;
                /**
                 * @description Get the range of PivotTable, consist of page & content. They are readonly, change the range will not take any effect.
                 * @returns {GC.Spread.Pivot.IPivotTableRange} return current pivot table range.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.filterField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Date", "Date", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var ranges = pivotTable.getRange();
                 * console.log(ranges.page, ranges.content);
                 * ```
                 */
                getRange(): GC.Spread.Pivot.IPivotTableRange;
                /**
                 * @description get PivotTable data refer
                 * @returns {string}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.getSource();
                 * ```
                 */
                getSource(): string;
                /**
                 * @description get PivotTable source field information
                 * @returns {GC.Spread.Pivot.ISourceFieldInfo[]}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.addCalcField("calcField", "=Amount*2");
                 * pivotTable.getSourceFields();
                 * ```
                 */
                getSourceFields(): GC.Spread.Pivot.ISourceFieldInfo[];
                /**
                 * @description Get style by the specific pivotArea.
                 * @param {GC.Spread.Pivot.IPivotArea} pivotArea the specific pivotArea
                 * @returns GC.Spread.Sheets.Style
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Date", "Date", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var pivotArea = {
                 *     dataOnly: false,
                 *     references: [
                 *         {
                 *             fieldName: "Buyer",
                 *             items: ["Mom", "Dad"]
                 *         }
                 *     ]
                 * };
                 * var style = new GC.Spread.Sheets.Style();
                 * redBack.backColor = '#ff0000';
                 * pivotTable.setStyle(pivotArea, style);
                 * pivotTable.getStyle(pivotArea);
                 * ```
                 */
                getStyle(pivotArea: GC.Spread.Pivot.IPivotArea): GC.Spread.Sheets.Style;
                /**
                 * Gets or sets a style name for the pivot table.
                 * @returns {string} returns the pivot table style name.
                 */
                getThemeName(): string | undefined;
                /**
                 * @description Group the items of field
                 * @param {GC.Spread.Pivot.ITextGroupInfo | GC.Spread.Pivot.INumberGroupInfo | GC.Spread.Pivot.IDateGroupsInfo} groupInfo Indicates the grouped info.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["19-Jan","David","Books",120],
                 *                   ["20-Jan","Dad","Food",160],
                 *                   ["21-Jan","David","Sports",15],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 8, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var groupInfo = {
                 *     originFieldName: "Buyer",
                 *     textGroup: {
                 *         fieldName: "FamilyMembers",
                 *         groupItems: {
                 *              "parent": ["Mom", "Dad"],
                 *              "children": ["David", "Kelly"]
                 *         }
                 *     }
                 * };
                 * pivotTable.group(groupInfo);
                 * pivotTable.add("FamilyMembers", "FamilyMembers", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * sheet.resumePaint();
                 * ```
                 */
                group(groupInfo: GC.Spread.Pivot.ITextGroupInfo | GC.Spread.Pivot.INumberGroupInfo | GC.Spread.Pivot.IDateGroupsInfo): void;
                /**
                 * Whether the slicer is connected with the PivotTable
                 * @param {string} name name of slicer
                 * @returns {boolean} Whether the slicer is connected with the PivotTable
                 */
                isConnectedSlicer(name: string): boolean;
                /**
                 * Get or set label filter info for a field.
                 * @param {string} fieldName Indicates the target field name of pivot table.
                 * @param {object} [filterInfo] Indicates the label filter info when set.
                 * @returns {GC.Spread.Pivot.IPivotTextFilterInfo | GC.Spread.Pivot.IPivotConditionFilterInfo} return pivot table labelFilter information
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.suspendLayout();
                 * pivotTable.options.showRowHeader = true;
                 * pivotTable.options.showColumnHeader = true;
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * var condition = { conType: GC.Pivot.PivotConditionType.caption, operator: GC.Pivot.PivotCaptionFilterOperator.contains, val: ["Mom"] };
                 * var filterInfo = { condition };
                 * pivotTable.labelFilter("Buyer", filterInfo);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.resumeLayout();
                 * ```
                 */
                labelFilter(fieldName: string,  filterInfo?: GC.Spread.Pivot.IPivotTextFilterInfo | GC.Spread.Pivot.IPivotConditionFilterInfo | null): GC.Spread.Pivot.IPivotTextFilterInfo | GC.Spread.Pivot.IPivotConditionFilterInfo | void;
                /**
                 * @description Get or set the layoutType of pivot table
                 * @param {GC.Spread.Pivot.PivotTableLayoutType} [type] Indicates the pivot Table layout Type
                 * @returns {GC.Spread.Pivot.PivotTableLayoutType} If no parameters are passed in, get the current layout type.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTableLayoutType = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * pivotTable.layoutType(pivotTableLayoutType);
                 * pivotTable.layoutType();//GC.Spread.Pivot.PivotTableLayoutType.compact
                 * ```
                 */
                layoutType(type?: GC.Spread.Pivot.PivotTableLayoutType): GC.Spread.Pivot.PivotTableLayoutType;
                /**
                 * @description Get or set pivot table name.
                 * @param {string} name Indicates the pivot table name.
                 * @returns {string | void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.name("pivotTable_2")
                 * console.log(pivotTable.name()); //pivotTable_2
                 * ```
                 */
                name(name?: string): string | void;
                /**
                 * @description Get or set pivot table start position, the position of pivot table will auto change when there are enough cells to put the pivot table
                 * @param {number} [row] Indicates pivot table start row.
                 * @param {number} [col] Indicates pivot table start col.
                 * @param {string} [sheetName] Which sheet is pivot table on.
                 * @returns {GC.Spread.Pivot.IPivotTablePosition | void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var toSheet = spread.getSheet(2);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.position(10,10,toSheet.name());
                 * pivotTable.position(); //{row:10,col:10, sheetName: "Sheet3"}
                 * ```
                 */
                position(row?: number,  col?: number,  sheetName?: string): GC.Spread.Pivot.IPivotTablePosition | void;
                /**
                 * @description  Refresh fields Layout, re calc all field data in sheet.
                 */
                refresh(): void;
                /**
                 * @description Delete a field by name.
                 * @param {string} fieldName Indicates the fieldName which will be removed.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer", "Buyer", 1, GC.Pivot.SubtotalType.count, 0);
                 * pivotTable.remove("Buyer");
                 * ```
                 */
                remove(fieldName: string):void;
                /**
                 * @description remove a calculated field
                 * @param {string} fieldName Indicates the calculated field name.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme);
                 * pivotTable.addCalcField("PercentOfEach", "=Amount/454");
                 * var calcFieldsInfo = pivotTable.getCalcFields();
                 * pivotTable.removeCalcField(calcFieldsInfo[0].fieldName);
                 * ```
                 */
                removeCalcField(fieldName: string): void;
                /**
                 * @description remove a calcItem
                 * @param {string} sourceName The name of sourceField name
                 * @param {string} calcItemName The name of sourceField calcItem name
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.addCalcItem("Buyer", "formula1", "=Buyer[Mom]+Buyer[Dad]");
                 * pivotTable.removeCalcItem("Buyer", "formula1");
                 * ```
                 */
                removeCalcItem(sourceName: string,  calcItemName: string): void;
                /**
                 * Remove the rule of the pivot table.
                 * @param {GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase} conditionalRule The rules set to the pivot table.
                 * @example
                 * ```javascript
                 * //This example uses the getRule method.
                 * var pivotTable = activeSheet.pivotTables.all()[0];
                 * var rule = new GC.Spread.Sheets.ConditionalFormatting.ScaleRule();
                 * rule.ruleType(GC.Spread.Sheets.ConditionalFormatting.RuleType.threeScaleRule);
                 * rule.midColor("#12ff34");
                 * rule.midType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.midValue(50000);
                 * rule.maxColor("#EE3344");
                 * rule.maxType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.maxValue(400000);
                 * rule.minColor("#AAff34");
                 * rule.minType(GC.Spread.Sheets.ConditionalFormatting.ScaleValueType.number);
                 * rule.minValue(5000);
                 * var AmericaPivotArea = {
                 *        dataOnly: true,
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["America"]
                 *        }]
                 *    }
                 * var BritainPivotArea = {
                 *        dataOnly: true,
                 *        references: [{
                 *            fieldName: "Country",
                 *            items: ["Britain"]
                 *        }]
                 *    }
                 * pivotTable.addConditionalRule([AmericaPivotArea, BritainPivotArea], rule);
                 * pivotTable.removeConditionalRule(rule);
                 * ```
                 */
                removeConditionalRule(conditionalRule: GC.Spread.Sheets.ConditionalFormatting.ConditionRuleBase): void;
                /**
                 * @description Stop hold off update field, end of suspendLayout effect, it must be in pairs with suspendLayout.
                 */
                resumeLayout(): void;
                /**
                 * @description get serialized pivot table data
                 * @returns {GC.Spread.Pivot.ISerializeInfo} serialized pivot table data
                 */
                serialize(): GC.Spread.Pivot.ISerializeInfo;
                /**
                 * Set overwrite value to pivot cache.
                 * @param {GC.Spread.Pivot.IPivotNodeInfo} nodeInfo The node info to be set.
                 * @param {number} [value] the value set to the node info, if value is null or undefined, will remove the node info's value.
                 */
                setNodeValue(nodeInfo: GC.Spread.Pivot.IPivotNodeInfo,  value?: number): void;
                /**
                 * @description Set or remove style to the specific pivotArea.
                 * @param {GC.Spread.Pivot.IPivotArea} pivotArea the specific pivotArea
                 * @param {GC.Spread.Sheets.Style} style the style set to the specific pivotArea, null or undefined to remove style of the specific pivotArea.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Date", "Date", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var pivotArea = {
                 *     dataOnly: false,
                 *     references: [
                 *         {
                 *             fieldName: "Buyer",
                 *             items: ["Mom", "Dad"]
                 *         }
                 *     ]
                 * };
                 * var style = new GC.Spread.Sheets.Style();
                 * style.backColor = '#ff0000';
                 * pivotTable.setStyle(pivotArea, style);
                 * ```
                 */
                setStyle(pivotArea: GC.Spread.Pivot.IPivotArea,  style: GC.Spread.Sheets.Style): void;
                /**
                 * @description get or set a value field's  showDataAs info.
                 * @param {string} fieldName the value field name the show value as will apply.
                 * @param {object} [showDataAsInfo]  the show data as info apply to value field
                 * @param {number} showDataAsInfo.showDataAs  the show data as type.
                 * @param {string} [showDataAsInfo.baseFieldName]  the base on field name
                 * @param {GC.Pivot.PivotShowDataAsBaseItemType} [showDataAsInfo.baseFieldItemType] the base on field item type
                 * @param {string} [showDataAsInfo.baseFieldItem]  the base on item name
                 * @returns {GC.Spread.Pivot.IPivotShowDataAsInfo}
                 */
                showDataAs(fieldName: string,  showDataAsInfo?: GC.Spread.Pivot.IPivotShowDataAsInfo): GC.Spread.Pivot.IPivotShowDataAsInfo | void;
                /**
                 * @description set or get field 'show items with no data' information
                 * @param {string} fieldName The name of cache field name
                 * @param {boolean} isShow The flag indicates whether items without data need to be displayed
                 * @returns {boolean}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Date","Date",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.showNoData("Buyer", true);
                 * ```
                 */
                showNoData(cacheFieldName: string,  isShow: boolean): boolean;
                /**
                 * @description get or set sort for a field of pivot table.
                 * @param {string} fieldName Indicates the target field name.
                 * @param {GC.Pivot.IPivotViewSortInfo} sortInfo Indicates the sort info.
                 * @returns {GC.Pivot.IPivotViewSortInfo | void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.workbook(document.getElementById("ss"));
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = spread.getSheet(0).pivotTables.add("pivotTable_1",sourceData,1,1,layout,theme,options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField);
                 * pivotTable.sort("Type", { sortType: GC.Pivot.SortType.asc, sortValueFieldName: "Sum of Amount"});
                 * pivotTable.sort("Buyer", { sortType: GC.Pivot.SortType.asc });
                 * pivotTable.sort("Buyer", { customSortCallback: function(fieldItemNameArray) {
                 *     return fieldItemNameArray.sort((a, b) => a.length - b.length);
                 *     }
                 * });
                 * ```
                 */
                sort(fieldName: string,  sortInfo: GC.Pivot.IPivotViewSortInfo): GC.Pivot.IPivotViewSortInfo | void;
                /**
                 * @description set or get field show subtotal position information.
                 * @param {string} fieldName Indicates the field name.
                 * @param {GC.Spread.Pivot.SubtotalsPosition} position The indicates set whether subtotal position, only top and bottom is supported.
                 * @returns {GC.Spread.Pivot.SubtotalsPosition}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Date","Date",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.subtotalPosition("Buyer", GC.Spread.Pivot.SubtotalsPosition.top);
                 * ```
                 */
                subtotalPosition(fieldName: string,  position: GC.Spread.Pivot.SubtotalsPosition): GC.Spread.Pivot.SubtotalsPosition | void;
                /**
                 * Get or set SubtotalType for a field.
                 * @param {string} fieldName Indicates the target field name of pivot table.
                 * @param {GC.Pivot.SubtotalType} type Indicates the subtotal type to set.
                 * @returns {GC.Pivot.SubtotalType | void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * var subtotalType = GC.Pivot.SubtotalType.average;
                 * pivotTable.subtotalType("Buyer", subtotalType) //set a subtotalType for a Field of name is "fieldName"
                 * ```
                 */
                subtotalType(fieldName: string,  type?:GC.Pivot.SubtotalType): GC.Pivot.SubtotalType | void;
                /**
                 * @description set or get whether the field displays subtotal information.
                 * @param {string} fieldName Indicates the field name.
                 * @param {boolean} position Indicates whether to display subtotal information for the field.
                 * @returns {boolean}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Date","Date",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.subtotalVisible("Buyer", false);
                 * ```
                 */
                subtotalVisible(fieldName: string,  isVisible: boolean): boolean;
                /**
                 * @description Stop update field util resumeFieldsLayout, it must be used in pairs with resumeFieldsLayout.
                 */
                suspendLayout(): void;
                /**
                 * Get or set pivot table theme
                 * @param {string | GC.Spread.Pivot.PivotTableTheme} [theme] Indicates the pivot table internal theme name or pivot table theme instance.
                 * @returns {void | GC.Spread.Pivot.PivotTableTheme} If no parameters are passed in, get the current theme
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.theme("light3");
                 * ```
                 */
                theme(theme?: string | GC.Spread.Pivot.PivotTableTheme): void | GC.Spread.Pivot.PivotTableTheme;
                /**
                 * @description Ungroup the field by field name.
                 * @param {string} fieldName Indicates the ungroup field name.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"), { sheetCount: 3 });
                 * var sheet = spread.getActiveSheet();
                 * sheet.suspendPaint();
                 * var sourceDataArray = [["Date", "Buyer", "Type", "Amount"],
                 * ["01-Jan", "Mom", "Fuel", 74],
                 * ["15-Jan", "Mom", "Food", 235],
                 * ["17-Jan", "Dad", "Sports", 20],
                 * ["19-Jan", "David", "Books", 120],
                 * ["20-Jan", "Dad", "Food", 160],
                 * ["21-Jan", "David", "Sports", 15],
                 * ["21-Jan", "Kelly", "Books", 125]];
                 * sheet.setArray(3, 0, sourceDataArray);
                 * sheet.tables.add('Table1', 3, 0, 8, 4);
                 * sheet.setColumnWidth(6, 130);
                 * sheet.setColumnWidth(8, 100);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("PivotTable1", 'Table1', 3, 6, layout, theme, option);
                 * var groupInfo = {
                 *     originFieldName: "Buyer",
                 *     textGroup: {
                 *         "parent": ["Mom", "Dad"],
                 *         "children": ["David", "Kelly"]
                 *     }
                 * };
                 * pivotTable.group("FamilyMembers", groupInfo);
                 * pivotTable.add("FamilyMembers", "FamilyMembers", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * sheet.resumePaint();
                 *
                 * pivotTable.ungroup("FamilyMembers");
                 * ```
                 */
                ungroup(fieldName: string): void;
                /**
                 * @description update calcItem Information
                 * @param {string} sourceName The name of sourceField name
                 * @param {string} calcItemName The name of sourceField calcItem name
                 * @param {string} formula The new formula for this calcItem
                 * @param {number} priority The new priority for this calcItem
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.addCalcItem("Buyer", "formula1", "=Buyer[Mom]+Buyer[Dad]");
                 * pivotTable.updateCalcItem("Buyer", "formula1", "=Buyer[Mom]+Buyer[Kelly]", 1);
                 * ```
                 */
                updateCalcItem(sourceName: string,  calcItemName: string,  formula: string,  priority: number): void;
                /**
                 * @description Update the field area and index
                 * @param {string} name Indicates the field name.
                 * @param {GC.Spread.Pivot.PivotTableFieldType} area Indicates which area the field to be put.
                 * @param {number} [index] Indicates which index will be set.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * pivotTable.updateField("Buyer", GC.Spread.Pivot.PivotTableFieldType.columnField, 0) //The Field of name is "Buyer" move to column area and Field index is 2
                 * ```
                 */
                updateField(name: string,  area: GC.Spread.Pivot.PivotTableFieldType,  index?: number): void;
                /**
                 * @description Update the exist field Name.
                 * @param {string} oldName Indicates the old display name of field in pivot table.
                 * @param {string} newName Indicates the new display name of field in pivot table.
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.updateFieldName("Buyer", "newBuyer")
                 * ```
                 */
                updateFieldName(oldName: string,  newName: string): void;
                /**
                 * @description refresh pivotTable data source
                 * @returns {void}
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"));
                 * var sourceSheet = spread.getSheet(0)
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                  ["01-Jan","Mom","Fuel",74],
                 *                  ["15-Jan","Mom","Food",235],
                 *                  ["17-Jan","Dad","Sports",20],
                 *                  ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData);
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * var pivotTable = sheet.pivotTables.get("pivotTable_1");
                 * pivotTable.add("Buyer","Buyer",GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Type","Type",GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Amount","Sum of Amount",GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * sourceSheet.setValue(1,3,1000);
                 * pivotTable.updateSource();
                 * var newSourceData = [["Salesman","ProductType","Product","Sales"],
                 *                  ["SunLin","Drinks","Apple Juice",74],
                 *                  ["JinShiPeng","Drinks","Milk",235],
                 *                  ["ZhangShang","Dessert","Chocolate",20],
                 *                  ["SunYang","Dessert","Beef Jerky",125]];
                 * sourceSheet.setArray(10, 0, newSourceData);
                 * sourceSheet.tables.add('newSourceData', 10, 0, 5, 4);
                 * pivotTable.updateSource('newSourceData');
                 * ```
                 */
                updateSource(source?: string): void;
                /**
                 * Get or set value filter info for a field.
                 * @param {string} fieldName Indicates the target field name of pivot table.
                 * @param {GC.Spread.Pivot.IPivotConditionFilterInfo | void} [filterInfo] Indicates the value filter info when set.
                 * @returns {GC.Spread.Pivot.IPivotConditionFilterInfo} return the pivot table value information.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var options = {showRowHeader: true, showColumnHeader: true};
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme, options);
                 * pivotTable.suspendLayout();
                 * pivotTable.options.showRowHeader = true;
                 * pivotTable.options.showColumnHeader = true;
                 * pivotTable.add("Buyer", "Buyer", GC.Spread.Pivot.PivotTableFieldType.columnField);
                 * pivotTable.add("Type", "Type", GC.Spread.Pivot.PivotTableFieldType.rowField);
                 * pivotTable.add("Amount", "Sum of Amount", GC.Spread.Pivot.PivotTableFieldType.valueField, GC.Pivot.SubtotalType.sum);
                 * let condition = { conType: GC.Pivot.PivotConditionType.value, operator: GC.Pivot.PivotValueFilterOperator.between, val: [0, 100] };
                 * let filterInfo = { condition: condition, conditionByName: "Sum of Amount" };
                 * pivotTable.valueFilter("Buyer", filterInfo);
                 * pivotTable.resumeLayout();
                 * ```
                 */
                valueFilter(fieldName: string,  filterInfo?: GC.Spread.Pivot.IPivotConditionFilterInfo | null): GC.Spread.Pivot.IPivotConditionFilterInfo | void;
            }

            export class PivotTableItemSlicer{
                /**
                 * Represents a pivot slicer.
                 * @class GC.Spread.Pivot.PivotTableItemSlicer
                 */
                constructor();
                /**
                 * Gets or sets the allowMove of the slicer.
                 * @param {boolean} [value] The allowMove of the slicer. The allowMove property of a slicer specifies whether users are permitted to move the slicer, enabling or disabling the repositioning functionality.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the allowMove of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.allowMove();
                 * console.log(oldValue);
                 * slicer.allowMove(false);
                 * var newValue = slicer.allowMove();
                 * console.log(newValue);
                 * ```
                 */
                allowMove(value?: boolean): any;
                /**
                 * Gets or sets the allowResize of the slicer.
                 * @param {boolean} [value] The allowResize of the slicer. The allowResize property of a slicer specifies whether users are permitted to adjust the size of the slicer, enabling or disabling the resizing functionality.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the allowResize of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.allowResize();
                 * console.log(oldValue);
                 * slicer.allowResize(false);
                 * var newValue = slicer.allowResize();
                 * console.log(newValue);
                 * ```
                 */
                allowResize(value?: boolean): any;
                /**
                 * Gets or sets the captionName of the slicer.
                 * @param {string} [value] The captionName of the slicer. captionName property is displayed in the header of the slicer.
                 * @returns {string | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the captionName of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.captionName();
                 * console.log(oldValue);
                 * slicer.captionName('Slicer_Caption');
                 * var newValue = slicer.captionName();
                 * console.log(newValue);
                 * ```
                 */
                captionName(value?: string): any;
                /**
                 * Gets or sets the columnCount of the slicer.
                 * @param {number} [value] The columnCount of the slicer.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the columnCount of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.columnCount();
                 * console.log(oldValue);
                 * slicer.columnCount(3);
                 * var newValue = slicer.columnCount();
                 * console.log(newValue);
                 * ```
                 */
                columnCount(value?: number): any;
                /**
                 * Connect pivot table with the slicer
                 * @param {string} ptName name of pivot table
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                connectPivotTable(ptName: string): void;
                /**
                 * Gets or sets the disableResizingAndMoving of the slicer.
                 * @param {boolean} [value] The disableResizingAndMoving of the slicer. The disableResizingAndMoving property of a slicer specifies whether the user is allowed to resize or move the slicer control, restricting any changes to its size or position.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the disableResizingAndMoving of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.disableResizingAndMoving();
                 * console.log(oldValue);
                 * slicer.disableResizingAndMoving(true);
                 * var newValue = slicer.disableResizingAndMoving();
                 * console.log(newValue);
                 * ```
                 */
                disableResizingAndMoving(value?: boolean): any;
                /**
                 * Disconnect pivot table with the slicer
                 * @param {string} ptName name of pivot table
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                disconnectPivotTable(ptName: string): void;
                /**
                 * Gets or sets the dynamicMove of the slicer.
                 * @param {boolean} [value] The dynamicMove of the slicer. The dynamicMove property of a slicer specifies whether the slicer is configured to dynamically adjust its position or size based on changes in the associated data or PivotTable layout.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the dynamicMove of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.dynamicMove();
                 * console.log(oldValue);
                 * slicer.dynamicMove(false);
                 * var newValue = slicer.dynamicMove();
                 * console.log(newValue);
                 * ```
                 */
                dynamicMove(value?: boolean): any;
                /**
                 * Gets or sets the dynamicSize of the slicer.
                 * @param {boolean} [value] The dynamicSize of the slicer. The dynamicSize property of a slicer specifies whether the slicer is configured to dynamically adjust its size based on changes in the associated data or PivotTable layout.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the dynamicSize of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.dynamicSize();
                 * console.log(oldValue);
                 * slicer.dynamicSize(false);
                 * var newValue = slicer.dynamicSize();
                 * console.log(newValue);
                 * ```
                 */
                dynamicSize(value?: boolean): any;
                /**
                 * Gets or sets the endColumn of the slicer.
                 * @param {number} [value] The endColumn of the slicer. The endColumn property of a slicer specifies the ending column index or position within the worksheet where the slicer concludes or is bounded.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the endColumn of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.endColumn();
                 * console.log(oldValue);
                 * slicer.endColumn(20);
                 * var newValue = slicer.endColumn();
                 * console.log(newValue);
                 * ```
                 */
                endColumn(value?: number): any;
                /**
                 * Gets or sets the endColumnOffset of the slicer.
                 * @param {number} [value] The endColumnOffset of the slicer. The endColumnOffset property of a slicer specifies the horizontal offset or distance from the ending column index where the slicer concludes its placement within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the endColumnOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.endColumnOffset();
                 * console.log(oldValue);
                 * slicer.endColumnOffset(5);
                 * var newValue = slicer.endColumnOffset();
                 * console.log(newValue);
                 * ```
                 */
                endColumnOffset(value?: number): any;
                /**
                 * Gets or sets the endRow of the slicer.
                 * @param {number} [value] The endRow of the slicer. The endRow property of a slicer specifies the ending row index or position within the worksheet where the slicer concludes or is bounded.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the endRow of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.endRow();
                 * console.log(oldValue);
                 * slicer.endRow(20);
                 * var newValue = slicer.endRow();
                 * console.log(newValue);
                 * ```
                 */
                endRow(value?: number): any;
                /**
                 * Gets or sets the endRowOffset of the slicer.
                 * @param {number} [value] The endRowOffset of the slicer. The endRowOffset property of a slicer specifies the vertical offset or distance from the ending row index where the slicer concludes its placement within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the endRowOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.endRowOffset();
                 * console.log(oldValue);
                 * slicer.endRowOffset(5);
                 * var newValue = slicer.endRowOffset();
                 * console.log(newValue);
                 * ```
                 */
                endRowOffset(value?: number): any;
                /**
                 * get all connected PivotTables
                 * @returns {GC.Spread.Pivot.PivotTable[]} PivotTables connected with the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getAllConnectedPivotTables());
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getAllConnectedPivotTables());
                 * ```
                 */
                getAllConnectedPivotTables(): GC.Spread.Pivot.PivotTable[];
                /**
                 * get all PivotTables whether connected or not.
                 * @returns {GC.Spread.Pivot.PivotTable[]} PivotTables whose source is same as the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getAllPivotTables());
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getAllPivotTables());
                 * ```
                 */
                getAllPivotTables(): GC.Spread.Pivot.PivotTable[];
                /**
                 * get all connected PivotTables' name.
                 * @returns {string[]} name list of PivotTables connected with the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getConnectedPivotTableNameList());
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * console.log(slicer.getConnectedPivotTableNameList());
                 * ```
                 */
                getConnectedPivotTableNameList(): string[];
                /**
                 * Gets or sets a style name for the pivot table item slicer.
                 * @returns {string} returns the pivot table item slicer style name.
                 */
                getStyleName(): string | undefined;
                /**
                 * Gets or sets the height of the slicer.
                 * @param {number} [value] The height of the slicer. The height property of a slicer specifies the vertical dimension or height of the slicer, determining its size along the y-axis.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the height of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.height();
                 * console.log(oldValue);
                 * slicer.height(200);
                 * var newValue = slicer.height();
                 * console.log(newValue);
                 * ```
                 */
                height(value?: number): any;
                /**
                 * Check whether pivot table connect with the slicer
                 * @param {string} ptName name of pivot table
                 * @returns {boolean} whether pivot table is connected with slicer
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * slicer.connectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * slicer.disconnectPivotTable('pivotTable1');
                 * slicer.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                isConnectedPivotTable(ptName: string): boolean;
                /**
                 * Gets or sets the isLocked of the slicer.
                 * @param {boolean} [value] The isLocked of the slicer. The isLocked property of a slicer specifies whether the slicer is currently locked or unlocked, indicating whether users can make changes to its settings or selections when Worksheet is protected.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the isLocked of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.isLocked();
                 * console.log(oldValue);
                 * slicer.isLocked(false);
                 * var newValue = slicer.isLocked();
                 * console.log(newValue);
                 * ```
                 */
                isLocked(value?: boolean): any;
                /**
                 * Gets or sets the isSelected of the slicer.
                 * @param {boolean} [value] The isSelected of the slicer. The isSelected property of a slicer specifies whether a particular item within the slicer is currently selected or highlighted.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the isSelected of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.isSelected();
                 * console.log(oldValue);
                 * slicer.isSelected(true);
                 * var newValue = slicer.isSelected();
                 * console.log(newValue);
                 * ```
                 */
                isSelected(value?: boolean): any;
                /**
                 * Gets or sets the isVisible of the slicer.
                 * @param {boolean} [value] The isVisible of the slicer. The isVisible property of a slicer specifies whether the slicer is currently visible or hidden within the workbook.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the isVisible of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.isVisible();
                 * console.log(oldValue);
                 * slicer.isVisible(false);
                 * var newValue = slicer.isVisible();
                 * console.log(newValue);
                 * ```
                 */
                isVisible(value?: boolean): any;
                /**
                 * Gets or sets the itemHeight of the slicer.
                 * @param {number} [value] The itemHeight of the slicer. The itemHeight property of a slicer specifies the height of each item or row within the slicer control.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the itemHeight of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.itemHeight();
                 * console.log(oldValue);
                 * slicer.itemHeight(34);
                 * var newValue = slicer.itemHeight();
                 * console.log(newValue);
                 * ```
                 */
                itemHeight(value?: number): any;
                /**
                 * Gets or sets the multiSelect of the slicer.
                 * @param {boolean} [value] The multiSelect of the slicer. The multiSelect property of a slicer specifies whether multiple items can be selected simultaneously within the slicer.
                 * @returns {boolean} If no value is set, returns the multiSelect of the slicer
                 * @example
                 * ```javascript
                 * var spread = GC.Spread.Sheets.findControl('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.multiSelect();
                 * console.log(oldValue);
                 * slicer.multiSelect(true);
                 * var newValue = slicer.multiSelect();
                 * console.log(newValue);
                 * ```
                 */
                multiSelect(value?: boolean): any;
                /**
                 * Gets or sets the name of the slicer.
                 * @param {string} [value] The name of the slicer. The name property of a slicer specifies the unique identifier or label assigned to the slicer, allowing for identification and reference within the workbook or programmatic interactions.
                 * @returns {string | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the name of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.name();
                 * console.log(oldValue);
                 * slicer.name('SlicerA');
                 * var newValue = slicer.name();
                 * console.log(newValue);
                 * ```
                 */
                name(value?: string): any;
                /**
                 * Gets the nameInFormula of the slicer.
                 * @returns {string} returns the nameInFormula of the slicer.
                 */
                nameInFormula(): string;
                /**
                 * Gets or sets the position of the slicer.
                 * @param {GC.Spread.Sheets.Point} [value] The position of the slicer. The position property of a slicer specifies the placement of the slicer within the worksheet.
                 * @returns {GC.Spread.Sheets.Point | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the position of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.position();
                 * console.log(oldValue);
                 * slicer.position(new GC.Spread.Sheets.Point(100, 200));
                 * var newValue = slicer.position();
                 * console.log(newValue);
                 * ```
                 */
                position(value?: GC.Spread.Sheets.Point): any;
                /**
                 * Refresh the slicer.
                 */
                refresh(): void;
                /**
                 * Get the worksheet of the slicer.
                 * @returns {GC.Spread.Sheets.Worksheet} returns the worksheet of the slicer.
                 */
                sheet(): GC.Spread.Sheets.Worksheet;
                /**
                 * Gets or sets the showHeader of the slicer.
                 * @param {boolean} [value] The showHeader of the slicer. The showHeader property of a slicer specifies whether the header, which includes the title and filter-related controls, is displayed in the slicer.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the showHeader of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.showHeader();
                 * console.log(oldValue);
                 * slicer.showHeader(false);
                 * var newValue = slicer.showHeader();
                 * console.log(newValue);
                 * ```
                 */
                showHeader(value?: boolean): any;
                /**
                 * Gets or sets the showNoDataItems of the slicer.
                 * @param {boolean} [value] The showNoDataItems of the slicer. The showNoDataItems property of a slicer specifies whether to display items with no associated data in the connected PivotTable or data source within the slicer control.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the showNoDataItems of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.showNoDataItems();
                 * console.log(oldValue);
                 * slicer.showNoDataItems(false);
                 * var newValue = slicer.showNoDataItems();
                 * console.log(newValue);
                 * ```
                 */
                showNoDataItems(value?: boolean): any;
                /**
                 * Gets or sets the showNoDataItemsInLast of the slicer.
                 * @param {boolean} [value] The showNoDataItemsInLast of the slicer. The showNoDataItemsInLast property of a slicer specifies whether to display items with no associated data at the end of the slicer control's list when presenting the items.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the showNoDataItemsInLast of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.showNoDataItemsInLast();
                 * console.log(oldValue);
                 * slicer.showNoDataItemsInLast(false);
                 * var newValue = slicer.showNoDataItemsInLast();
                 * console.log(newValue);
                 * ```
                 */
                showNoDataItemsInLast(value?: boolean): any;
                /**
                 * Gets or sets the sortState of the slicer.
                 * @param {GC.Spread.Sheets.SortState} [value] The sortState of the slicer. The sortState property of a slicer specifies the current sorting state applied to the items within the slicer, indicating whether they are sorted in ascending, descending order, or not sorted.
                 * @returns {GC.Spread.Sheets.SortState | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the sortState of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.sortState();
                 * console.log(oldValue);
                 * slicer.sortState(GC.Spread.Sheets.SortState.descending);
                 * var newValue = slicer.sortState();
                 * console.log(newValue);
                 * ```
                 */
                sortState(value?: GC.Spread.Sheets.SortState): any;
                /**
                 * Get the sourceName of the slicer. The sourceName property of a slicer specifies the name of the source data field associated with the slicer.
                 * @returns {string} returns the sourceName of the slicer.
                 */
                sourceName(): string;
                /**
                 * Gets or sets the startColumn of the slicer.
                 * @param {number} [value] The startColumn of the slicer. The startColumn property of a slicer specifies the starting column index or position within the worksheet where the slicer is anchored
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the startColumn of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.startColumn();
                 * console.log(oldValue);
                 * slicer.startColumn(10);
                 * var newValue = slicer.startColumn();
                 * console.log(newValue);
                 * ```
                 */
                startColumn(value?: number): any;
                /**
                 * Gets or sets the startColumnOffset of the slicer.
                 * @param {number} [value] The startColumnOffset of the slicer. The startColumnOffset property of a slicer specifies the horizontal offset or distance from the starting column index where the slicer begins its placement within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the startColumnOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.startColumnOffset();
                 * console.log(oldValue);
                 * slicer.startColumnOffset(15);
                 * var newValue = slicer.startColumnOffset();
                 * console.log(newValue);
                 * ```
                 */
                startColumnOffset(value?: number): any;
                /**
                 * Gets or sets the startRow of the slicer.
                 * @param {number} [value] The startRow of the slicer. The startRow property of a slicer specifies the starting row index or position within the worksheet where the slicer is anchored.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the startRow of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.startRow();
                 * console.log(oldValue);
                 * slicer.startRow(10);
                 * var newValue = slicer.startRow();
                 * console.log(newValue);
                 * ```
                 */
                startRow(value?: number): any;
                /**
                 * Gets or sets the startRowOffset of the slicer.
                 * @param {number} [value] The startRowOffset of the slicer. The startRowOffset property of a slicer specifies the vertical offset or distance from the starting row index where the slicer begins its placement within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the startRowOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.startRowOffset();
                 * console.log(oldValue);
                 * slicer.startRowOffset(15);
                 * var newValue = slicer.startRowOffset();
                 * console.log(newValue);
                 * ```
                 */
                startRowOffset(value?: number): any;
                /**
                 * Gets or sets the style of the slicer.
                 * @param {string | GC.Spread.Sheets.Slicers.SlicerStyle} [value] The style or style name of the slicer. The style property of a slicer specifies the visual appearance and formatting style applied to the slicer, defining its overall look and presentation.
                 * @returns {GC.Spread.Sheets.Slicers.SlicerStyle | GC.Spread.Sheets.Slicers.IItemSlicer} If no style is set, returns the style of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 *
                 * var style = new GC.Spread.Sheets.Slicers.SlicerStyle();
                 * style.wholeSlicerStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo('red', 'blue', '16pt Calibri'));
                 * style.headerStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo(undefined, 'green'));
                 * style.selectedItemWithDataStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo(undefined, undefined, undefined, new GC.Spread.Sheets.LineBorder('pink', GC.Spread.Sheets.LineStyle.double)));
                 *
                 * var oldValue = slicer.style();
                 * console.log(oldValue);
                 * slicer.style(style);
                 * var newValue = slicer.style();
                 * console.log(newValue);
                 * ```
                 */
                style(value?: string | GC.Spread.Sheets.Slicers.SlicerStyle): any;
                /**
                 * Gets or sets the visuallyNoDataItems of the slicer.
                 * @param {boolean} [value] The visuallyNoDataItems of the slicer. The visuallyNoDataItems property of a slicer specifies the visual representation or handling of items with no associated data within the slicer control.
                 * @returns {boolean | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the visuallyNoDataItems of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.visuallyNoDataItems();
                 * console.log(oldValue);
                 * slicer.visuallyNoDataItems(false);
                 * var newValue = slicer.visuallyNoDataItems();
                 * console.log(newValue);
                 * ```
                 */
                visuallyNoDataItems(value?: boolean): any;
                /**
                 * Gets or sets the width of the slicer.
                 * @param {number} [value] The width of the slicer. The width property of a slicer specifies the horizontal dimension or width of the slicer, determining its size along the x-axis.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the width of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.width();
                 * console.log(oldValue);
                 * slicer.width(200);
                 * var newValue = slicer.width();
                 * console.log(newValue);
                 * ```
                 */
                width(value?: number): any;
                /**
                 * Gets or sets the x of the slicer.
                 * @param {number} [value] The x of the slicer. The x property of a slicer specifies the horizontal position or coordinate of the upper-left corner of the slicer within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the x of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.x();
                 * console.log(oldValue);
                 * slicer.x(100);
                 * var newValue = slicer.x();
                 * console.log(newValue);
                 * ```
                 */
                x(value?: number): any;
                /**
                 * Gets or sets the y of the slicer.
                 * @param {number} [value] The y of the slicer. The y property of a slicer specifies the vertical position or coordinate of the upper-left corner of the slicer within the worksheet.
                 * @returns {number | GC.Spread.Sheets.Slicers.IItemSlicer} If no value is set, returns the y of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var slicer = activeSheet.slicers.add("slicer", "pivotTable1", "Name", GC.Spread.Sheets.Slicers.SlicerStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTable);
                 * var oldValue = slicer.y();
                 * console.log(oldValue);
                 * slicer.y(100);
                 * var newValue = slicer.y();
                 * console.log(newValue);
                 * ```
                 */
                y(value?: number): any;
            }

            export class PivotTableStyle{
                /**
                 * Represents a PivotTable Style.
                 * @class
                 * @param {string} [backColor] Indicates the backColor of pivot table style.
                 * @param {string} [foreColor] Indicates the foreColor of pivot table style.
                 * @param {string} [font] Indicates the font of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderLeft] Indicates the borderLeft of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderTop] Indicates the borderTop of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderRight] Indicates the borderRight of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderBottom] Indicates the borderBottom of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderHorizontal] Indicates the borderHorizontal of pivot table style.
                 * @param {GC.Spread.Sheets.LineBorder} [borderVertical] Indicates the borderVertical of pivot table style.
                 * @param {GC.Spread.Sheets.TextDecorationType} [textDecoration] Indicates the textDecoration of pivot table style.
                 * @param {string} [fontStyle] The font style.
                 * @param {string} [fontWeight] The font weight.
                 * @param {string} [fontSize] The font size.
                 * @param {string} [fontFamily] The font family.
                 */
                constructor(backColor?: string,  foreColor?: string,  font?: string,  borderLeft?: GC.Spread.Sheets.LineBorder,  borderTop?: GC.Spread.Sheets.LineBorder,  borderRight?: GC.Spread.Sheets.LineBorder,  borderBottom?: GC.Spread.Sheets.LineBorder,  borderHorizontal?: GC.Spread.Sheets.LineBorder,  borderVertical?: GC.Spread.Sheets.LineBorder,  textDecoration?: GC.Spread.Sheets.TextDecorationType,  fontStyle?: string,  fontWeight?: string,  fontSize?: string,  fontFamily?: string);
                /**
                 * Indicates the background color.
                 * @type {string}
                 */
                backColor: string;
                /**
                 * Indicates the bottom border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderBottom: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the horizontal border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderHorizontal: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the left border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderLeft: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the right border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderRight: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the top border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderTop: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the vertical border.
                 * @type {GC.Spread.Sheets.LineBorder}
                 */
                borderVertical: GC.Spread.Sheets.LineBorder;
                /**
                 * Indicates the font.
                 * @type {string}
                 */
                font: string;
                /**
                 * Indicates the font family.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example sets the fontFamily property.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var wholePivotTableStyle = new GC.Spread.Pivot.PivotTableStyle();
                 * wholePivotTableStyle.fontFamily = "Arial Black";
                 * theme.wholeTableStyle(wholePivotTableStyle);
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * ```
                 */
                fontFamily: string;
                /**
                 * Indicates the font size.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example sets the fontSize property.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var wholePivotTableStyle = new GC.Spread.Pivot.PivotTableStyle();
                 * wholePivotTableStyle.fontSize = "16px";
                 * theme.wholeTableStyle(wholePivotTableStyle);
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * ```
                 */
                fontSize: string;
                /**
                 * Indicates the font style.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example sets the fontStyle property.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var wholePivotTableStyle = new GC.Spread.Pivot.PivotTableStyle();
                 * wholePivotTableStyle.fontStyle = "italic";
                 * theme.wholeTableStyle(wholePivotTableStyle);
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * ```
                 */
                fontStyle: string;
                /**
                 * Indicates the font weight.
                 * @type {string}
                 * @example
                 * ```javascript
                 * //This example sets the fontWeight property.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var sourceSheet = spread.getSheet(0);
                 * var sheet = spread.getSheet(1);
                 * var sourceData = [["Date","Buyer","Type","Amount"],
                 *                   ["01-Jan","Mom","Fuel",74],
                 *                   ["15-Jan","Mom","Food",235],
                 *                   ["17-Jan","Dad","Sports",20],
                 *                   ["21-Jan","Kelly","Books",125]];
                 * sourceSheet.setArray(0, 0, sourceData );
                 * sourceSheet.tables.add('sourceData', 0, 0, 5, 4);
                 * var layout = GC.Spread.Pivot.PivotTableLayoutType.compact;
                 * var theme = GC.Spread.Pivot.PivotTableThemes.medium2;
                 * var wholePivotTableStyle = new GC.Spread.Pivot.PivotTableStyle();
                 * wholePivotTableStyle.fontWeight = "bold";
                 * theme.wholeTableStyle(wholePivotTableStyle);
                 * var pivotTable = sheet.pivotTables.add("pivotTable_1", 'sourceData', 1, 1, layout, theme);
                 * ```
                 */
                fontWeight: string;
                /**
                 * Indicates the fore color.
                 * @type {string}
                 */
                foreColor: string;
                /**
                 * Indicates the text decoration.
                 * @type {GC.Spread.Sheets.TextDecorationType}
                 */
                textDecoration: GC.Spread.Sheets.TextDecorationType;
            }

            export class PivotTableTheme{
                /**
                 * Represents a PivotTableTheme class.
                 * @class
                 */
                constructor();
                /**
                 * @description get or set the blankRowStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the blank row area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                blankRowStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the columnSubheading1Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading column 1 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                columnSubheading1Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the columnSubheading2Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading column 2 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                columnSubheading2Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the columnSubheading3Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading column 3 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                columnSubheading3Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the firstColumnStripeStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The first alternating column style.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                firstColumnStripeStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the firstColumnStripSize of pivot table theme
                 * @param {number} [value] The size of the first alternating column.
                 * @returns {number | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the size of the size of the first alternating column; otherwise, returns the pivot table theme.
                 */
                firstColumnStripSize(value?: number): any;
                /**
                 * @description get or set the firstColumnStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the first column.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                firstColumnStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the firstHeaderCellStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the first header cell.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                firstHeaderCellStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the firstRowStripeStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The first alternating row style.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                firstRowStripeStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the firstRowStripSize of pivot table theme
                 * @param {number} [value] The size of the first alternating row.
                 * @returns {number | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the size of the first alternating row; otherwise, returns the pivot table theme.
                 */
                firstRowStripSize(value?: number): any;
                /**
                 * Loads the object state from the specified JSON string.
                 * @param {Object} data The pivot table theme data from deserialization.
                 * @example
                 * ```javascript
                 * //This example uses the fromJSON method.
                 * const light1 = GC.Spread.Pivot.PivotTableThemes.light1;
                 * //export
                 * const jsonStr = JSON.stringify(light1.toJSON());
                 * //import
                 * const newTheme = new GC.Spread.Pivot.PivotTableTheme();
                 * newTheme.fromJSON(JSON.parse(jsonStr));
                 * newTheme.name('custom1');
                 * alert(jsonStr);
                 * ```
                 */
                fromJSON(data: Object): void;
                /**
                 * @description get or set the grandTotalColumnStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the grandTotal column area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                grandTotalColumnStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the grandTotalRowStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the grandTotal row area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                grandTotalRowStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the headerRowStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the header area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                headerRowStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the name of pivot table theme
                 * @param {string} [value] name of pivot table theme
                 * @returns {string | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the name of the style; otherwise, returns the table theme.
                 */
                name(value?: string): any;
                /**
                 * @description get or set the reportFilterLabelsStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the report filter labels area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                reportFilterLabelsStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the reportFilterValuesStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the report filter values area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                reportFilterValuesStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the rowSubheading1Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading row 1 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                rowSubheading1Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the rowSubheading2Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading row 2 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                rowSubheading2Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the rowSubheading3Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subheading row 3 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                rowSubheading3Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the secondColumnStripeStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The second alternating column style.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                secondColumnStripeStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the secondColumnStripSize of pivot table theme
                 * @param {number} [value] The size of the second alternating column.
                 * @returns {number | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the size of the second alternating column; otherwise, returns the pivot table theme.
                 */
                secondColumnStripSize(value?: number): any;
                /**
                 * @description get or set the secondRowStripeStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The second alternating row style.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                secondRowStripeStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the secondRowStripSize of pivot table theme
                 * @param {number} [value] The size of the second alternating row.
                 * @returns {number | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the size of the second alternating row; otherwise, returns the pivot table theme.
                 */
                secondRowStripSize(value?: number): any;
                /**
                 * @description get or set the subtotalColumn1Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal column 1 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalColumn1Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the subtotalColumn2Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal column 2 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalColumn2Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the subtotalColumn3Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal column 3 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalColumn3Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the subtotalRow1Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal row 1 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalRow1Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the subtotalRow2Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal row 2 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalRow2Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * @description get or set the subtotalRow3Style of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of the subtotal row 3 area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                subtotalRow3Style(value?: GC.Spread.Pivot.PivotTableStyle): any;
                /**
                 * Saves the object state to a JSON string.
                 * @returns {Object} The pivot table theme data.
                 * @example
                 * ```javascript
                 * //This example uses the toJSON method.
                 * const light1 = GC.Spread.Pivot.PivotTableThemes.light1;
                 * //export
                 * const jsonStr = JSON.stringify(light1.toJSON());
                 * //import
                 * const newTheme = new GC.Spread.Pivot.PivotTableTheme();
                 * newTheme.fromJSON(JSON.parse(jsonStr));
                 * newTheme.name('custom1');
                 * alert(jsonStr);
                 * ```
                 */
                toJSON(): Object;
                /**
                 * @description get or set the wholeTableStyle of pivot table theme
                 * @param {GC.Spread.Pivot.PivotTableStyle} [value] The default style of he data area.
                 * @returns {GC.Spread.Pivot.PivotTableStyle | GC.Spread.Pivot.PivotTableTheme} If no value is set, returns the default style of the data area; otherwise, returns the pivot table theme.
                 */
                wholeTableStyle(value?: GC.Spread.Pivot.PivotTableStyle): any;
            }

            export class PivotTableThemes{
                /**
                 * Represents a built-in Pivot table theme collection.
                 * @class
                 */
                constructor();
                /**
                 * Gets the dark1 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark1: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark10 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark10: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark11 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark11: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark12 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark12: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark13 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark13: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark14 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark14: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark15 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark15: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark16 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark16: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark17 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark17: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark18 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark18: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark19 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark19: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark2 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark2: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark20 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark20: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark21 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark21: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark22 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark22: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark23 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark23: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark24 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark24: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark25 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark25: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark26 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark26: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark27 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark27: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark28 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark28: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark3 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark3: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark4 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark4: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark5 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark5: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark6 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark6: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark7 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark7: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark8 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark8: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the dark9 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static dark9: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light0 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light0: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light1 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light1: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light10 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light10: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light11 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light11: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light12 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light12: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light13 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light13: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light14 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light14: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light15 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light15: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light16 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light16: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light17 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light17: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light18 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light18: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light19 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light19: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light2 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light2: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light20 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light20: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light21 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light21: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light22 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light22: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light23 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light23: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light24 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light24: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light25 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light25: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light26 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light26: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light27 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light27: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light28 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light28: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light3 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light3: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light4 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light4: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light5 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light5: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light6 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light6: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light7 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light7: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light8 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light8: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the light9 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static light9: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium1 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium1: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium10 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium10: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium11 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium11: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium12 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium12: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium13 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium13: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium14 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium14: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium15 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium15: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium16 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium16: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium17 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium17: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium18 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium18: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium19 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium19: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium2 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium2: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium20 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium20: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium21 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium21: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium22 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium22: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium23 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium23: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium24 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium24: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium25 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium25: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium26 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium26: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium27 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium27: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium28 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium28: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium3 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium3: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium4 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium4: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium5 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium5: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium6 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium6: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium7 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium7: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium8 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium8: GC.Spread.Pivot.PivotTableTheme;
                /**
                 * Gets the medium9 style.
                 * @returns {GC.Spread.Pivot.PivotTableTheme}
                 */
                static medium9: GC.Spread.Pivot.PivotTableTheme;
            }

            export class PivotTableTimelineSlicer{
                /**
                 * Represents a pivot timeline slicer.
                 * @class GC.Spread.Pivot.PivotTableTimelineSlicer
                 */
                constructor();
                /**
                 * Gets or sets the allowMove of the slicer.
                 * @param {boolean} [value] The allowMove of the slicer. The allowMove property of a timeline slicer specifies whether users are permitted to move the timeline, enabling or disabling the repositioning functionality.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the allowMove of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.allowMove();
                 * console.log(oldValue);
                 * slicer.allowMove(false);
                 * var newValue = slicer.allowMove();
                 * console.log(newValue);
                 * ```
                 */
                allowMove(value?: boolean): any;
                /**
                 * Gets or sets the allowResize of the slicer.
                 * @param {boolean} [value] The allowResize of the slicer. The allowResize property of a timeline slicer specifies whether users are permitted to adjust the size of the timeline, enabling or disabling the resizing functionality.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the allowResize of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.allowResize();
                 * console.log(oldValue);
                 * slicer.allowResize(false);
                 * var newValue = slicer.allowResize();
                 * console.log(newValue);
                 * ```
                 */
                allowResize(value?: boolean): any;
                /**
                 * Gets or sets the captionName of the slicer.
                 * @param {string} [value] The captionName of the slicer. captionName property is displayed in the header of the timeline slicer.
                 * @returns {string | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the captionName of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.captionName();
                 * console.log(oldValue);
                 * slicer.captionName('timeline_caption');
                 * var newValue = slicer.captionName();
                 * console.log(newValue);
                 * ```
                 */
                captionName(value?: string): any;
                /**
                 * Connect pivot table with the slicer
                 * @param {string} ptName name of pivot table
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                connectPivotTable(ptName: string): void;
                /**
                 * Gets or sets the disableResizingAndMoving of the slicer.
                 * @param {boolean} [value] The disableResizingAndMoving of the slicer. The disableResizingAndMoving property of a timeline slicer specifies whether the user is allowed to resize or move the timeline slicer, restricting any changes to its size or position.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the disableResizingAndMoving of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.disableResizingAndMoving();
                 * console.log(oldValue);
                 * slicer.disableResizingAndMoving(false);
                 * var newValue = slicer.disableResizingAndMoving();
                 * console.log(newValue);
                 * ```
                 */
                disableResizingAndMoving(value?: boolean): any;
                /**
                 * Disconnect pivot table with the slicer
                 * @param {string} ptName name of pivot table
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                disconnectPivotTable(ptName: string): void;
                /**
                 * Gets or sets the dynamicMove of the slicer.
                 * @param {boolean} [value] The dynamicMove of the slicer. The dynamicMove property of a timeline slicer specifies whether the timeline slicer is configured to dynamically adjust its position or size based on changes in the associated data or PivotTable layout.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the dynamicMove of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.dynamicMove();
                 * console.log(oldValue);
                 * slicer.dynamicMove(false);
                 * var newValue = slicer.dynamicMove();
                 * console.log(newValue);
                 * ```
                 */
                dynamicMove(value?: boolean): any;
                /**
                 * Gets or sets the dynamicSize of the slicer.
                 * @param {boolean} [value] The dynamicSize of the slicer. The dynamicSize property of a timeline slicer specifies whether the timeline is configured to dynamically adjust its size based on changes in the associated data or PivotTable layout.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the dynamicSize of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.dynamicSize();
                 * console.log(oldValue);
                 * slicer.dynamicSize(false);
                 * var newValue = slicer.dynamicSize();
                 * console.log(newValue);
                 * ```
                 */
                dynamicSize(value?: boolean): any;
                /**
                 * Gets or sets the endColumn of the slicer.
                 * @param {number} [value] The endColumn of the slicer. The endColumn property of a timeline slicer specifies the ending column index or position within the worksheet where the timeline concludes or is bounded.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the endColumn of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.endColumn();
                 * console.log(oldValue);
                 * slicer.endColumn(9);
                 * var newValue = slicer.endColumn();
                 * console.log(newValue);
                 * ```
                 */
                endColumn(value?: number): any;
                /**
                 * Gets or sets the endColumnOffset of the slicer.
                 * @param {number} [value] The endColumnOffset of the slicer. The endColumnOffset property of a timeline slicer specifies the horizontal offset or distance from the ending column index where the timeline concludes its placement within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the endColumnOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.endColumnOffset();
                 * console.log(oldValue);
                 * slicer.endColumnOffset(10);
                 * var newValue = slicer.endColumnOffset();
                 * console.log(newValue);
                 * ```
                 */
                endColumnOffset(value?: number): any;
                /**
                 * Gets or sets the endRow of the slicer.
                 * @param {number} [value] The endRow of the slicer. The endRow property of a timeline slicer specifies the ending row index or position within the worksheet where the timeline concludes or is bounded.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the endRow of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.endRow();
                 * console.log(oldValue);
                 * slicer.endRow(7);
                 * var newValue = slicer.endRow();
                 * console.log(newValue);
                 * ```
                 */
                endRow(value?: number): any;
                /**
                 * Gets or sets the endRowOffset of the slicer.
                 * @param {number} [value] The endRowOffset of the slicer. The endRowOffset property of a timeline slicer specifies the vertical offset or distance from the ending row index where the timeline concludes its placement within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the endRowOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.endRowOffset();
                 * console.log(oldValue);
                 * slicer.endRowOffset(5);
                 * var newValue = slicer.endRowOffset();
                 * console.log(newValue);
                 * ```
                 */
                endRowOffset(value?: number): any;
                /**
                 * get all connected PivotTables
                 * @returns {GC.Spread.Pivot.PivotTable[]} PivotTables connected with the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * ```
                 */
                getAllConnectedPivotTables(): GC.Spread.Pivot.PivotTable[];
                /**
                 * get all PivotTables whether connected or not.
                 * @returns {GC.Spread.Pivot.PivotTable[]} PivotTables whose source is same as the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * ```
                 */
                getAllPivotTables(): GC.Spread.Pivot.PivotTable[];
                /**
                 * get all connected PivotTables' name.
                 * @returns {string[]} name list of PivotTables connected with the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * console.log(timeline.getAllPivotTables());
                 * ```
                 */
                getConnectedPivotTableNameList(): string[];
                /**
                 * Gets or sets a style name for the pivot table timeline slicer.
                 * @returns {string} returns the pivot table timeline slicer style name.
                 */
                getStyleName(): string | undefined;
                /**
                 * Gets or sets the height of the slicer.
                 * @param {number} [value] The height of the slicer. The height property of a timeline slicer specifies the vertical dimension or height of the timeline, determining its size along the y-axis.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the height of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.height();
                 * console.log(oldValue);
                 * slicer.height(120);
                 * var newValue = slicer.height();
                 * console.log(newValue);
                 * ```
                 */
                height(value?: number): any;
                /**
                 * Check whether pivot table connect with the slicer
                 * @param {string} ptName name of pivot table
                 * @returns {boolean} whether pivot table is connected with timeline slicer
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * timeline.connectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * timeline.disconnectPivotTable('pivotTable1');
                 * timeline.isConnectedPivotTable('pivotTable1');
                 * ```
                 */
                isConnectedPivotTable(ptName: string): boolean;
                /**
                 * Gets or sets the isLocked of the slicer.
                 * @param {boolean} [value] The isLocked of the slicer. The isLocked property of a timeline slicer specifies whether the timeline is currently locked or unlocked, indicating whether users can make changes to its settings or selections when Worksheet is protected.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the isLocked of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.isLocked();
                 * console.log(oldValue);
                 * slicer.isLocked(false);
                 * var newValue = slicer.isLocked();
                 * console.log(newValue);
                 * ```
                 */
                isLocked(value?: boolean): any;
                /**
                 * Gets or sets the isSelected of the slicer.
                 * @param {boolean} [value] The isSelected of the slicer. The isSelected property of a timeline slicer specifies whether a particular time period within the timeline slicer is currently selected or highlighted.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the isSelected of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.isSelected();
                 * console.log(oldValue);
                 * slicer.isSelected(false);
                 * var newValue = slicer.isSelected();
                 * console.log(newValue);
                 * ```
                 */
                isSelected(value?: boolean): any;
                /**
                 * Gets or sets the isVisible of the slicer.
                 * @param {boolean} [value] The isVisible of the slicer. The isVisible property of a timeline slicer specifies whether the timeline slicer is currently visible or hidden within the workbook.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the isVisible of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.isVisible();
                 * console.log(oldValue);
                 * slicer.isVisible(false);
                 * var newValue = slicer.isVisible();
                 * console.log(newValue);
                 * ```
                 */
                isVisible(value?: boolean): any;
                /**
                 * Gets or sets the level of the slicer.
                 * @param {GC.Spread.Sheets.Slicers.TimelineLevel} [value] The level of the slicer. The level property of a timeline slicer specifies the temporal granularity or time unit (e.g., year, quarter, month) at which the timeline is currently operating, determining the level of detail for date-based filtering and selection.
                 * @returns {GC.Spread.Sheets.Slicers.TimelineLevel | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the level of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.level();
                 * console.log(oldValue);
                 * slicer.level(GC.Spread.Sheets.Slicers.TimelineLevel.years);
                 * var newValue = slicer.level();
                 * console.log(newValue);
                 * ```
                 */
                level(value?: GC.Spread.Sheets.Slicers.TimelineLevel): any;
                /**
                 * Gets or sets the name of the slicer.
                 * @param {string} [value] The name of the slicer. The name property of a timeline slicer specifies the unique identifier or label assigned to the timeline, allowing for identification and reference within the workbook or programmatic interactions.
                 * @returns {string | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the name of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.name();
                 * console.log(oldValue);
                 * slicer.name('timeline2');
                 * var newValue = slicer.name();
                 * console.log(newValue);
                 * ```
                 */
                name(value?: string): any;
                /**
                 * Gets the nameInFormula of the slicer.
                 * @returns {string} returns the nameInFormula of the slicer.
                 */
                nameInFormula(): string;
                /**
                 * Gets or sets the position of the slicer.
                 * @param {GC.Spread.Sheets.Point} [value] The position of the slicer. The position property of a timeline slicer specifies the placement of the timeline within the worksheet.
                 * @returns {GC.Spread.Sheets.Point | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the position of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.position();
                 * console.log(oldValue);
                 * slicer.position(new GC.Spread.Sheets.Point(10, 20));
                 * var newValue = slicer.position();
                 * console.log(newValue);
                 * ```
                 */
                position(value?: GC.Spread.Sheets.Point): any;
                /**
                 * Refresh the slicer.
                 */
                refresh(): void;
                /**
                 * Gets or sets the scrollPosition of the slicer.
                 * @param {Date} [value] The scrollPosition of the slicer. The scrollPosition property of a timeline slicer specifies the current scroll position or offset within the timeline, indicating the point where the visible time range begins in relation to the overall timeline data.
                 * @returns {Date | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the scrollPosition of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.showHeader();
                 * console.log(oldValue);
                 * slicer.showHeader(new Date('1973/7/3'));
                 * var newValue = slicer.showHeader();
                 * console.log(newValue);
                 * ```
                 */
                scrollPosition(value?: Date): any;
                /**
                 * Get the worksheet of the slicer.
                 * @returns {GC.Spread.Sheets.Worksheet} returns the worksheet of the slicer.
                 */
                sheet(): GC.Spread.Sheets.Worksheet;
                /**
                 * Gets or sets the showHeader of the slicer.
                 * @param {boolean} [value] The showHeader of the slicer. The showHeader property of a timeline slicer specifies whether the header, which includes the title and filter-related controls, is displayed in the timeline slicer.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the showHeader of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.showHeader();
                 * console.log(oldValue);
                 * slicer.showHeader(false);
                 * var newValue = slicer.showHeader();
                 * console.log(newValue);
                 * ```
                 */
                showHeader(value?: boolean): any;
                /**
                 * Gets or sets the showHorizontalScrollbar of the slicer.
                 * @param {boolean} [value] The showHorizontalScrollbar of the slicer. The showHorizontalScrollbar property of a timeline slicer specifies whether a horizontal scrollbar is displayed, allowing users to navigate through the content.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the showHorizontalScrollbar of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.showHorizontalScrollbar();
                 * console.log(oldValue);
                 * slicer.showHorizontalScrollbar(false);
                 * var newValue = slicer.showHorizontalScrollbar();
                 * console.log(newValue);
                 * ```
                 */
                showHorizontalScrollbar(value?: boolean): any;
                /**
                 * Gets or sets the showSelectionLabel of the slicer.
                 * @param {boolean} [value] The showSelectionLabel of the slicer. The showSelectionLabel property of a timeline slicer specifies whether a label indicating the selected time range is displayed, providing users with information about the currently chosen period on the timeline.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the showSelectionLabel of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.showSelectionLabel();
                 * console.log(oldValue);
                 * slicer.showSelectionLabel(false);
                 * var newValue = slicer.showSelectionLabel();
                 * console.log(newValue);
                 * ```
                 */
                showSelectionLabel(value?: boolean): any;
                /**
                 * Gets or sets the showTimeLevel of the slicer.
                 * @param {boolean} [value] The showTimeLevel of the slicer. The showTimeLevel property of a timeline slicer specifies whether different time levels (e.g., years, quarters, months) are displayed as options for users to select and filter data on the timeline.
                 * @returns {boolean | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the showTimeLevel of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.showTimeLevel();
                 * console.log(oldValue);
                 * slicer.showTimeLevel(false);
                 * var newValue = slicer.showTimeLevel();
                 * console.log(newValue);
                 * ```
                 */
                showTimeLevel(value?: boolean): any;
                /**
                 * Get the sourceName of the slicer.
                 * @returns {string} returns the sourceName of the slicer. The sourceName property of a timeline slicer specifies the name of the source data field associated with the timeline.
                 */
                sourceName(): string;
                /**
                 * Gets or sets the startColumn of the slicer.
                 * @param {number} [value] The startColumn of the slicer. The startColumn property of a timeline slicer specifies the starting column index or position within the worksheet where the timeline is anchored.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the startColumn of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.startColumn();
                 * console.log(oldValue);
                 * slicer.startColumn(5);
                 * var newValue = slicer.startColumn();
                 * console.log(newValue);
                 * ```
                 */
                startColumn(value?: number): any;
                /**
                 * Gets or sets the startColumnOffset of the slicer.
                 * @param {number} [value] The startColumnOffset of the slicer. The startColumnOffset property of a timeline slicer specifies the horizontal offset or distance from the starting column index where the timeline begins its placement within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the startColumnOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.startColumnOffset();
                 * console.log(oldValue);
                 * slicer.startColumnOffset(15);
                 * var newValue = slicer.startColumnOffset();
                 * console.log(newValue);
                 * ```
                 */
                startColumnOffset(value?: number): any;
                /**
                 * Gets or sets the startRow of the slicer.
                 * @param {number} [value] The startRow of the slicer. The startRow property of a timeline slicer specifies the starting row index or position within the worksheet where the timeline is anchored.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the startRow of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.startRow();
                 * console.log(oldValue);
                 * slicer.startRow(3);
                 * var newValue = slicer.startRow();
                 * console.log(newValue);
                 * ```
                 */
                startRow(value?: number): any;
                /**
                 * Gets or sets the startRowOffset of the slicer.
                 * @param {number} [value] The startRowOffset of the slicer. The startRowOffset property of a timeline slicer specifies the vertical offset or distance from the starting row index where the timeline begins its placement within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the startRowOffset of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.startRowOffset();
                 * console.log(oldValue);
                 * slicer.startRowOffset(10);
                 * var newValue = slicer.startRowOffset();
                 * console.log(newValue);
                 * ```
                 */
                startRowOffset(value?: number): any;
                /**
                 * Gets or sets the style of the slicer.
                 * @param {string | GC.Spread.Sheets.Slicers.TimelineStyle} [value] The style or style name of the slicer. The style property of a timeline slicer specifies the visual appearance and formatting style applied to the timeline, defining its overall look and presentation.
                 * @returns {GC.Spread.Sheets.Slicers.TimelineStyle | GC.Spread.Pivot.PivotTableTimelineSlicer} If no style is set, returns the style of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: "1968/6/8" },
                 *     { Name: "Betty", City: "NewYork", Birthday: "1972/7/3" },
                 *     { Name: "Alice", City: "Washington", Birthday: "2012/2/15" },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 *
                 * var style = new GC.Spread.Sheets.Slicers.TimelineStyle()
                 * style.wholeSlicerStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo('red', 'blue', '12pt Calibri'));
                 * style.headerStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo(undefined, 'green', '14pt Calibri'));
                 * style.timeLevelStyle(new GC.Spread.Sheets.Slicers.SlicerStyleInfo('yellow'));
                 *
                 * var oldValue = slicer.style();
                 * console.log(oldValue);
                 * timeline.style(style);
                 * var newValue = slicer.style();
                 * console.log(newValue);
                 * ```
                 */
                style(value?: string | GC.Spread.Sheets.Slicers.TimelineStyle): GC.Spread.Sheets.Slicers.TimelineStyle | GC.Spread.Pivot.PivotTableTimelineSlicer;
                /**
                 * Gets or sets the width of the slicer.
                 * @param {number} [value] The width of the slicer. The width property of a timeline slicer specifies the horizontal dimension or width of the timeline, determining its size along the x-axis.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the width of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.width();
                 * console.log(oldValue);
                 * slicer.width(150);
                 * var newValue = slicer.width();
                 * console.log(newValue);
                 * ```
                 */
                width(value?: number): any;
                /**
                 * Gets or sets the x of the slicer.
                 * @param {number} [value] The x of the slicer. The x property of a timeline slicer specifies the horizontal position or coordinate of the upper-left corner of the timeline within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the x of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.x();
                 * console.log(oldValue);
                 * slicer.x(30);
                 * var newValue = slicer.x();
                 * console.log(newValue);
                 * ```
                 */
                x(value?: number): any;
                /**
                 * Gets or sets the y of the slicer.
                 * @param {number} [value] The y of the slicer. The y property of a timeline slicer specifies the vertical position or coordinate of the upper-left corner of the timeline within the worksheet.
                 * @returns {number | GC.Spread.Pivot.PivotTableTimelineSlicer} If no value is set, returns the y of the slicer; otherwise, returns the slicer.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook('ss');
                 * var activeSheet = spread.getActiveSheet();
                 * var dataSource = [
                 *     { Name: "Bob", City: "NewYork", Birthday: new Date("1968/6/8") },
                 *     { Name: "Betty", City: "NewYork", Birthday: new Date("1972/7/3") },
                 *     { Name: "Alice", City: "Washington", Birthday: new Date("2012/2/15") },
                 * ];
                 * var table = activeSheet.tables.addFromDataSource("table1", 1, 1, dataSource);
                 * var pivotTable = activeSheet.pivotTables.add("pivotTable1", "table1", 6, 1);
                 * var timeline = activeSheet.slicers.add("timeline", "pivotTable1", "Birthday", GC.Spread.Sheets.Slicers.TimelineStyles.dark1(), GC.Spread.Sheets.Slicers.SlicerType.pivotTimeline);
                 * var oldValue = slicer.y();
                 * console.log(oldValue);
                 * slicer.y(50);
                 * var newValue = slicer.y();
                 * console.log(newValue);
                 * ```
                 */
                y(value?: number): any;
            }
            module PivotTable{

                export class PivotTableViewManager{
                    /**
                     * Represents a PivotTableViewManager.
                     * @class
                     */
                    constructor(applyCallback: Function,  saveCallback: Function);
                    /**
                     * @description Add a view to pivot table views.
                     * @param {IPivotTableView} view Indicates the view to add.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.add({
                     *      name: 'config1',
                     *      config: pivotTable.serialize()
                     * });
                     * viewsManager.get('config1');
                     * ```
                     */
                    add(view: IPivotTableView): boolean;
                    /**
                     * @description get all views from pivot table views.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.save('config1');
                     * console.log(viewsManager.all());
                     * ```
                     */
                    all(): IPivotTableView[];
                    /**
                     * @description apply a view to current pivot table.
                     * @param {string} name Indicates the name of view to apply.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.save('config1');
                     * viewsManager.apply('config1');
                     * ```
                     */
                    apply(name: string): void;
                    /**
                     * @description get a view from pivot table views.
                     * @param {string} name Indicates the name of view to query.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.save('config1');
                     * viewsManager.get('config1');
                     * ```
                     */
                    get(name: string): IPivotTableView;
                    /**
                     * @description remove a view from pivot table views.
                     * @param {string} name Indicates the name of view to remove.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.remove('config1');
                     * viewsManager.get('config1');
                     * ```
                     */
                    remove(name: string): void;
                    /**
                     * @description  Add a view to pivot table views.
                     * @param {string} name Indicates the name of view to save.
                     * @example
                     * ```javascript
                     * var viewsManager = pivotTable.views;
                     * viewsManager.save('config1');
                     * viewsManager.get('config1');
                     * ```
                     */
                    save(name: string): boolean;
                }
            }

        }

        module Report{

            export interface IColumnLayoutSetting{
                type: 'ColumnLayout';
                dataRange: string;
                columnCount: number;
                repeatCols?: { start: number; end: number; };
            }


            export interface IConditionRuleCellType{
                cell: string;
            }


            export interface IConditionRuleDataColumnType{
                dataColumn: GC.Spread.Report.IDataColumn;
            }


            export interface IConditionRuleFormulaType{
                formula: string;
            }


            export interface IConditionRuleParameterType{
                parameter: string;
            }


            export interface IConditionRuleValueType{
                value: string | boolean | number | Date;
            }


            export interface IDataCell{
                value: any;
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
            }


            export interface IDataCellBase{
                binding?: GC.Spread.Report.Binding;
                spillDirection?: GC.Spread.Report.SpillDirection;
                filter?: GC.Spread.Report.IFilter;
                sortOptions?: GC.Spread.Report.SortOption[];
                showCollapseButton?: boolean;
                initialExpansionState?: 'Expanded' | 'Collapsed';
                alias?: string;
                autoExpand?: 'Both' | 'Horizontal' | 'Vertical' | 'None';
            }


            export interface IDataColumn{
                tableName: string;
                columnName: string;
            }


            export interface IEntityFieldInfo{
                oldValue: GC.Spread.Report.DataType | GC.Spread.Report.DataType[];
                state: GC.Spread.Report.FieldState;
                row: number;
                col: number;
                depRowCols?: {
                row: number;
                col: number;
                }[];
            }


            export interface IFailedRecord extends GC.Spread.Report.IRecord{
                /**
                 * the reason is the DataManager table submitChanges failed records reason.
                 */
                reason?: string;
            }


            export interface IField{
                dbColumnName: string;
                formula: string;
                isPrimary?: boolean;
            }


            export interface IFilter{
                condition?: GC.Spread.Report.FilterCondition;
            }


            export interface IFilterConditionAndRelationType{
                and?: GC.Spread.Report.FilterCondition[];
            }


            export interface IFilterConditionOrRelationType{
                or?: GC.Spread.Report.FilterCondition[];
            }


            export interface IGroupBin{
                caption: string;
                groupBy: string;
                stopIfTrue?: boolean;
                alwaysVisible?: boolean;
            }


            export interface IPaginationSetting{
                titleRow?: { start: number; end: number; };
                endRow?: { start: number; end: number; };
                titleCol?: { start: number; end: number; };
                endCol?: { start: number; end: number; };
                rowPagination?: { paginationDataCell: string; rowCountPerPage: number; };
                paginationOrder?: GC.Spread.Report.PaginationOrder;
                paperSizePagination?: boolean;
            }


            export interface IParameterChangedArgs{
                tag: string;
                oldValue: string | number | boolean | Date;
                newValue: string | number | boolean | Date;
            }


            export interface IRecord{
                /**
                 * record key-value object.
                 */
                entity: {
                [key: string]: DataType;
                };
                /**
                 * record property info.
                 */
                info: {
                [key: string]: GC.Spread.Report.IEntityFieldInfo
                };
            }


            export interface IReportOptions{
                dirtyStyle?: GC.Spread.Sheets.Style;
                printAllPages?: boolean;
                showHiddenRowCol?: boolean;
                hiddenRowColStyle?: GC.Spread.Sheets.Style;
            }


            export interface IReportSheetDataChangedEventArgs{
                sheet: GC.Spread.Report.ReportSheet;
                sheetName: string;
                type: "update" | "insert" | "delete";
                row: number;
                col: number;
            }


            export interface IReportSheetDataChangingEventArgs{
                sheet: GC.Spread.Report.ReportSheet;
                sheetName: string;
                type: "update" | "insert" | "delete";
                row: number;
                col: number;
                cancel: boolean;
                oldValue?: any;
                newValue?: any;
            }


            export interface IReportSheetRecordsSubmittedEventArgs{
                sheet: GC.Spread.Report.ReportSheet;
                sheetName: string;
                /**
                 * include updated and inserted records.
                 */
                updateSuccessRecords: GC.Spread.Report.IRecord[];
                updateFailedRecords: GC.Spread.Report.IFailedRecord[];
                /**
                 * include deleted records.
                 */
                deleteSuccessRecords: GC.Spread.Report.IRecord[];
                deleteFailedRecords: GC.Spread.Report.IFailedRecord[];
            }


            export interface IReportSheetRecordsSubmittingEventArgs{
                sheet: GC.Spread.Report.ReportSheet;
                sheetName: string;
                cancel: boolean;
            }


            export interface IRowLayoutSetting{
                type: 'RowLayout';
                dataRange: string;
                rowCount: number;
                repeatRows?: { start: number; end: number; };
            }


            export interface ITemplateCellCommonSetting{
                context?: {
                horizontal?: GC.Spread.Report.CellAddress;
                vertical?: GC.Spread.Report.CellAddress;
                },
                paginateBeforeRow?: boolean;
                paginateAfterRow?: boolean;
                paginateBeforeColumn?: boolean;
                paginateAfterColumn?: boolean;
                canBreakWhenPaging?: boolean;
                repeatContentWhenPaging?: boolean;
                defaultValue?: GC.Spread.Report.Formula;
                autoFit?: 'None' | 'Row' | 'Column';
            }


            export interface ITemplateOptions{
                expansionDirectionArrowColor: string;
                parentArrowColor: string;
            }


            export interface IWriteBackRule{
                name: string;
                fields: GC.Spread.Report.IField[];
                tableName: string;
                includeUnmodified?: boolean;
                skipRecordWithEmptyValue?: boolean;
            }


            /**
             * @typedef GC.Spread.Report.Binding
             * @type {string}
             */
            export type Binding = 
                string
            

            /**
             * @typedef GC.Spread.Report.CellAddress
             * @type {string}
             */
            export type CellAddress = 
                string
            

            /**
             * @typedef GC.Spread.Report.Change
             * @property {GC.Spread.Report.IRecord[]} records - The updated and inserted records.
             * @property {GC.Spread.Report.IRecord[]} deleteRecords - The deleted records.
             * @property {GC.Spread.Report.IWriteBackRule} rule - The report sheet dataEntry write back rule.
             */
            export type Change = 
                {
                    /**
                     * The updated and inserted records.
                     */
                    records: GC.Spread.Report.IRecord[];
                    /**
                     * The deleted records.
                     */
                    deleteRecords: GC.Spread.Report.IRecord[];
                    /**
                     * The report sheet dataEntry write back rule.
                     */
                    rule: GC.Spread.Report.IWriteBackRule;
                }
            

            /**
             * @typedef GC.Spread.Report.ChartCell
             * @type {{type: 'Chart'; dataChartName: string;chartPreviewVisible: boolean;}}
             */
            export type ChartCell = 
                {type: 'Chart'; dataChartName: string; chartPreviewVisible?: boolean;}
            

            /**
             * @typedef GC.Spread.Report.ConditionRule
             * @type {{ column: string; operator: GC.Spread.Report.ConditionRuleOperator; } & (GC.Spread.Report.IConditionRuleValueType | GC.Spread.Report.IConditionRuleCellType | GC.Spread.Report.IConditionRuleDataColumnType | GC.Spread.Report.IConditionRuleParameterType | GC.Spread.Report.IConditionRuleFormulaType)}
             */
            export type ConditionRule = 
                { column: string; operator: GC.Spread.Report.ConditionRuleOperator; }
                & (GC.Spread.Report.IConditionRuleValueType | GC.Spread.Report.IConditionRuleCellType | GC.Spread.Report.IConditionRuleDataColumnType | GC.Spread.Report.IConditionRuleParameterType | GC.Spread.Report.IConditionRuleFormulaType)
            

            /**
             * @typedef GC.Spread.Report.ConditionRuleOperator
             * @type {'Equal' | 'NotEqual' | 'GreaterThanOrEqual' | 'GreaterThan' | 'LessThanOrEqual' | 'LessThan' | 'StartWith' | 'NotStartWith' | 'EndWith' | 'NotEndWith' | 'Contains' | 'NotContains' | 'In' | 'NotIn'}
             */
            export type ConditionRuleOperator = 
                'Equal' | 'NotEqual' | 'GreaterThanOrEqual' | 'GreaterThan' | 'LessThanOrEqual' | 'LessThan' | 'StartWith' | 'NotStartWith' | 'EndWith' | 'NotEndWith' | 'Contains' | 'NotContains' | 'In' | 'NotIn'
            

            /**
             * @typedef GC.Spread.Report.DataEntrySetting
             * @type {GC.Spread.Report.IWriteBackRule[]}
             */
            export type DataEntrySetting = 
                GC.Spread.Report.IWriteBackRule[]
            

            /**
             * @typedef GC.Spread.Report.DataType
             * @type {number | string | Date | boolean}
             */
            export type DataType = 
                number | string | Date | boolean
            

            /**
             * @typedef GC.Spread.Report.FieldState
             * @type {'normal' | 'new' | 'updated' | 'deleted'}
             */
            export type FieldState = 
                'normal' | 'new' | 'updated' | 'deleted'
            

            /**
             * @typedef GC.Spread.Report.FilterCondition
             * @type {GC.Spread.Report.ConditionRule | GC.Spread.Report.FilterConditionFormulaRule | GC.Spread.Report.IFilterConditionAndRelationType | GC.Spread.Report.IFilterConditionOrRelationType}
             */
            export type FilterCondition = 
                GC.Spread.Report.ConditionRule | GC.Spread.Report.FilterConditionFormulaRule | GC.Spread.Report.IFilterConditionAndRelationType | GC.Spread.Report.IFilterConditionOrRelationType
            

            /**
             * @typedef GC.Spread.Report.FilterConditionFormulaRule
             * @type { formula: string; }
             */
            export type FilterConditionFormulaRule = 
                { formula: string; }
            

            /**
             * @typedef GC.Spread.Report.Formula
             * @type {string}
             */
            export type Formula = 
                string
            

            /**
             * @typedef GC.Spread.Report.GroupCell
             * @type {GC.Spread.Report.IDataCellBase & { type: 'Group'; bins?: GC.Spread.Report.IGroupBin[]; formula?: GC.Spread.Report.Formula; fixedFillCount?: number; }}
             */
            export type GroupCell = 
                GC.Spread.Report.IDataCellBase & { type: 'Group'; bins?: GC.Spread.Report.IGroupBin[]; formula?: GC.Spread.Report.Formula; fixedFillCount?: number; }
            

            /**
             * @typedef GC.Spread.Report.InitParametersUIFunctionType
             * @param {GC.Spread.Sheets.Worksheet} sheet - The init parametersUI sheet, users can update the parameterUI sheet here.
             * @description The callback when parameterUI sheet is rendering.
             */
            export type InitParametersUIFunctionType = 
                (sheet: GC.Spread.Sheets.Worksheet) => void;
            

            /**
             * @typedef GC.Spread.Report.LayoutSetting
             * @type {GC.Spread.Report.IRowLayoutSetting | GC.Spread.Report.IColumnLayoutSetting}
             */
            export type LayoutSetting = 
                GC.Spread.Report.IRowLayoutSetting | GC.Spread.Report.IColumnLayoutSetting
            

            /**
             * @typedef GC.Spread.Report.ListCell
             * @type {GC.Spread.Report.IDataCellBase & { type: 'List'; spillMode?: 'Insert' | 'Overwrite'; fixedFillCount?: number; }}
             */
            export type ListCell = 
                GC.Spread.Report.IDataCellBase & { type: 'List'; spillMode?: 'Insert' | 'Overwrite'; fixedFillCount?: number; }
            

            /**
             * @typedef GC.Spread.Report.OnChangeFunctionType
             * @param {GC.Spread.Report.ReportSheet} sheet - The parametersUI bound reportSheet.
             * @param {GC.Spread.Report.IParameterChangedArgs} changedArgs - The changed parameter values.
             * @description The callback when the parameterUI value changed.
             */
            export type OnChangeFunctionType = 
                (sheet: GC.Spread.Report.ReportSheet, changedArgs: GC.Spread.Report.IParameterChangedArgs) => void;
            

            /**
             * @typedef GC.Spread.Report.PaginationOrder
             * @type {'DownThenOver' | 'OverThenDown'}
             */
            export type PaginationOrder = 
                'DownThenOver' | 'OverThenDown'
            

            /**
             * @typedef GC.Spread.Report.RenderMode
             * @type {'Design' | 'PaginatedPreview' | 'Preview'}
             */
            export type RenderMode = 
                'Design' | 'PaginatedPreview' | 'Preview'
            

            /**
             */
            export type SheetNameGenerator = 
                (pageIndex: number) => string
            

            /**
             * @typedef GC.Spread.Report.SortOption
             * @type {({ asc: string } | { desc: string }) & { list?: string[] }}
             */
            export type SortOption = 
                ({ asc: string } | { desc: string }) & { list?: string[] };
            

            /**
             * @typedef GC.Spread.Report.SpillDirection
             * @type {'Horizontal' | 'Vertical' | 'None'}
             */
            export type SpillDirection = 
                'Horizontal' | 'Vertical' | 'None'
            

            /**
             * @typedef GC.Spread.Report.StaticCell
             * @type {{type: 'Static'; pin?: 'None' | 'Row' | 'Column' | 'Both'; autoExpand?: 'Both' | 'Horizontal' | 'Vertical' | 'None'; showCollapseButton?: boolean; initialExpansionState?: 'Expanded' | 'Collapsed'; preserveFormulaInExport?: boolean; }}
             */
            export type StaticCell = 
                {type: 'Static'; pin?: 'None' | 'Row' | 'Column' | 'Both'; autoExpand?: 'Both' | 'Horizontal' | 'Vertical' | 'None'; showCollapseButton?: boolean; initialExpansionState?: 'Expanded' | 'Collapsed'; preserveFormulaInExport?: boolean; }
            

            /**
             * @typedef GC.Spread.Report.SummaryAggregate
             * @type {'Sum' | 'Avg' | 'Max' | 'Min' | 'Count'}
             */
            export type SummaryAggregate = 
                'Sum' | 'Avg' | 'Max' | 'Min' | 'Count'
            

            /**
             * @typedef GC.Spread.Report.SummaryCell
             * @type {GC.Spread.Report.IDataCellBase & { type: 'Summary'; aggregate: GC.Spread.Report.SummaryAggregate; }}
             */
            export type SummaryCell = 
                GC.Spread.Report.IDataCellBase & { type: 'Summary'; aggregate: GC.Spread.Report.SummaryAggregate; }
            

            /**
             * @typedef GC.Spread.Report.ReportTemplateCell
             * @type {(GC.Spread.Report.ListCell | GC.Spread.Report.GroupCell | GC.Spread.Report.SummaryCell | GC.Spread.Report.StaticCell | GC.Spread.Report.ChartCell) & GC.Spread.Report.ITemplateCellCommonSetting}
             */
            export type TemplateCell = 
                (GC.Spread.Report.ListCell | GC.Spread.Report.GroupCell | GC.Spread.Report.SummaryCell | GC.Spread.Report.StaticCell | GC.Spread.Report.ChartCell) & GC.Spread.Report.ITemplateCellCommonSetting
            

            export class ReportSheet{
                /**
                 * Represents a ReportSheet.
                 * @class
                 * @param {string} name The name of the ReportSheet.
                 * @example
                 * ```javascript
                 * const spread = new GC.Spread.Sheets.Workbook('spread-host', { sheetCount: 0 });
                 * const reportSheet = spread.addSheetTab(0, 'orders-report', GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = reportSheet.getTemplate();
                 *
                 * const ordersTable = spread.dataManager().addTable('Orders', {
                 *     remote: {
                 *         read: {
                 *             url: 'https://demodata.mescius.io/northwind/api/v1/orders'
                 *         }
                 *     }
                 * });
                 *
                 * // load the data from remote.
                 * ordersTable.fetch().then(() => {
                 *     // set style for the template.
                 *     const headerStyle = new GC.Spread.Sheets.Style();
                 *     headerStyle.backColor = '#80CBC4';
                 *     headerStyle.foreColor = '#424242';
                 *     headerStyle.hAlign = GC.Spread.Sheets.HorizontalAlign.right;
                 *     headerStyle.font = '12px Maine';
                 *     const dataStyle = new GC.Spread.Sheets.Style();
                 *     dataStyle.foreColor = '#424242';
                 *     dataStyle.hAlign = GC.Spread.Sheets.HorizontalAlign.right;
                 *     dataStyle.font = '12px Maine';
                 *     const border = new GC.Spread.Sheets.LineBorder('#E0E0E0', 1);
                 *     dataStyle.borderBottom = border;
                 *     dataStyle.borderTop = border;
                 *     dataStyle.borderLeft = border;
                 *     dataStyle.borderRight = border;
                 *     const colWidthArray = [90, 90, 90, 80, 220, 150, 110];
                 *     colWidthArray.forEach((width, i) => {
                 *         templateSheet.setColumnWidth(i, width);
                 *     });
                 *     templateSheet.getRange('A1:G1').setStyle(headerStyle);
                 *     templateSheet.getRange('A2:G2').setStyle(dataStyle);
                 *     templateSheet.setFormatter(1, 2, 'yyyy-MM-dd');
                 *
                 *     // set value and binding for the template.
                 *     const columns = ['orderId', 'customerId', 'orderDate', 'freight', 'shipName', 'shipCity', 'shipCountry'];
                 *     columns.forEach((columnName, i) => {
                 *         templateSheet.setValue(0, i, `${columnName[0].toUpperCase()}${columnName.substring(1)}`);
                 *         templateSheet.setTemplateCell(1, i, {
                 *             type: 'List',
                 *             binding: `Orders[${columnName}]`,
                 *         });
                 *     });
                 *
                 *     reportSheet.refresh();
                 * });
                 * ```
                 */
                constructor(name: string);
                /**
                 * Indicates the options of the ReportSheet.
                 * @type {Object}
                 * @property {GC.Spread.Sheets.Style} [dirtyStyle] The style will be used to paint the modified cells in the preview mode.
                 * @property {boolean} [printAllPages] Indicates whether print all pages.
                 * @property {boolean} [showHiddenRowCol] Indicates whether to show hidden rows or columns.
                 * @property {GC.Spread.Sheets.Style} [hiddenRowColStyle] This style will be used to style the header when displaying hidden rows and columns.
                 */
                options: GC.Spread.Report.IReportOptions;
                /**
                 * Add record based on the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @example
                 * ```javascript
                 * // add record at A2 cell.
                 * reportSheet.addRecordAt(1, 0);
                 * ```
                 */
                addRecordAt(row: number,  col: number): void;
                /**
                 * Binds an event to the report sheet.
                 * @param {string} type The event type.
                 * @param {Object} data Optional. Specifies additional data to pass along to the function.
                 * @param {Function} fn Specifies the function to run when the event occurs.
                 * @example
                 * ```javascript
                 * //This example bind the ReportSheetDataChanged event into report sheet.
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setTemplateCell(0, 1, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setDataEntrySetting([ {
                 *    name: "Write Back Rule 1",
                 *    tableName: "Orders",
                 *    fields: [
                 *        { dbColumnName: "orderId", formula: "A1", isPrimary: true },
                 *        { dbColumnName: "customerId", formula: "B1" },
                 *    ],
                 *    includeUnmodified: false,
                 *    skipRecordWithEmptyValue: false
                 * } ]);
                 * report.renderMode("Preview");
                 * report.bind(GC.Spread.Sheets.Events.ReportSheetDataChanged, (event, args) => {
                 *     let reportsheet = args.sheet, changes = reportsheet.getChanges();
                 *     if (allowSubmit(changes)) { // users can submit or drop this changing.
                 *          reportsheet.submit(); // submit changes.
                 *     } else {
                 *          reportsheet.refresh(); // drop changes.
                 *     }
                 * });
                 * // after reportsheet edit / update / delete records in UI will trigger ReportSheetDataChanged event.
                 * ```
                 */
                bind(type: string,  data?: any,  fn?: Function): void;
                /**
                 * Get or set the current page index.
                 * @param {number} page The page index (0 base).
                 * @returns {number} Return the current page index.
                 * @example
                 * ```javascript
                 * // get the current page index.
                 * const page = reportSheet.currentPage();
                 *
                 * // go to the next page.
                 * reportSheet.currentPage(page + 1);
                 * ```
                 */
                currentPage(page?: number): number;
                /**
                 * Delete record based on the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @example
                 * ```javascript
                 * // delete record at A2 cell.
                 * reportSheet.deleteRecordAt(1, 0);
                 * ```
                 */
                deleteRecordAt(row: number,  col: number): void;
                /**
                 * Generate the every page of the current report as a worksheet, and return them in an array.
                 * @param {boolean} addToSpread Control whether add the page sheets to the current spread.
                 * @param {Function} sheetNameGenerator The optional sheet name generator.
                 * @returns {GC.Spread.Sheets.Worksheet[]} Return the generated worksheets.
                 * @example
                 * ```javascript
                 * // generate the pages and add them to the spread.
                 * const pageSheets = reportSheet.generatePageSheets(true);
                 *
                 * // generate the pages and add them to the spread, and customize the sheet name.
                 * const pageSheets = reportSheet.generatePageSheets(true, (i) => `report-page-${i + 1}`);
                 * ```
                 */
                generatePageSheets(addToSpread: boolean,  sheetNameGenerator?: GC.Spread.Report.SheetNameGenerator): GC.Spread.Sheets.Worksheet[];
                /**
                 * Get the actual style of the cell in current render mode.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Sheets.Style} Return actual style of the cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.getCell(0, 0).backColor("red");
                 * templateSheet.setTemplateCell(1, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.getCell(1, 0).backColor("green");
                 * const designActualStyle1 = report.getActualStyle(0, 0); // backColor: red
                 * const designActualStyle2 = report.getActualStyle(1, 0); // backColor: green
                 * report.renderMode("Preview");
                 * const previewActualStyle1 = report.getActualStyle(0, 0); // backColor: red
                 * const previewActualStyle2 = report.getActualStyle(1, 0); // backColor: red
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewActualStyle1 = report.getActualStyle(0, 0); // backColor: red
                 * const paginatedPreviewActualStyle2 = report.getActualStyle(1, 0); // backColor: red
                 * ```
                 */
                getActualStyle(row: number,  col: number): GC.Spread.Sheets.Style;
                /**
                 * Get expanded cells of the template cell base on the current cell in preview.
                 * @param {number} templateRow The row index in template.
                 * @param {number} templateCol The column index in template.
                 * @param {number} currentRow The row index of current cell in preview.
                 * @param {number} currentCol The column index of current cell in preview.
                 * @returns {GC.Spread.Report.IDataCell[]} Return expanded cells of the template cell base on the current cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setTemplateCell(1, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * report.regenerateReport();
                 * const allCustomerIdCells = report.getCells(0, 0); // all 89 cells
                 * const fistCustomerIdCell = report.getCells(0, 0, 0, 0); // value: VINET
                 * const allOrderIdCells = report.getCells(1, 0); // all 830 cells
                 * const firstOrderIdCell = report.getCells(1, 0, 89, 0); // value: 10248
                 * ```
                 */
                getCells(templateRow: number,  templateCol: number,  currentRow?: number,  currentCol?: number): GC.Spread.Report.IDataCell[];
                /**
                 * Return the report sheet data entry mode update insert and delete data changes.
                 * @returns {GC.Spread.Report.Change[]} Returns the reportsheet update insert and delete data changes.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    showCollapseButton: true
                 * });
                 * templateSheet.setTemplateCell(0, 1, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setDataEntrySetting([ {
                 *    name: "Write Back Rule 1",
                 *    tableName: "Orders",
                 *    fields: [
                 *        { dbColumnName: "orderId", formula: "A1", isPrimary: true },
                 *        { dbColumnName: "customerId", formula: "B1" },
                 *    ],
                 *    includeUnmodified: false,
                 *    skipRecordWithEmptyValue: false
                 * } ]);
                 * report.renderMode("Preview");
                 * report.updateCellValue(0, 1, "test");
                 * report.addRecordAt(1, 0);
                 * report.updateCellValue(2, 0, 111);
                 * report.updateCellValue(2, 1, "test2");
                 * report.deleteRecordAt(3, 0);
                 * report.getChanges(); // one delete record and two update records.
                 * ```
                 */
                getChanges(): GC.Spread.Report.Change[];
                /**
                 * Return the collapse state of the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    showCollapseButton: true
                 * });
                 * templateSheet.setTemplateCell(0, 1, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.addSpan(0, 0, 2, 1);
                 * report.renderMode("Preview");
                 * report.getCollapseState(0, 0); // false
                 * report.toggleCollapseState(0, 0);
                 * report.getCollapseState(0, 0); // true
                 * ```
                 */
                getCollapseState(row: number,  col: number): boolean;
                /**
                 * Get the column width by col index.
                 * @param {number} col The column index.
                 * @returns {number} Return column width.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    spillDirection: "Horizontal"
                 * });
                 * templateSheet.setColumnWidth(0, 100);
                 * const designColumnWidth1 = report.getColumnWidth(0); // ColumnWidth: 100
                 * const designColumnWidth2 = report.getColumnWidth(1); // ColumnWidth: 62
                 * report.renderMode("Preview");
                 * const previewColumnWidth1 = report.getColumnWidth(0); // ColumnWidth: 100
                 * const previewColumnWidth2 = report.getColumnWidth(1); // ColumnWidth: 100
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewColumnWidth1 = report.getColumnWidth(0); // ColumnWidth: 100
                 * const paginatedPreviewColumnWidth2 = report.getColumnWidth(1); // ColumnWidth: 100
                 * ```
                 */
                getColumnWidth(col: number): number;
                /**
                 * Get the number of pages in the report.
                 * @returns {number} Return the number of pages in the report.
                 * @example
                 * ```javascript
                 * const pageCount = reportSheet.getPageCount();
                 * ```
                 */
                getPageCount(): number;
                /**
                 * Get the pages count of the report.
                 * @returns {number} Return the pages count of the report.
                 * @deprecated since version 19.0.0, please use 'getPageCount' instead
                 * @example
                 * ```javascript
                 * // get the pages count.
                 * const pagesCount = reportSheet.getPagesCount();
                 * ```
                 */
                getPagesCount(): number;
                /**
                 * @returns {GC.Spread.Sheets.Range} Return the report sheet range.
                 * If the render mode is Design, return the template sheet range.
                 * If the render mode is Preview, return the whole report range.
                 * If the render mode is PaginatedPreview, return the current page's range.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(199, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * const designRange = report.getRange(); // row: 0, col: 0, rowCount: 200, colCount: 20.
                 * report.renderMode("Preview");
                 * const previewRange = report.getRange(); // row: 0, col: 0, rowCount: 288, colCount: 1.
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewRange = report.getRange(); // row: 0, col: 0, rowCount: 45, colCount: 1.
                 * ```
                 */
                getRange(): GC.Spread.Sheets.Range;
                /**
                 * Get the row height by row index.
                 * @param {number} row The row index.
                 * @returns {number} Return row height.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setRowHeight(0, 30);
                 * const designRowHeight1 = report.getRowHeight(0); // RowHeight: 30
                 * const designRowHeight2 = report.getRowHeight(1); // RowHeight: 20
                 * report.renderMode("Preview");
                 * const previewRowHeight1 = report.getRowHeight(0); // RowHeight: 30
                 * const previewRowHeight2 = report.getRowHeight(1); // RowHeight: 30
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewRowHeight1 = report.getRowHeight(0); // RowHeight: 30
                 * const paginatedPreviewRowHeight2 = report.getRowHeight(1); // RowHeight: 30
                 * ```
                 */
                getRowHeight(row: number): number;
                /**
                 * Get the range of the cell in current render mode.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Sheets.Range} Return range of the cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.addSpan(0, 0, 2, 1);
                 * const designSpan1 = report.getSpan(0, 0); // span: row: 0, col: 0, rowCount: 2, colCount: 1
                 * const designSpan2 = report.getSpan(2, 0); // span: null
                 * report.renderMode("Preview");
                 * const previewSpan1 = report.getSpan(0, 0); // span: row: 0, col: 0, rowCount: 2, colCount: 1
                 * const previewSpan2 = report.getSpan(2, 0); // span: row: 2, col: 0, rowCount: 2, colCount: 1
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewSpan1 = report.getSpan(0, 0); // span: row: 0, col: 0, rowCount: 2, colCount: 1
                 * const paginatedPreviewSpan2 = report.getSpan(2, 0); // span: row: 2, col: 0, rowCount: 2, colCount: 1
                 * ```
                 */
                getSpan(row: number,  col: number): GC.Spread.Sheets.Range;
                /**
                 * Get the style of the cell in current render mode.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Sheets.Style} Return style of the cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.getCell(0, 0).backColor("red");
                 * templateSheet.setTemplateCell(1, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.getCell(1, 0).backColor("green");
                 * const designStyle1 = report.getStyle(0, 0); // backColor: red
                 * const designStyle2 = report.getStyle(1, 0); // backColor: green
                 * report.renderMode("Preview");
                 * const previewStyle1 = report.getStyle(0, 0); // backColor: red
                 * const previewStyle2 = report.getStyle(1, 0); // backColor: red
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewStyle1 = report.getStyle(0, 0); // backColor: red
                 * const paginatedPreviewStyle2 = report.getStyle(1, 0); // backColor: red
                 * ```
                 */
                getStyle(row: number,  col: number): GC.Spread.Sheets.Style;
                /**
                 * get the template sheet of the current report sheet.
                 * @returns {GC.Spread.Report.TemplateSheet} Return the template sheet of the current report sheet.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate(); // get the reportSheet templateSheet
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * report.renderMode("Preview");
                 * ```
                 */
                getTemplate(): GC.Spread.Report.TemplateSheet;
                /**
                 * Get template cell of specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Report.TemplateCell} Return the template cell of specified cell.
                 * If the render mode is Design, return the template cell of the template sheet in the specified row col.
                 * If the render mode is Preview, return the template cell of the specified cell.
                 * If the render mode is PaginatedPreview, return the template cell of the specified cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setTemplateCell(1, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * const designTemplateCell1 = report.getTemplateCell(0, 0); // binding: Orders[customerId]
                 * const designTemplateCell2 = report.getTemplateCell(1, 0); // binding: Orders[orderId]
                 * report.renderMode("Preview");
                 * const previewTemplateCell1 = report.getTemplateCell(0, 0); // binding: Orders[customerId]
                 * const previewTemplateCell2 = report.getTemplateCell(1, 0); // binding: Orders[customerId]
                 * const previewTemplateCell3 = report.getTemplateCell(89, 0); // binding: Orders[orderId]
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewTemplateCell1 = report.getTemplateCell(0, 0); // binding: Orders[customerId]
                 * const paginatedPreviewTemplateCell2 = report.getTemplateCell(1, 0); // binding: Orders[customerId]
                 * ```
                 */
                getTemplateCell(row: number,  col: number): GC.Spread.Report.TemplateCell;
                /**
                 * Get the value of the cell in current render mode.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {any} Return value of the cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setValue(1, 0, "test");
                 * const designValue = report.getValue(1, 0); // test
                 * report.renderMode("Preview");
                 * const previewValue = report.getValue(1, 0); // TOMSP
                 * report.renderMode("PaginatedPreview");
                 * const paginatedPreviewValue = report.getValue(1, 0); // TOMSP
                 * ```
                 */
                getValue(row: number,  col: number): any;
                /**
                 * Return a boolean value indicate wether the report has un-submit changes.
                 * @returns {boolean} Return true if the report has un-submit changes, otherwise return false.
                 * @example
                 * ```javascript
                 * // if the current reportSheet has un-submit changes, isDirty will be true.
                 * const isDirty = reportSheet.isDirty();
                 * ```
                 */
                isDirty(): boolean;
                /**
                 * Update the report sheet template, and also the report will be regenerated based on the new template and parameter values.
                 * @param {Object} templateJson The template json.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = new GC.Spread.Report.TemplateSheet("Template");
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * report.loadTemplate(templateSheet.toJSON());
                 * ```
                 */
                loadTemplate(templateJson: Object): void;
                /**
                 * Gets or sets the name of the ReportSheet.
                 * @param {string} [value] The name of the ReportSheet.
                 * @returns {string} returns the ReportSheet name.
                 * @example
                 * ```javascript
                 * // get the report sheet name
                 * const name = reportSheet.name();
                 *
                 * // set the report sheet name
                 * reportSheet.name('new-name');
                 * ```
                 */
                name(value?: string): any;
                /**
                 * Get or set the parameter in reportsheet.
                 * @param {GC.Spread.Sheets.IParameter} [parameter] Set new parameter into reportsheet.
                 * @returns {GC.Spread.Sheets.IParameter} Returns the reportsheet current working parameter.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * report.getTemplate().setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    filter: {
                 *        condition: {
                 *            column: "customerId",
                 *            operator: "Equal",
                 *            parameter: "customerId"
                 *        }
                 *    }
                 * });
                 * report.renderMode("Preview");
                 *
                 * let parameter = report.parameter(); // get the reportsheet parameter
                 * parameter.customerId = "VINET";
                 * report.parameter(parameter); // set the parameter.
                 * report.regenerateReport(); // regenerate reportsheet according to the new parameter.
                 * ```
                 */
                parameter(parameter?: GC.Spread.Sheets.IParameter): GC.Spread.Sheets.IParameter;
                /**
                 * Gets or sets the print information for the report sheet.
                 * @param {GC.Spread.Sheets.Print.PrintInfo} [value] The print information for the report sheet.
                 * @returns {GC.Spread.Sheets.Print.PrintInfo | GC.Spread.Report.ReportSheet} If no value is set, returns the print information for the report sheet; otherwise, returns the report sheet.
                 * @example
                 * ```javascript
                 * // set the paper size.
                 * const printInfo = reportSheet.printInfo();
                 * printInfo.paperSize(new GC.Spread.Sheets.Print.PaperSize(GC.Spread.Sheets.Print.PaperKind.a3));
                 * ```
                 */
                printInfo(value?: GC.Spread.Sheets.Print.PrintInfo): GC.Spread.Sheets.Print.PrintInfo | GC.Spread.Report.ReportSheet;
                /**
                 * Get or set the page index array that will be print. empty array indicates print all the pages.
                 * @param {number[]} pageIndexes The print page index array (0 base).
                 * @returns {number[]} Return the page index array that will be print.
                 * @example
                 * ```javascript
                 * // print only the first and fifth pages of the current report.
                 * reportSheet.printPageIndexes([0, 4]);
                 * reportSheet.options.printAllPages = true;
                 * spread.print();
                 *
                 * // clear the print page indexes setting.
                 * reportSheet.printPageIndexes([]);
                 * ```
                 */
                printPageIndexes(pageIndexes?: number[]): number[];
                /**
                 * Refresh the current report sheet, all the render modes can support this function.
                 * Design: Refresh the cache of the template sheet and repaint.
                 * Preview: Regenerate the report base on the current template sheet and repaint.
                 * PaginatedPreview: Regenerate the report base on the current template sheet and repaint.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * report.refresh();
                 * ```
                 */
                refresh(): void;
                /**
                 * Regenerate the report base on the current template.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * report.regenerateReport();
                 * ```
                 */
                regenerateReport(): void;
                /**
                 * Get or set the render mode.
                 * @param {GC.Spread.Report.RenderMode} renderMode The render mode.
                 * @returns {GC.Spread.Report.RenderMode} Return the render mode.
                 * @example
                 * ```javascript
                 * // switch to design mode.
                 * reportSheet.renderMode('Design');
                 *
                 * // switch to preview mode.
                 * reportSheet.renderMode('Preview');
                 *
                 * // switch to paginated preview mode.
                 * reportSheet.renderMode('PaginatedPreview');
                 * ```
                 */
                renderMode(renderMode?: GC.Spread.Report.RenderMode): GC.Spread.Report.RenderMode;
                /**
                 * Repaint the current report sheet.
                 */
                repaint(): void;
                /**
                 * Reset specified cell's value.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @example
                 * ```javascript
                 * // If the A2 cell's value is modified, resetCellValue can reset the A2 cell to its original value.
                 * reportSheet.resetCellValue(1, 0);
                 * ```
                 */
                resetCellValue(row: number,  col: number): void;
                /**
                 * Set the reportSheet parametersUI.
                 * @param {HTMLElement | string} host - The parameterUI host element or host id.
                 * @param {GC.Spread.Report.InitParametersUIFunctionType} initParametersUI - The callback to modify the parameterUI sheet.
                 * @param {GC.Spread.Report.OnChangeFunctionType} onChange - The callback when the parameterUI value changed or button clicked.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * report.parameter({ customerId: "VINET" });
                 * report.getTemplate().setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    filter: {
                 *        condition: {
                 *            column: "customerId",
                 *            operator: "Equal",
                 *            parameter: "customerId"
                 *        }
                 *    }
                 * });
                 * report.renderMode("Preview");
                 *
                 * const host = document.getElementById("parameterUIHost");
                 * report.setParametersUI(host, initParametersUI, onChanged);
                 * function initParametersUI (sheet) {
                 *     sheet.getCell(3, 3).value("CustomerId:"); // add static label cell
                 *     sheet.getCell(3, 4).bindingPath("customerId").tag("customerId"); // add parameter binding path and tag to this cell
                 *     const submitButton = new GC.Spread.Sheets.CellTypes.Button(); // add submit button
                 *     submitButton.text("Submit");
                 *     sheet.getCell(3, 5).cellType(submitButton).tag("submitButton"); // set button cell type and tag to this cell
                 * }
                 * function onChanged (reportSheet, changedArgs) {
                 *     if (changedArgs.tag === "submitButton") { // submit button clicked.
                 *         reportSheet.regenerateReport();
                 *     }
                 *     if (changedArgs.tag === "customerId") {
                 *         changedArgs.newValue = changedArgs.newValue.toUpperCase(); // update newValue here.
                 *     }
                 * }
                 * ```
                 */
                setParametersUI(host: HTMLElement | string,  initParametersUI: GC.Spread.Report.InitParametersUIFunctionType,  onChange: GC.Spread.Report.OnChangeFunctionType): void;
                /**
                 * Submit the changes to the remote database based on your data entry setting through the DataManger.
                 * @example
                 * ```javascript
                 * // submit the changes.
                 * reportSheet.submit();
                 * ```
                 */
                submit(): void;
                /**
                 * Toggle the collapse state of the specified cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {'Collapsed' | 'Expanded'} targetState The target state, collapse or expand the specific cell.
                 * @param {boolean} recursive True will update the state of the current cell and all the descendant cells, false will only update the current cell.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 *    showCollapseButton: true
                 * });
                 * templateSheet.setTemplateCell(0, 1, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.addSpan(0, 0, 2, 1);
                 * report.renderMode("Preview");
                 * report.toggleCollapseState(0, 0);
                 * ```
                 */
                toggleCollapseState(row: number,  col: number,  targetState?: 'Collapsed' | 'Expanded',  recursive?: boolean): void;
                /**
                 * Generated a static worksheet(no formula, no cell binding) base on the current report sheet.
                 * @returns {GC.Spread.Sheets.Worksheet} Return the generated worksheet.
                 * @example
                 * ```javascript
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * report.renderMode("Preview");
                 * const reportWorksheet = report.toWorksheet();
                 * ```
                 */
                toWorksheet(): GC.Spread.Sheets.Worksheet;
                /**
                 * Removes the binding of an event to the report sheet.
                 * @param {string} type The event type.
                 * @param {Function} fn Specifies the function for which to remove the binding.
                 * @example
                 * ```javascript
                 * //This example unbind the ReportSheetDataChanged event after first data changing.
                 * const report = spread.addSheetTab(0, "Report", GC.Spread.Sheets.SheetType.reportSheet);
                 * report.renderMode("Design");
                 * const templateSheet = report.getTemplate();
                 * templateSheet.setTemplateCell(0, 0, {
                 *    binding: "Orders[orderId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setTemplateCell(0, 1, {
                 *    binding: "Orders[customerId]",
                 *    type: "Group",
                 * });
                 * templateSheet.setDataEntrySetting([ {
                 *    name: "Write Back Rule 1",
                 *    tableName: "Orders",
                 *    fields: [
                 *        { dbColumnName: "orderId", formula: "A1", isPrimary: true },
                 *        { dbColumnName: "customerId", formula: "B1" },
                 *    ],
                 *    includeUnmodified: false,
                 *    skipRecordWithEmptyValue: false
                 * } ]);
                 * report.renderMode("Preview");
                 * report.bind(GC.Spread.Sheets.Events.ReportSheetDataChanged, (event, args) => {
                 *     let reportsheet = args.sheet, changes = reportsheet.getChanges();
                 *     console.log(changes);
                 *     reportsheet.unbind(GC.Spread.Sheets.Events.ReportSheetDataChanged);
                 * });
                 * // after reportsheet edit / update / delete records in UI will trigger ReportSheetDataChanged event.
                 * ```
                 */
                unbind(type: string,  fn?: Function): void;
                /**
                 * Update the specified cell's value.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {any} value The new value of the cell.
                 * @example
                 * ```javascript
                 * // update the A2 cell's value.
                 * reportSheet.updateCellValue(1, 0, 'test');
                 * ```
                 */
                updateCellValue(row: number,  col: number,  value: any): void;
            }

            export class TemplateSheet extends GC.Spread.Sheets.Worksheet{
                /**
                 * Represents a TemplateSheet.
                 * @class
                 * @param {string} name The name of the TemplateSheet.
                 * @example
                 * ```javascript
                 * const spread = new GC.Spread.Sheets.Workbook('spread-host', { sheetCount: 0 });
                 * const reportSheet = spread.addSheetTab(0, 'orders-report', GC.Spread.Sheets.SheetType.reportSheet);
                 * const templateSheet = reportSheet.getTemplate();
                 * templateSheet.setValue(0, 0, 'test');
                 * ```
                 */
                constructor(name: string);
                /**
                 * Get the data entry setting.
                 * @returns {GC.Spread.Report.DataEntrySetting} Return the data entry setting.
                 * @example
                 * ```javascript
                 * // get the data entry setting.
                 * const dataEntrySetting = templateSheet.getDataEntrySetting();
                 * ```
                 */
                getDataEntrySetting(): GC.Spread.Report.DataEntrySetting;
                /**
                 * Get the layout setting.
                 * @returns {GC.Spread.Report.LayoutSetting} Return the layout setting.
                 * @example
                 * ```javascript
                 * // get the layout setting.
                 * const layoutSetting = templateSheet.getLayoutSetting();
                 * ```
                 */
                getLayoutSetting(): GC.Spread.Report.LayoutSetting;
                /**
                 * Get the pagination setting.
                 * @returns {GC.Spread.Report.IPaginationSetting} Return the pagination setting.
                 * @example
                 * ```javascript
                 * // get the pagination setting.
                 * const paginationSetting = templateSheet.getPaginationSetting();
                 * ```
                 */
                getPaginationSetting(): GC.Spread.Report.IPaginationSetting;
                /**
                 * Get the template cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @returns {GC.Spread.Report.TemplateCell} Returns the template cell.
                 * @example
                 * ```javascript
                 * // get the A2 template cell's setting.
                 * const templateCell = templateSheet.getTemplateCell(1, 0);
                 * ```
                 */
                getTemplateCell(row: number,  col: number): GC.Spread.Report.TemplateCell;
                /**
                 * Set the data entry setting.
                 * @param {GC.Spread.Report.DataEntrySetting} dataEntrySetting The data entry setting.
                 * @example
                 * ```javascript
                 * // add the customers table.
                 * const customersTable = spread.dataManager().addTable('Customers', {
                 *     remote: {
                 *         read: { url: 'https://demodata.mescius.io/northwind/api/v1/customers' },
                 *         batch: (data) => {
                 *             // sync the changes to the server here.
                 *             console.log('changes: ', data);
                 *             return Promise.resolve(data.map((item) => ({ succeed: true })));
                 *         },
                 *     },
                 *     batch: true,
                 * });
                 *
                 * // set binding for the template
                 * const columns = ['customerId', 'companyName', 'contactName', 'contactTitle', 'address', 'region', 'country', 'city', 'postalCode', 'phone', 'fax'];
                 * columns.forEach((columnName, i) => {
                 *     templateSheet.setValue(0, i, `${columnName[0].toUpperCase()}${columnName.substring(1)}`);
                 *     templateSheet.setTemplateCell(1, i, {
                 *         type: 'List',
                 *         binding: `Customers[${columnName}]`,
                 *     });
                 * });
                 *
                 * // config the data entry setting
                 * const customerRule = {
                 *     name: 'customers',
                 *     tableName: 'Customers',
                 *     fields: [
                 *         { formula: 'A2', dbColumnName: 'customerId', isPrimary: true },
                 *         { formula: 'B2', dbColumnName: 'companyName' },
                 *         { formula: 'C2', dbColumnName: 'contactName' },
                 *         { formula: 'D2', dbColumnName: 'contactTitle' },
                 *         { formula: 'E2', dbColumnName: 'address' },
                 *         { formula: 'F2', dbColumnName: 'region' },
                 *         { formula: 'G2', dbColumnName: 'country' },
                 *         { formula: 'H2', dbColumnName: 'city' },
                 *         { formula: 'I2', dbColumnName: 'postalCode' },
                 *         { formula: 'J2', dbColumnName: 'phone' },
                 *         { formula: 'K2', dbColumnName: 'fax' },
                 *     ],
                 * };
                 * templateSheet.setDataEntrySetting([customerRule]);
                 * ```
                 */
                setDataEntrySetting(dataEntrySetting: GC.Spread.Report.DataEntrySetting): void;
                /**
                 * Set the layout setting.
                 * @param {GC.Spread.Report.LayoutSetting} setting The layout setting.
                 * @example
                 * ```javascript
                 * const dataManager = spread.dataManager();
                 * dataManager.addTable("orders",
                 *      {
                 *           remote: {
                 *               read: {
                 *                   url: "https://demodata.mescius.io/northwind/api/v1/orders"
                 *               }
                 *           }
                 *      }
                 * );
                 * templateSheet.setTemplateCell(1, 0, { type: 'List', binding: 'orders[orderId]' });
                 * const setting = {
                 *      dataRange: "A2",
                 *      type: "RowLayout",
                 *      rowCount: 10
                 * }
                 * templateSheet.setLayoutSetting(setting);
                 * ```
                 */
                setLayoutSetting(setting: GC.Spread.Report.LayoutSetting): void;
                /**
                 * Set the pagination setting.
                 * @param {GC.Spread.Report.IPaginationSetting} setting The pagination setting.
                 * @example
                 * ```javascript
                 * // set the paper size pagination.
                 * const printInfo = templateSheet.printInfo();
                 * printInfo.paperSize().kind(GC.Spread.Sheets.Print.PaperKind.a4);
                 * templateSheet.setPaginationSetting({
                 *     paperSizePagination: true,
                 *     titleRow: {
                 *         start: 0,
                 *         end: 0,
                 *     },
                 *     titleCol: {
                 *         start: 0,
                 *         end: 0,
                 *     },
                 *     paginationOrder: 'OverThenDown',
                 * });
                 *
                 * // set row pagination, row pagination only can work for the list template cell.
                 * templateSheet.setTemplateCell(1, 0, { type: 'List', binding: 'orders[orderId]' });
                 * templateSheet.setPaginationSetting({
                 *     rowPagination: {
                 *         paginationDataCell: 'A2',
                 *         rowCountPerPage: 20,
                 *     },
                 *     titleRow: {
                 *         start: 0,
                 *         end: 0,
                 *     },
                 * });
                 * ```
                 */
                setPaginationSetting(setting: GC.Spread.Report.IPaginationSetting): void;
                /**
                 * Set the template cell.
                 * @param {number} row The row index.
                 * @param {number} col The column index.
                 * @param {GC.Spread.Report.TemplateCell} templateCell The template cell.
                 * @example
                 * ```javascript
                 * const dataManager = spread.dataManager();
                 * dataManager.addTable("orders",
                 *      {
                 *           remote: {
                 *               read: {
                 *                   url: "https://demodata.mescius.io/northwind/api/v1/orders"
                 *               }
                 *           }
                 *      }
                 * );
                 * // set the template cell setting for the A2 cell.
                 * templateSheet.setTemplateCell(1, 0, { type: 'List', binding: 'orders[shipCity]' });
                 * ```
                 */
                setTemplateCell(row: number,  col: number,  templateCell: GC.Spread.Report.TemplateCell): void;
            }
        }

        module Sheets{
            /**
             * Represents the license key for evaluation version and production version.
             */
            var LicenseKey: string;
            /**
             * Gets the Workbook instance by the host element.
             * @param {HTMLElement|string} host The host element or the host element id.
             * @returns {GC.Spread.Sheets.Workbook} The Workbook instance.
             */
            function findControl(host: HTMLElement|string): GC.Spread.Sheets.Workbook;
            /**
             * Gets the type from the type string. This method is used for supporting the serialization of the custom object.
             * @param {string} typeString The type string.
             * @returns {Object} The type.
             */
            function getTypeFromString(typeString: string): any;

            export interface FloatingObjectLoadedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                element: HTMLElement;
            }


            export interface FormattedData{
                conditionalForeColor?: string | CanvasGradient | CanvasPattern;
                value?: string | CanvasGradient | CanvasPattern;
                content?: {
                type: string;
                value: string;
                }[];
            }


            export interface IActiveSheetChangedEventArgs{
                oldSheet: GC.Spread.Sheets.Worksheet;
                newSheet: GC.Spread.Sheets.Worksheet;
            }


            export interface IActiveSheetChangingEventArgs{
                oldSheet: GC.Spread.Sheets.Worksheet;
                newSheet: GC.Spread.Sheets.Worksheet;
                cancel: boolean;
            }


            export interface IAddColumnButtonOption{
                visible?: boolean;
                command?: string;
                style?: GC.Spread.Sheets.Style;
                width?: number;
                tooltip?: string;
            }


            export interface IAddRowButtonOption{
                visible?: boolean;
                command?: string;
                style?: GC.Spread.Sheets.Style;
                height?: number;
                tooltip?: string;
            }


            export interface IAppDocProps{
                manager?: string;
                company?: string;
                hyperlinkBase?: string;
                totalTime?: number;
                template?: string;
            }


            export interface IBaseSpreadOption{
                width?: number;
                height?: number;
            }


            export interface IBounds{
                x?: number;
                y?: number;
                width: number;
                height: number;
            }


            export interface IBuiltInFileIcons{
                [key: string]: string;
            }


            export interface IButtonClickedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                sheetArea: SheetArea;
            }


            export interface ICalendarDateRange{
                start: Date;
                end: Date;
            }


            export interface ICalendarOption{
                startDay?: GC.Spread.Sheets.CalendarStartDay;
                calendarPage?: GC.Spread.Sheets.CalendarPage;
                showTime?: boolean;
                showDateRange?: boolean;
                showBuiltInDateRange?: boolean;
                defaultDateTime?: Date | GC.Spread.Sheets.ICalendarDateRange;
                maxDate?: Date;
                minDate?: Date;
            }


            export interface ICellButton{
                position?: GC.Spread.Sheets.ButtonPosition;
                useButtonStyle?: boolean;
                enabled?: boolean;
                width?: number;
                caption?: string;
                imageSrc?: string;
                imageSize?: GC.Spread.Sheets.IImageSize;
                captionAlign?: GC.Spread.Sheets.CaptionAlignment;
                command?: string | ((sheet: GC.Spread.Sheets.Worksheet, row: number, col: number, option: any) => void);
                imageType?: GC.Spread.Sheets.ButtonImageType;
                visibility?: GC.Spread.Sheets.ButtonVisibility;
                hoverBackColor?: string;
                buttonBackColor?: string;
            }


            export interface ICellChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                isUndo?: boolean;
            }


            export interface ICellClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetArea: SheetArea;
                row: number;
                col: number;
            }


            export interface ICellColorSortInfo{
                order: "top" | "bottom";
                backColor: string | GC.Spread.Sheets.IGradientFill | GC.Spread.Sheets.IGradientPathFill | GC.Spread.Sheets.IPatternFill | undefined;
                index: number;
            }


            export interface ICellDoubleClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetArea: SheetArea;
                row: number;
                col: number;
            }


            export interface ICellPosition{
                row: number;
                col: number;
            }


            export interface ICellsInfo{
                row: number;
                col: number;
                rowCount?: number;
                colCount?: number;
                sheetName: string;
            }


            export interface IClearChangeInfo{
                row?: number;
                col?: number;
                rowCount?: number;
                colCount?: number;
                clearType?: GC.Spread.Sheets.ClearPendingChangeType;
            }


            export interface IClipboardChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                copyData: GC.Spread.Sheets.ICopyData;
            }


            export interface IClipboardChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                copyData: GC.Spread.Sheets.ICopyData;
                cancel: boolean;
                action: GC.Spread.Sheets.ClipboardActionType;
            }


            export interface IClipboardPastedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                cellRange: GC.Spread.Sheets.Range;
                pasteOption: ClipboardPasteOptions;
                pasteData: GC.Spread.Sheets.IPasteData;
                fromSheet?:  GC.Spread.Sheets.Worksheet;
                fromRange?: GC.Spread.Sheets.Range;
                isCutting: boolean;
                objects?: any[];
                shiftCells?: GC.Spread.Sheets.InsertShiftCell;
                action: GC.Spread.Sheets.ClipboardActionType;
            }


            export interface IClipboardPastingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                cellRange: GC.Spread.Sheets.Range;
                pasteOption: ClipboardPasteOptions;
                pasteData: GC.Spread.Sheets.IPasteData;
                fromSheet?:  GC.Spread.Sheets.Worksheet;
                fromRange?: GC.Spread.Sheets.Range;
                isCutting: boolean;
                objects?: any[];
                cancel: boolean;
                shiftCells?: GC.Spread.Sheets.InsertShiftCell;
                action: GC.Spread.Sheets.ClipboardActionType;
            }


            export interface IColorPickerGroup{
                name?: string;
                colors: string[][];
                needScaleColor?: boolean;
            }


            export interface IColorPickerOption{
                colorBlockSize?: number;
                groups?: GC.Spread.Sheets.IColorPickerGroup[];
            }


            export interface IColumn{
                name?: string;
                formatter?: string;
                cellType?: GC.Spread.Sheets.CellTypes.Base;
                width?: number;
                visible?: boolean;
                resizable?: boolean;
                pageBreak?: boolean;
                displayName?: string;
                size?: number | string;
                value?: Function;
                starSize?: string;
            }


            export interface IColumnBindingInfo{
                name: string;
                displayName?: string;
                formatter?: string;
                size?: number | string;
                visible?: boolean;
            }


            export interface IColumnChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                isUndo?: boolean;
            }


            export interface IColumnChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                cancel: boolean;
            }


            export interface IColumnWidthChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                colList: number[];
                header: boolean;
            }


            export interface IColumnWidthChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                colList: any[];
                header: boolean;
                cancel: boolean;
            }


            export interface ICommandMetaData{
                viewModePermissions: GC.Spread.Sheets.Collaboration.PermissionTypes;
            }


            export interface ICommentChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
                propertyName: string;
                row: number;
                col: number;
            }


            export interface ICommentRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
                row: number;
                col: number;
            }


            export interface ICommentRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                comment: Comments.Comment;
                cancel: boolean;
                row: number;
                col: number;
            }


            export interface ICopyData{
                text?: string;
                html?: string
            }


            export interface ICoreDocProps{
                title?: string;
                subject?: string;
                creator?: string;
                keywords?: string;
                description?: string;
                category?: string;
                contentStatus?: string;
                modified?: Date;
                created?: Date;
                lastModifiedBy?: string;
                lastPrinted?: Date;
                revision?: number;
            }


            export interface ICornerFold{
                size?: number;
                position?: GC.Spread.Sheets.CornerPosition;
                color?: string;
            }


            export interface ICustomDocPropsManager{
                all: (props?: GC.Spread.Sheets.ICustomDocumentProperty[]) => GC.Spread.Sheets.ICustomDocumentProperty[];
                get: (propName: string) => GC.Spread.Sheets.ICustomDocumentProperty;
                add: (propName: string, value: GC.Spread.Sheets.CustomDocumentPropertyValueType, isLinkTarget?: boolean) => void;
                remove: (propName: string) => void;
                clear: () => void;
            }


            export interface ICustomDocumentProperty{
                name: string;
                value?: GC.Spread.Sheets.CustomDocumentPropertyValueType;
                linkTarget?: string;
            }


            export interface ICustomSortInfo{
                compareFunction: (value1: any, value2: any) => number;
                ascending: boolean;
                index: number;
            }


            export interface IDecoration{
                cornerFold?: GC.Spread.Sheets.ICornerFold;
                icons?: GC.Spread.Sheets.IIcon[];
            }


            export interface IDeserializationOptions{
                /**
                 * Whether to ignore the style when converting json to the workbook.
                 */
                ignoreStyle?: boolean;
                /**
                 * Whether to ignore the formula when converting json to the workbook.
                 */
                ignoreFormula?: boolean;
                /**
                 * Whether to treat the frozen columns as row headers when converting json to the workbook.
                 */
                frozenColumnsAsRowHeaders?: boolean;
                /**
                 * Whether to treat the frozen rows as column headers when converting json to the workbook.
                 */
                frozenRowsAsColumnHeaders?: boolean;
                /**
                 * Whether to prevent recalculation after loading the json data.
                 */
                doNotRecalculateAfterLoad?: boolean;
                /**
                 * Whether to use the incremental loading or the callbacks of incremental loading when converting json to the workbook.
                 */
                incrementalLoading?: boolean | GC.Spread.Sheets.IIncrementalLoadingOptions;
            }


            export interface IDirtyCellInfo{
                row: number;
                col: number;
                newValue: any;
                oldValue: any;
            }


            export interface IDocProps{
                coreDocProps?:  GC.Spread.Sheets.ICoreDocProps;
                appDocProps?:  GC.Spread.Sheets.IAppDocProps;
                customDocPropsManager:  GC.Spread.Sheets.ICustomDocPropsManager;
            }


            export interface IDragDropBlockCompletedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fromRow: number;
                fromCol: number;
                toRow: number;
                toCol: number;
                rowCount: number;
                colCount: number;
                copy: boolean;
                insert: boolean;
                copyOption: CopyToOptions;
            }


            export interface IDragDropBlockEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fromRow: number;
                fromCol: number;
                toRow: number;
                toCol: number;
                rowCount: number;
                colCount: number;
                copy: boolean;
                insert: boolean;
                copyOption: CopyToOptions;
                cancel: boolean;
            }


            export interface IDragFillBlockCompletedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fillRange: GC.Spread.Sheets.Range;
                autoFillType: GC.Spread.Sheets.Fill.AutoFillType;
                fillDirection: GC.Spread.Sheets.Fill.FillDirection;
            }


            export interface IDragFillBlockEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                fillRange: GC.Spread.Sheets.Range;
                autoFillType: GC.Spread.Sheets.Fill.AutoFillType;
                fillDirection: GC.Spread.Sheets.Fill.FillDirection;
                cancel: boolean;
            }


            export interface IDropdown{
                type: GC.Spread.Sheets.DropDownType;
                option?: ICalendarOption | IColorPickerOption | IListOption | IMonthPickerOption | ISliderOption | ITimePickerOption | IWorkFlowOption | IMultiColumnOption;
                submitCommand?: string | ((sheet: GC.Spread.Sheets.Worksheet, value:any, option: any) => void);
            }


            export interface IEditChangeEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editingText: any;
            }


            export interface IEditEndedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editingText: Object;
            }


            export interface IEditEndingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                editor: Object;
                editingText: Object;
                cancel: boolean;
            }


            export interface IEditorStatusChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldStatus: EditorStatus;
                newStatus: EditorStatus;
            }


            export interface IEditStartingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                cancel: boolean;
            }


            export interface IEnterCellEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
            }


            export interface IExternalReference{
                name: string;
                filePath?: string;
                items?: GC.Spread.Sheets.IExternalReferenceDetail[];
            }


            export interface IExternalReferenceDetail{
                target: GC.Spread.Sheets.ICellsInfo;
                source: GC.Spread.Sheets.ICellsInfo | GC.Spread.Sheets.ICellsInfo[];
            }


            export interface IFilterButtonHitInfo{
                rowFilter: GC.Spread.Sheets.Filter.RowFilterBase;
                row?: number;
                col?: number;
                x?: number;
                y?: number;
                width?: number;
                height?: number;
                sheetArea?: GC.Spread.Sheets.SheetArea;
            }


            export interface IFloatingObjectChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                propertyName: string;
            }


            export interface IFloatingObjectRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
            }


            export interface IFloatingObjectRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
                cancel: boolean;
            }


            export interface IFloatingObjectSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                floatingObject: FloatingObjects.FloatingObject;
            }


            export interface IFontColorSortInfo{
                order: "top" | "bottom";
                fontColor?: string;
                index: number;
            }


            export interface IFormulaInfo{
                hasFormula?: boolean;
                baseRange?: GC.Spread.Sheets.Range;
                isArrayFormula?: boolean;
                formula?: string;
                formulaWithCulture?: string;
                isDynamicArray?: boolean;
            }


            export interface IFormulaRangeHitInfo{
                /**
                 * param range info
                 */
                paramRange: GC.Spread.Sheets.IParamRange;
                inTopLeft?: boolean;
                inTopRight?: boolean;
                inBottomLeft?: boolean;
                inBottomRight?: boolean;
                inBorder?: boolean;
            }


            export interface IGradientFill{
                degree?: number;
                stops: { color: string; position: number }[];
            }


            export interface IGradientPathFill{
                type: string;
                left?: number;
                right?: number;
                top?: number;
                bottom?: number;
                stops: { color: string; position: number }[];
            }


            export interface IHitTestCellTypeHitInfo{
                x?: number;
                y?: number;
                row?: number;
                col?: number;
                cellRect?: GC.Spread.Sheets.Rect;
                sheetArea?: GC.Spread.Sheets.SheetArea;
                sheet?: GC.Spread.Sheets.Worksheet;
                isReservedLocation?: boolean;
            }


            export interface IHitTestCommentHitInfo{
                x?: number;
                y?: number;
                comment?: GC.Spread.Sheets.Comments.Comment;
                area?: string;
            }


            export interface IHitTestDragInfo{
                action?: string;
                side?: string;
                outside?: boolean;
            }


            export interface IHitTestFloatingObjectHitInfo{
                x?: number;
                y?: number;
                floatingObject?: GC.Spread.Sheets.FloatingObjects.FloatingObject;
            }


            export interface IHitTestFooterCornerInfo{
                element?: string,
            }


            export interface IHitTestInformation{
                x?: number;
                y?: number;
                rowViewportIndex?: number;
                colViewportIndex?: number;
                row?: number;
                col?: number;
                hitTestType?: GC.Spread.Sheets.SheetArea;
                resizeInfo?: GC.Spread.Sheets.IHitTestResizeInfo;
                outlineHitInfo?: GC.Spread.Sheets.IHitTestOutlineHitInfo;
                filterButtonHitInfo?: GC.Spread.Sheets.IFilterButtonHitInfo;
                dragInfo?: GC.Spread.Sheets.IHitTestDragInfo;
                cellTypeHitInfo?: GC.Spread.Sheets.IHitTestCellTypeHitInfo;
                floatingObjectHitInfo?: GC.Spread.Sheets.IHitTestFloatingObjectHitInfo;
                shapeHitInfo?: GC.Spread.Sheets.IHitTestShapeHitInfo;
                formulaRangeHitInfo?: GC.Spread.Sheets.IFormulaRangeHitInfo;
                commentHitInfo?: GC.Spread.Sheets.IHitTestCommentHitInfo;
                pivotTableInfo?: any;
                pivotTableCellInfo?: any;
            }


            export interface IHitTestOutlineHitInfo{
                what?: string;
                info?: GC.Spread.Sheets.IOutlineHitInfo;
            }


            export interface IHitTestResizeInfo{
                action?: string;
                index?: number;
                sheetArea?: GC.Spread.Sheets.SheetArea;
                startY?: number;
                movingY?: number;
                startX?: number;
                movingX?: number;
            }


            export interface IHitTestScrollBarInfo{
                element?: string,
            }


            export interface IHitTestShapeHitInfo{
                x?: number;
                y?: number;
                type?: GC.Spread.Sheets.Shapes.HitTestType;
                shape?: GC.Spread.Sheets.Shapes.ShapeBase;
                target?: GC.Spread.Sheets.Shapes.ShapeBase;
                connectionPointIndex?: number;
                lineEnd?: GC.Spread.Sheets.Shapes.HitTestLineEnd;
                resizeHandle?: GC.Spread.Sheets.Shapes.IHitTestResizeHandle;
                adjustmentHandle?: GC.Spread.Sheets.Shapes.IHitTestAdjustmentHandle;
            }


            export interface IHyperlink{
                url: string;
                tooltip?: string;
                linkColor?: string;
                visitedLinkColor?: string;
                target?: number;
                drawUnderline?: boolean;
                command?: string | ((sheet: GC.Spread.Sheets.Worksheet, row: number, col: number, sheetArea?: GC.Spread.Sheets.SheetArea) => void);
            }


            export interface IIcon{
                src: string;
                position?: GC.Spread.Sheets.IconPosition;
                width?: number;
                height?: number;
            }


            export interface IIHitTestTabStripInfo{
                navButton?: string;
                sheetTab?: GC.Spread.Sheets.SheetTabInfo;
                resize?: boolean;
                blank?: boolean;
            }


            export interface IImageSize{
                /**
                 * Specific the image's width, default value is 16, unit is px.
                 */
                width: number;
                /**
                 * Specific the image's height, default value is 16, unit is px.
                 */
                height: number;
            }


            export interface IIncrementalLoadingOptions{
                /**
                 * The callback when of the incremental loading finished.
                 */
                loaded?: () => void;
                /**
                 * The callback when of the incremental loading progress.
                 */
                loading?: (progress: number, args: { sheet: GC.Spread.Sheets.Worksheet }) => void;
            }


            export interface IInvalidOperationEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                invalidType: InvalidOperationType;
                message: string;
            }


            export interface ILabelOptions{
                /**
                 * The cell label position.
                 */
                alignment?: GC.Spread.Sheets.LabelAlignment;
                /**
                 * The cell label visibility.
                 */
                visibility?: GC.Spread.Sheets.LabelVisibility;
                /**
                 * The cell label font.
                 */
                font?: string;
                /**
                 * The cell label foreColor.
                 */
                foreColor?: string;
                /**
                 * The cell label margin.
                 */
                margin?: string;
            }


            export interface ILeaveCellEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                cancel: boolean;
            }


            export interface ILeftColumnChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldLeftCol: number;
                newLeftCol: number;
                oldOffset: number;
                newOffset: number;
            }


            export interface IListItem{
                text?: string;
                value?: string;
                icon?: string;
            }


            export interface IListLayout{
                direction?: GC.Spread.Sheets.LayoutDirection;
                displayAs?: GC.Spread.Sheets.LayoutDisplayAs;
                collapsible?: boolean;
            }


            export interface IListOption{
                text?: string;
                layout?: GC.Spread.Sheets.IListLayout;
                multiSelect?: boolean;
                valueType?: GC.Spread.Sheets.DropdownListValue;
                onItemSelected?: (e: MouseEvent) => string;
                /**
                 * a function returns a DOM element
                 */
                items:  GC.Spread.Sheets.IListItem[] | GC.Spread.Sheets.IListOption[] | (() => HTMLElement);
            }


            export interface IMaskType{
                pattern: string;
                excludeLiteral?: boolean;
                excludePlaceholder?: boolean;
                placeholder?: string;
            }


            export interface IMonthPickerOption{
                startYear?: number;
                stopYear?: number;
                height?: number;
            }


            export interface IMultiColumnOption extends GC.Spread.Sheets.IBaseSpreadOption{
                width?: number;
                height?: number;
                dataSource: string | any[];
                bindingInfos?: GC.Spread.Sheets.IColumnBindingInfo[];
                valuePath?: string;
            }


            export interface IOutlineColumnCheckStatusChanged{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number,
                col: number,
                status: boolean
            }


            export interface IOutlineHitInfo{
                index?: number;
                isExpanded?: boolean;
                level?: number;
                lineDirection?: GC.Spread.Sheets.Outlines.OutlineDirection;
                paintLine?: boolean;
            }


            export interface IParameter{
                [parameterName: string]: string | number | boolean | Date;
            }


            export interface IParamRange{
                /**
                 * range text offset in formulatextbox's value
                 */
                textOffset: number;
                /**
                 * range text
                 */
                text: string;
                /**
                 * index in all ranges
                 */
                index: number;
            }


            export interface IPasteData{
                text?: string;
                html?: string;
                image?: string;
            }


            export interface IPatternFill{
                type: GC.Spread.Sheets.PatternType;
                patternColor: string;
                backgroundColor?: string;
            }


            export interface IPictureChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                picture: FloatingObjects.Picture;
                propertyName: string;
            }


            export interface IPictureSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                picture: FloatingObjects.Picture;
            }


            export interface IPivotTableChangedEventArgs{
                pivotTableName: string;
                type: "filter" | "sort" | "collapse" | "fieldChanged" | "summarizedValueBy" | "showValueAs" | "dataPositionChanged" | "viewChanged" | "grandTotal" | "showNoData" | "group";
                fieldName: string;
                value?: boolean;
                sortType?: number;
                area?: GC.Spread.Pivot.PivotTableFieldType;
                index?: number;
            }


            export interface IProtectionOptions{
                /**
                 * True or undefined if the user can select locked cells.
                 */
                allowSelectLockedCells?: boolean;
                /**
                 * True or undefined if the user can select unlocked cells.
                 */
                allowSelectUnlockedCells?: boolean;
                /**
                 * True if the user can sort ranges.
                 */
                allowSort?: boolean;
                /**
                 * True if the user can filter ranges.
                 */
                allowFilter?: boolean;
                /**
                 * True if the user can edit floating objects.
                 */
                allowEditObjects?: boolean;
                /**
                 * True if the user can resize rows.
                 */
                allowResizeRows?: boolean;
                /**
                 * True if the user can resize columns.
                 */
                allowResizeColumns?: boolean;
                /**
                 * True if the user can drag to insert rows.
                 */
                allowDragInsertRows?: boolean;
                /**
                 * True if the user can drag to insert columns.
                 */
                allowDragInsertColumns?: boolean;
                /**
                 * True if the user can insert rows.
                 */
                allowInsertRows?: boolean;
                /**
                 * True if the user can insert columns.
                 */
                allowInsertColumns?: boolean;
                /**
                 * True if the user can delete rows.
                 */
                allowDeleteRows?: boolean;
                /**
                 * True if the user can delete columns.
                 */
                allowDeleteColumns?: boolean;
                /**
                 * True if the user can expand or collapse the column groups.
                 */
                allowOutlineColumns?: boolean;
                /**
                 * True if the user can expand or collapse the row groups.
                 */
                allowOutlineRows?: boolean;
                /**
                 * True if the user can use PivotTables.
                 */
                allowUsePivotTable?: boolean;
            }


            export interface IRange{
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
            }


            export interface IRangeChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
                changedCells: ICellPosition[];
                action: RangeChangedAction;
                tableNames?: string[];
                isUndo?: boolean;
            }


            export interface IRangeFilterClearedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
            }


            export interface IRangeFilterClearingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
            }


            export interface IRangeFilteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                filterValues: any[];
            }


            export interface IRangeFilteringEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                filterValues: any[];
                conditionInfo : Object;
            }


            export interface IRangeGroupStateChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                isRowGroup: boolean;
                index: number;
                level: number;
            }


            export interface IRangeGroupStateChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                isRowGroup: boolean;
                index: number;
                level: number;
                cancel: boolean;
            }


            export interface IRangeSortedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                ascending: boolean;
            }


            export interface IRangeSortingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                col: number;
                ascending: boolean;
                compareFunction?: ((value1: any, value2: any) => number);
                groupSort: GC.Spread.Sheets.GroupSort;
            }


            export interface IRowChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                isUndo?: boolean;
            }


            export interface IRowChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                sheetArea: SheetArea;
                propertyName: string;
                oldValue: any;
                newValue: any;
                count?: number;
                cancel: boolean;
            }


            export interface IRowHeightChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                rowList: number[];
                header: boolean;
            }


            export interface IRowHeightChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                rowList: any[];
                header: boolean;
                cancel: boolean;
            }


            export interface ISelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldSelections: Sheets.Range[];
                newSelections: Sheets.Range[];
            }


            export interface ISelectionChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldSelections: GC.Spread.Sheets.Range[];
                newSelections: GC.Spread.Sheets.Range[];
            }


            export interface ISerializationOption{
                /**
                 * Whether to include the binding source when converting the workbook to json.
                 */
                includeBindingSource?: boolean;
                /**
                 * Whether to ignore the style when converting the workbook to json.
                 */
                ignoreStyle?: boolean;
                /**
                 * Whether to ignore the formula when converting the workbook to json.
                 */
                ignoreFormula?: boolean;
                /**
                 * Whether to apply the format string to exporting value when converting the workbook to json, default false.
                 */
                saveAsView?: boolean;
                /**
                 * Whether to treat the row headers as frozen columns when converting the workbook to json.
                 */
                rowHeadersAsFrozenColumns?: boolean;
                /**
                 * Whether to treat the column headers as frozen rows when converting the workbook to json.
                 */
                columnHeadersAsFrozenRows?: boolean;
                /**
                 * Whether to include the automatically merged cells when converting the workbook to json.
                 */
                includeAutoMergedCells?: boolean;
                /**
                 * Whether to save the r1c1 formula in the file.
                 */
                saveR1C1Formula?: boolean;
            }


            export interface ISetBorderOptions{
                all?: boolean;
                left?: boolean;
                top?: boolean;
                right?: boolean;
                bottom?: boolean;
                outline?: boolean;
                inside?: boolean;
                innerHorizontal?: boolean;
                innerVertical?: boolean;
                diagonalUp?:boolean;
                diagonalDown?:boolean;
            }


            export interface IShapeChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: GC.Spread.Sheets.Shapes.Shape;
                propertyName: string;
            }


            export interface IShapeRemovedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
            }


            export interface IShapeRemovingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
                cancel: boolean;
            }


            export interface IShapeSelectionChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                shape: Shapes.Shape;
            }


            export interface ISheetChangedEventArgs{
                sheetName: string;
                propertyName: string;
                sheetIndex: number;
                newValue?: boolean;
                oldValue?: boolean;
            }


            export interface ISheetChangingEventArgs{
                sheetName: string;
                propertyName: string;
                sheetIndex: number;
                cancel: boolean;
                newValue?: boolean;
                oldValue?: boolean;
            }


            export interface ISheetDefaultOption{
                rowHeight?: number;
                colHeaderRowHeight?: number;
                colWidth?: number;
                rowHeaderColWidth?: number;
            }


            export interface ISheetNameChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                oldValue: string;
                newValue: string;
            }


            export interface ISheetNameChangingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                oldValue: string;
                newValue: string;
                cancel: boolean;
            }


            export interface ISheetNameTabStyleMap{
                [sheetName: string]: GC.Spread.Sheets.SheetTabStyles;
            }


            export interface ISheetTabClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetTabIndex: number;
            }


            export interface ISheetTabDoubleClickEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sheetTabIndex: number;
            }


            export interface ISheetTabStyle{
                backColor?: string;
                foreColor?: string;
                font?: string;
                icons?: GC.Spread.Sheets.IIcon[];
            }


            export interface ISlicerChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                slicer: GC.Spread.Sheets.Slicers.ISlicer;
                propertyName: string;
            }


            export interface ISliderOption{
                max?: number;
                min?: number;
                step?: number;
                direction? :GC.Spread.Sheets.LayoutDirection;
                tooltipVisible?: boolean;
                showNumberRange?: boolean;
                scaleVisible ?: boolean;
                marks?: number[];
                width?: number;
                height?: number;
                formatString?: string;
            }


            export interface ISortOptions{
                groupSort?: GC.Spread.Sheets.GroupSort;
                ignoreHidden?: boolean;
            }


            export interface ISortState{
                row: number;
                col: number;
                rowCount: number;
                colCount: number;
                byRow?: boolean;
                sortConditions?: Array<GC.Spread.Sheets.IValueSortInfo | GC.Spread.Sheets.ICellColorSortInfo | GC.Spread.Sheets.IFontColorSortInfo | GC.Spread.Sheets.ICustomSortInfo>;
            }


            export interface ISparklineChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                sparkline: Sparklines.Sparkline;
            }


            export interface ITableFilterClearedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
            }


            export interface ITableFilterClearingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
            }


            export interface ITableFilteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
                filterValues: any[];
            }


            export interface ITableFilteringEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                table: GC.Spread.Sheets.Tables.Table;
                tableCol: number;
                filterValues: any[];
                conditionInfo: Object;
            }


            export interface ITimePickerOption{
                min?: GC.Spread.Sheets.ITimePickerValue;
                max?: GC.Spread.Sheets.ITimePickerValue;
                step?: GC.Spread.Sheets.ITimePickerValue;
                formatString?: string;
                height?: number;
            }


            export interface ITimePickerValue{
                hour?: number;
                minute?: number;
                second?: number;
            }


            export interface ITopRowChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldTopRow: number;
                newTopRow: number;
                oldOffset: number;
                newOffset: number;
            }


            export interface ITouchToolStripOpeningEventArgs{
                x: number;
                y: number;
                handled: boolean;
            }


            export interface IUserFormulaEnteredEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                formula: string;
                isCircularReference: boolean;
            }


            export interface IUserZoomingEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                oldZoomFactor: number;
                newZoomFactor: number;
            }


            export interface IValidationErrorEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                validator: DataValidation.DefaultDataValidator;
                validationResult: DataValidation.DataValidationResult;
            }


            export interface IValueChangedEventArgs{
                sheet: GC.Spread.Sheets.Worksheet;
                sheetName: string;
                row: number;
                col: number;
                oldValue: any;
                newValue: any;
            }


            export interface IValueSortInfo{
                ascending: boolean;
                index: number;
            }


            export interface IWorkBookDefaultOptions{
                /**
                 * The number of sheets. The default value is 1.
                 */
                sheetCount?: number;
                /**
                 * The tab strip font.
                 */
                font?: string;
                /**
                 * Whether to allow the user to drag merge cells. The default value is false.
                 */
                allowUserDragMerge?: boolean;
                /**
                 * Specifies how to allow drag header to move. The default value is GC.Spread.Sheets.AllowDragHeaderToMove.none.
                 */
                allowDragHeaderToMove?: GC.Spread.Sheets.AllowDragHeaderToMove;
                /**
                 * Whether to allow the user to drag and drop range data. The default value is true.
                 */
                allowUserDragDrop?: boolean;
                /**
                 * Whether to allow the user to drag fill a range. The default value is true.
                 */
                allowUserDragFill?: boolean;
                /**
                 * Whether to zoom the display by scrolling the mouse wheel while pressing the Ctrl key. The default value is true.
                 */
                allowUserZoom?: boolean;
                /**
                 * Whether to allow the user to resize columns and rows. The default value is true.
                 */
                allowUserResize?: boolean;
                /**
                 * Whether to allow the user to undo edits. The default value is true.
                 */
                allowUndo?: boolean;
                /**
                 * Whether the user can reorder the sheets in the Spread component. The default value is true.
                 */
                allowSheetReorder?: boolean;
                /**
                 * Whether to allow the user to open the built-in context menu. The default value is true.
                 */
                allowContextMenu?: boolean;
                /**
                 * Whether to allow the user to can use deselect in selection. The default value is true.
                 */
                allowUserDeselect?: boolean;
                /**
                 * The default fill type. The default value is GC.Spread.Sheets.Fill.AutoFillType.auto.
                 */
                defaultDragFillType?: GC.Spread.Sheets.Fill.AutoFillType;
                /**
                 * Whether to display the drag fill dialog. The default value is true.
                 */
                showDragFillSmartTag?: boolean;
                /**
                 * Whether to display the horizontal scroll bar. The default value is true.
                 */
                showHorizontalScrollbar?: boolean;
                /**
                 * Whether to display the vertical scroll bar. The default value is true.
                 */
                showVerticalScrollbar?: boolean;
                /**
                 * Whether the displayed scroll bars are based on the entire number of columns and rows in the sheet. The default value is true.
                 */
                scrollbarShowMax?: boolean;
                /**
                 * Whether the scroll bar aligns with the last row and column of the active sheet. The default value is false.
                 */
                scrollbarMaxAlign?: boolean;
                /**
                 * Whether to display the sheet tab strip. The default value is true.
                 */
                tabStripVisible?: boolean;
                /**
                 * The width of the tab strip expressed as a percentage of the overall horizontal scroll bar width. The default value is 0.5.
                 */
                tabStripRatio?: number;
                /**
                 * Whether to allow the user to edit the sheet tab strip. The default value is true.
                 */
                tabEditable?: boolean;
                /**
                 * Whether the spreadsheet displays the special tab to let users insert new sheets. The default value is true.
                 */
                newTabVisible?: boolean;
                /**
                 * Whether the spreadsheet shows a special tab to allow the user to open the dialog to display all sheets. The default value is GC.Spread.Sheets.AllSheetsListVisibility.auto.
                 */
                allSheetsListVisible?: GC.Spread.Sheets.AllSheetsListVisibility;
                /**
                 * Whether to display the sheet tab navigation. The default value is true.
                 */
                tabNavigationVisible?: boolean;
                /**
                 * Whether to display an indicator when copying or cutting the selected item. The default value is true.
                 */
                cutCopyIndicatorVisible?: boolean;
                /**
                 * The border color for the indicator displayed when the user cuts or copies the selection. The default value is "#217346".
                 */
                cutCopyIndicatorBorderColor?: string;
                /**
                 * A color string used to represent the background color of the Spread component, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on. The default value is "white".
                 */
                backColor?: string;
                /**
                 * The background image of the Spread component. It can be a URL string or base64 string.
                 */
                backgroundImage?: null | string;
                /**
                 * The background image layout for the Spread component. The default value is GC.Spread.Sheets.ImageLayout.stretch.
                 */
                backgroundImageLayout?: GC.Spread.Sheets.ImageLayout;
                /**
                 * A color string used to represent the background color of the gray area , such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on. The default value is null which means using the theme default color.
                 */
                grayAreaBackColor?: null | string;
                /**
                 * How to display the resize tip. The default value is GC.Spread.Sheets.ShowResizeTip.none.
                 */
                showResizeTip?: GC.Spread.Sheets.ShowResizeTip;
                /**
                 * Whether to display the drag-drop tip. The default value is true.
                 */
                showDragDropTip?: boolean;
                /**
                 * Whether to display the drag-fill tip. The default value is true.
                 */
                showDragFillTip?: boolean;
                /**
                 * How to display the scroll tip. The default value is GC.Spread.Sheets.ShowScrollTip.none.
                 */
                showScrollTip?: GC.Spread.Sheets.ShowScrollTip;
                /**
                 * Whether the scroll bar ignores hidden rows or columns. The default value is false.
                 */
                scrollIgnoreHidden?: boolean;
                /**
                 * Whether to highlight invalid data. The default value is false.
                 */
                highlightInvalidData?: boolean;
                /**
                 * Whether to use touch layout to present the Spread component. The default value is false.
                 */
                useTouchLayout?: boolean;
                /**
                 * Whether to display the selection highlighting when the Spread component does not have focus. The default value is false.
                 */
                hideSelection?: boolean;
                /**
                 * The drawing policy when the row or column is resized to zero. The default value is GC.Spread.Sheets.ResizeZeroIndicator.enhanced.
                 */
                resizeZeroIndicator?: GC.Spread.Sheets.ResizeZeroIndicator;
                /**
                 * Whether the user can edit formulas in a cell in the spreadsheet. The default value is true.
                 */
                allowUserEditFormula?: boolean;
                /**
                 * Whether to enable the formula text box in the spreadsheet. The default value is true.
                 */
                enableFormulaTextbox?: boolean;
                /**
                 * Whether content will be formatted to fit in cells or in cells and headers. The default value is GC.Spread.Sheets.AutoFitType.cell.
                 */
                autoFitType?: GC.Spread.Sheets.AutoFitType;
                /**
                 * The style for cell and range references in cell formulas on the workbook. The default value is GC.Spread.Sheets.ReferenceStyle.a1.
                 */
                referenceStyle?: GC.Spread.Sheets.ReferenceStyle;
                /**
                 * Whether to enable dynamic array. The default value is false.
                 */
                allowDynamicArray?: boolean;
                /**
                 * Whether to enable the iterative calculation. The default value is true.
                 */
                iterativeCalculation?: boolean;
                /**
                 * The Maximum Iterations when iterative calculation. The default value is 1000.
                 */
                iterativeCalculationMaximumIterations?: number;
                /**
                 * The Maximum Change when iterative calculation. The default value is 0.01.
                 */
                iterativeCalculationMaximumChange?: number;
                /**
                 * Whether to calculate formulas only when they are demanded. The default value is false.
                 */
                calcOnDemand?: boolean;
                /**
                 * Whether to incremental calculate formulas without blocking UI. The default value is false.
                 */
                incrementalCalculation?: boolean;
                /**
                 * Whether the user can copy style from Spread Sheets then paste to Excel, or copy style from Excel then paste to Spread Sheets. The default value is true.
                 */
                allowCopyPasteExcelStyle?: boolean;
                /**
                 * Whether extend paste range if the paste range is not enough for pasting. The default value is false.
                 */
                allowExtendPasteRange?: boolean;
                /**
                 * Which headers are included when data is copied to or pasted. The default value is GC.Spread.Sheets.CopyPasteHeaderOptions.allHeaders.
                 */
                copyPasteHeaderOptions?: GC.Spread.Sheets.CopyPasteHeaderOptions;
                /**
                 * The recalculation behavior of the workbook. The default value is GC.Spread.Sheets.CalculationMode.auto.
                 */
                calculationMode?: GC.Spread.Sheets.CalculationMode;
                /**
                 * Whether to enable the precision scrolling by pixel. The default value is false.
                 */
                scrollByPixel?: boolean;
                /**
                 * Decides scrolling by that number of pixels at a time when scrollByPixel is true. The final scrolling pixels are the result of scrolling delta multiply scrollPixel. For example, the scrolling delta is 3, and the scrollPixel is 5, the final scrolling pixels are 15. The default value is 5.
                 */
                scrollPixel?: number;
                /**
                 * Whether to enable the accessibility support in the spreadsheet. The default value is false.
                 */
                enableAccessibility?: boolean;
                /**
                 * Whether to enable auto creating hyperlink in the spreadsheet. The default value is true.
                 */
                allowAutoCreateHyperlink?: boolean;
                /**
                 * Specifies the way to resize column. The default value is GC.Spread.Sheets.ResizeMode.normal.
                 */
                columnResizeMode?: GC.Spread.Sheets.ResizeMode;
                /**
                 * Specifies the way to resize row. The default value is GC.Spread.Sheets.ResizeMode.normal.
                 */
                rowResizeMode?: GC.Spread.Sheets.ResizeMode;
                /**
                 * The list for user to customize drag fill, prioritize matching this list in each fill. Each array item is type of string array. For example, [["Custom1", "C1"], ["Custom2", "C2", "C3"]]. When user drag fill "Custom1", the filled value is "C1"; when user drag fill "Custom2", the filled values are "C2", "C3".
                 */
                customList?: string[][];
                /**
                 * The scrollbar appearance, contains skin and mobile two enums. The default value is GC.Spread.Sheets.ScrollbarAppearance.skin.
                 */
                scrollbarAppearance?: GC.Spread.Sheets.ScrollbarAppearance;
                /**
                 * Whether paste skip invisible range. The default value is false.
                 */
                pasteSkipInvisibleRange?: boolean;
                /**
                 * Whether allow auto extend filter range like excel. The default value is false.
                 */
                allowAutoExtendFilterRange?: boolean;
                /**
                 * Whether allow input invalid formula string. The default value is false.
                 */
                allowInvalidFormula?: boolean;
                /**
                 * Whether automatically generate the format when formula inputs. The default value is true.
                 */
                formulaFormatHint?: boolean;
                /**
                 * Whether automatically generate the getPivotData formula or cell reference when choose pivot table data area. The default value is GC.Spread.Pivot.PivotAreaReference.getPivotData.
                 */
                pivotAreaReference?: GC.Spread.Pivot.PivotAreaReference;
                /**
                 * All default state styles for sheet tabs.
                 */
                defaultSheetTabStyles?: GC.Spread.Sheets.SheetTabStyles;
                /**
                 * All built-in file icons.
                 */
                builtInFileIcons?: GC.Spread.Sheets.IBuiltInFileIcons;
            }


            export interface IWorkbookHitTestInformation{
                x?: number;
                y?: number;
                worksheetHitInfo: GC.Spread.Sheets.IHitTestInformation;
                tabStripHitInfo: GC.Spread.Sheets.IIHitTestTabStripInfo;
                horizontalScrollBarHitInfo: GC.Spread.Sheets.IHitTestScrollBarInfo;
                verticalScrollBarHitInfo: GC.Spread.Sheets.IHitTestScrollBarInfo;
                footerCornerHitInfo: GC.Spread.Sheets.IHitTestFooterCornerInfo;
            }


            export interface IWorkbookOptions{
                /**
                 * Whether to allow the user to drag merge cells. The default value is false.
                 */
                allowUserDragMerge: boolean;
                /**
                 * Whether to allow the user to drag and drop range data. The default value is true.
                 */
                allowUserDragDrop: boolean;
                /**
                 * Specifies how to allow drag header to move. The default value is GC.Spread.Sheets.AllowDragHeaderToMove.none.
                 */
                allowDragHeaderToMove: GC.Spread.Sheets.AllowDragHeaderToMove;
                /**
                 * Whether to allow the user to drag fill a range. The default value is true.
                 */
                allowUserDragFill: boolean;
                /**
                 * Whether to zoom the display by scrolling the mouse wheel while pressing the Ctrl key. The default value is true.
                 */
                allowUserZoom: boolean;
                /**
                 * Whether to allow the user to resize columns and rows. The default value is true.
                 */
                allowUserResize: boolean;
                /**
                 * Whether to allow the user to undo edits. The default value is true.
                 */
                allowUndo: boolean;
                /**
                 * Whether the user can reorder the sheets in the Spread component. The default value is true.
                 */
                allowSheetReorder: boolean;
                /**
                 * Whether to allow the user to open the built-in context menu. The default value is true.
                 */
                allowContextMenu: boolean;
                /**
                 * Whether to allow the user to can use deselect in selection. The default value is true.
                 */
                allowUserDeselect: boolean;
                /**
                 * The default fill type. The default value is GC.Spread.Sheets.Fill.AutoFillType.auto.
                 */
                defaultDragFillType: GC.Spread.Sheets.Fill.AutoFillType;
                /**
                 * Whether to display the drag fill dialog. The default value is true.
                 */
                showDragFillSmartTag: boolean;
                /**
                 * Whether to display the horizontal scroll bar. The default value is true.
                 */
                showHorizontalScrollbar: boolean;
                /**
                 * Whether to display the vertical scroll bar. The default value is true.
                 */
                showVerticalScrollbar: boolean;
                /**
                 * Whether the displayed scroll bars are based on the entire number of columns and rows in the sheet. The default value is true.
                 */
                scrollbarShowMax: boolean;
                /**
                 * Whether the scroll bar aligns with the last row and column of the active sheet. The default value is false.
                 */
                scrollbarMaxAlign: boolean;
                /**
                 * Whether to display the sheet tab strip. The default value is true.
                 */
                tabStripVisible: boolean;
                /**
                 * The width of the tab strip expressed as a percentage of the overall horizontal scroll bar width. The default value is 0.5.
                 */
                tabStripRatio: number;
                /**
                 * The width of the tab strip when it is at the left or right position. The default value and minimum is 80.
                 */
                tabStripWidth: number;
                /**
                 * Whether to allow the user to edit the sheet tab strip. The default value is true.
                 */
                tabEditable: boolean;
                /**
                 * The position of tab strip. The default value is GC.Spread.Sheets.TabStripPosition.bottom.
                 */
                tabStripPosition: GC.Spread.Sheets.TabStripPosition;
                /**
                 * Whether the spreadsheet displays the special tab to let users insert new sheets. The default value is true.
                 */
                newTabVisible: boolean;
                /**
                 * Whether the spreadsheet shows a special tab to allow the user to open the dialog to display all sheets. The default value is GC.Spread.Sheets.AllSheetsListVisibility.auto.
                 */
                allSheetsListVisible: GC.Spread.Sheets.AllSheetsListVisibility;
                /**
                 * Whether to display the sheet tab navigation. The default value is true.
                 */
                tabNavigationVisible: boolean;
                /**
                 * Whether to display an indicator when copying or cutting the selected item. The default value is true.
                 */
                cutCopyIndicatorVisible: boolean;
                /**
                 * The border color for the indicator displayed when the user cuts or copies the selection. The default value is "#217346".
                 */
                cutCopyIndicatorBorderColor: string;
                /**
                 * A color string used to represent the background color of the Spread component, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on. The default value is "white".
                 */
                backColor: string;
                /**
                 * The background image of the Spread component. The default value is null.
                 */
                backgroundImage: string;
                /**
                 * The background image layout for the Spread component. The default value is GC.Spread.Sheets.ImageLayout.stretch.
                 */
                backgroundImageLayout: GC.Spread.Sheets.ImageLayout;
                /**
                 * A color string used to represent the background color of the gray area , such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on. The default value is null.
                 */
                grayAreaBackColor: string;
                /**
                 * How to display the resize tip. The default value is GC.Spread.Sheets.ShowResizeTip.none.
                 */
                showResizeTip: GC.Spread.Sheets.ShowResizeTip;
                /**
                 * Whether to display the drag-drop tip. The default value is true.
                 */
                showDragDropTip: boolean;
                /**
                 * Whether to display the drag-fill tip. The default value is true.
                 */
                showDragFillTip: boolean;
                /**
                 * How to display the scroll tip. The default value is GC.Spread.Sheets.ShowScrollTip.none.
                 */
                showScrollTip: GC.Spread.Sheets.ShowScrollTip;
                /**
                 * Whether the scroll bar ignores hidden rows or columns. The default value is false.
                 */
                scrollIgnoreHidden: boolean;
                /**
                 * Whether to highlight invalid data. The default value is false.
                 */
                highlightInvalidData: boolean;
                /**
                 * Whether to use touch layout to present the Spread component. The default value is false.
                 */
                useTouchLayout: boolean;
                /**
                 * Whether to display the selection highlighting when the Spread component does not have focus. The default value is false.
                 */
                hideSelection: boolean;
                /**
                 * The drawing policy when the row or column is resized to zero. The default value is GC.Spread.Sheets.ResizeZeroIndicator.enhanced.
                 */
                resizeZeroIndicator: GC.Spread.Sheets.ResizeZeroIndicator;
                /**
                 * Whether the user can edit formulas in a cell in the spreadsheet. The default value is true.
                 */
                allowUserEditFormula: boolean;
                /**
                 * Whether to enable the formula text box in the spreadsheet. The default value is true.
                 */
                enableFormulaTextbox: boolean;
                /**
                 * Whether content will be formatted to fit in cells or in cells and headers. The default value is GC.Spread.Sheets.AutoFitType.cell.
                 */
                autoFitType: GC.Spread.Sheets.AutoFitType;
                /**
                 * The style for cell and range references in cell formulas on the workbook. The default value is GC.Spread.Sheets.ReferenceStyle.a1.
                 */
                referenceStyle: GC.Spread.Sheets.ReferenceStyle;
                /**
                 * Whether to enable dynamic array. The default value is false.
                 */
                allowDynamicArray: boolean;
                /**
                 * Whether to enable the iterative calculation. The default value is true.
                 */
                iterativeCalculation: boolean;
                /**
                 * The Maximum Iterations when iterative calculation. The default value is 1000.
                 */
                iterativeCalculationMaximumIterations: number;
                /**
                 * The Maximum Change when iterative calculation. The default value is 0.01.
                 */
                iterativeCalculationMaximumChange: number;
                /**
                 * Whether to calculate formulas only when they are demanded. The default value is false.
                 */
                calcOnDemand: boolean;
                /**
                 * Whether to incremental calculate formulas without blocking UI. The default value is false.
                 */
                incrementalCalculation: boolean;
                /**
                 * Whether to calculate functions with dynamic reference. The default value is true.
                 */
                dynamicReferences: boolean;
                /**
                 * Whether the user can copy style from Spread Sheets then paste to Excel, or copy style from Excel then paste to Spread Sheets. The default value is true.
                 */
                allowCopyPasteExcelStyle: boolean;
                /**
                 * Whether extend paste range if the paste range is not enough for pasting. The default value is false.
                 */
                allowExtendPasteRange: boolean;
                /**
                 * Which headers are included when data is copied to or pasted. The default value is GC.Spread.Sheets.CopyPasteHeaderOptions.allHeaders.
                 */
                copyPasteHeaderOptions: GC.Spread.Sheets.CopyPasteHeaderOptions;
                /**
                 * The recalculation behavior of the workbook. The default value is GC.Spread.Sheets.CalculationMode.auto.
                 */
                calculationMode: GC.Spread.Sheets.CalculationMode;
                /**
                 * Whether to enable the precision scrolling by pixel. The default value is false.
                 */
                scrollByPixel: boolean;
                /**
                 * Decides scrolling by that number of pixels at a time when scrollByPixel is true. The final scrolling pixels are the result of scrolling delta multiply scrollPixel. For example, the scrolling delta is 3, and the scrollPixel is 5, the final scrolling pixels are 15. The default value is 5.
                 */
                scrollPixel: number;
                /**
                 * Whether to enable the accessibility support in the spreadsheet. The default value is false.
                 */
                enableAccessibility: boolean;
                /**
                 * Whether to enable auto creating hyperlink in the spreadsheet. The default value is true.
                 */
                allowAutoCreateHyperlink: boolean;
                /**
                 * Specifies the way to resize column. The default value is GC.Spread.Sheets.ResizeMode.normal.
                 */
                columnResizeMode: GC.Spread.Sheets.ResizeMode;
                /**
                 * Specifies the way to resize row. The default value is GC.Spread.Sheets.ResizeMode.normal.
                 */
                rowResizeMode: GC.Spread.Sheets.ResizeMode;
                /**
                 * The list for user to customize drag fill, prioritize matching this list in each fill. Each array item is type of string array.
                 */
                customList?: string[][];
                /**
                 * The scrollbar appearance, contains skin and mobile two enums. The default value is GC.Spread.Sheets.ScrollbarAppearance.skin.
                 */
                scrollbarAppearance?: GC.Spread.Sheets.ScrollbarAppearance;
                /**
                 * Whether paste skip invisible range. The default value is false.
                 */
                pasteSkipInvisibleRange: boolean;
                /**
                 * Whether allow auto extend filter range like excel. The default value is false.
                 */
                allowAutoExtendFilterRange: boolean;
                /**
                 * Whether allow input invalid formula string. The default value is false.
                 */
                allowInvalidFormula: boolean;
                /**
                 * Whether automatically generate the getPivotData formula or cell reference when choose pivot table data area. The default value is GC.Spread.Pivot.PivotAreaReference.getPivotData.
                 */
                pivotAreaReference: GC.Spread.Pivot.PivotAreaReference;
                /**
                 * Whether automatically generate the format when formula inputs. The default value is true.
                 */
                formulaFormatHint: boolean;
                /**
                 * All default state styles for sheet tabs.
                 */
                defaultSheetTabStyles: GC.Spread.Sheets.SheetTabStyles;
                /**
                 * All built-in file icons.
                 */
                builtInFileIcons: GC.Spread.Sheets.IBuiltInFileIcons;
            }


            export interface IWorkFlowItem{
                value: string;
                transitions: number[]|string[];
            }


            export interface IWorkFlowOption{
                items:GC.Spread.Sheets.IWorkFlowItem[];
            }


            export interface IWorkSheetGridlineOption{
                /**
                 * The grid line color
                 */
                color?: string;
                /**
                 * Whether to show the vertical grid line.
                 */
                showVerticalGridline?: boolean;
                /**
                 * Whether to show the horizontal grid line.
                 */
                showHorizontalGridline?: boolean;
            }


            export interface IWorksheetOptions{
                /**
                 * indicates whether data can overflow into adjacent empty cells.
                 */
                allowCellOverflow: boolean;
                /**
                 * indicates whether display the formulas string not the formula result.
                 */
                showFormulas: boolean;
                /**
                 * indicates whether display the 0 in cells containing zero value. Default is true.
                 */
                showZeros: boolean;
                /**
                 * A color string used to represent the sheet tab color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 */
                sheetTabColor: string;
                /**
                 * A color string used to represent the frozen line color, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 */
                frozenlineColor: string;
                /**
                 * The clipboard option.
                 */
                clipBoardOptions: GC.Spread.Sheets.ClipboardPasteOptions;
                /**
                 * The grid line's options.
                 */
                gridline: GC.Spread.Sheets.IWorkSheetGridlineOption;
                /**
                 * Indicates whether the row header is visible.
                 */
                rowHeaderVisible: boolean;
                /**
                 * Indicates whether the column header is visible.
                 */
                colHeaderVisible: boolean;
                /**
                 * Indicates whether the row header displays letters or numbers or is blank.
                 */
                rowHeaderAutoText: GC.Spread.Sheets.HeaderAutoText;
                /**
                 * Indicates whether the column header displays letters or numbers or is blank.
                 */
                colHeaderAutoText: GC.Spread.Sheets.HeaderAutoText;
                /**
                 * Specifies which row header column displays the automatic text when there are multiple row header columns.
                 */
                rowHeaderAutoTextIndex: GC.Spread.Sheets.HeaderAutoText;
                /**
                 * Specifies which column header row displays the automatic text when there are multiple column header rows.
                 */
                colHeaderAutoTextIndex: GC.Spread.Sheets.HeaderAutoText;
                /**
                 * Indicates whether cells on this sheet that are marked as protected cannot be edited.
                 */
                isProtected: boolean;
                /**
                 * A value that indicates the elements that you want users to be able to change.
                 */
                protectionOptions: GC.Spread.Sheets.IProtectionOptions;
                /**
                 * The selection's background color for the sheet.
                 */
                selectionBackColor: string;
                /**
                 * The selection's border color for the sheet.
                 */
                selectionBorderColor: string;
                /**
                 * The sheetAreaOffset's options.
                 */
                sheetAreaOffset: GC.Spread.Sheets.IWorkSheetSheetAreaOffsetOption;
                /**
                 * Indicates whether the unknown formulas could be included in sheet json data.
                 */
                keepUnknownFormulas: boolean;
                /**
                 * The add row button's options.
                 */
                addRowButtonOption?: GC.Spread.Sheets.IAddRowButtonOption;
                /**
                 * The add column button's options.
                 */
                addColumnButtonOption?: GC.Spread.Sheets.IAddColumnButtonOption;
                /**
                 * Indicates whether the sheet is rendered from right to left.
                 */
                rightToLeft: boolean;
                /**
                 * Indicates the URL of the background image.
                 */
                backgroundImage?: string;
            }


            export interface IWorkSheetSheetAreaOffsetOption{
                /**
                 * The offset left of sheet from host.
                 */
                left?: number;
                /**
                 * The offset top of sheet from host
                 */
                top?: number;
            }


            export interface SheetTabInfo{
                sheetName?: string;
                sheetIndex?: number;
                sheetType?: GC.Spread.Sheets.SheetType;
            }


            /**
             * @typedef GC.Spread.Sheets.CellValue - Indicates the cell value in col data.
             * @property {any} col - A key-value collection, which key is a col number, and value is cell value with any type.
             */
            export type CellValue = 
                {
                    [col: number]: any;
                }
            

            /**
             * @typedef GC.Spread.Sheets.CustomDocumentPropertyValueType
             * @type {string|number|boolean|Date}
             * @description Custom Document property type.
             */
            export type CustomDocumentPropertyValueType = 
                string|number|boolean|Date
            

            /**
             * @typedef GC.Spread.Sheets.ExcelFileType
             * @type {"XLSX" | "XLSM" | "XLTM"}
             * @description Represents the fileType of Excel.
             */
            export type ExcelFileType = 
                "XLSX" | "XLSM" | "XLTM"
            

            /**
             * @typedef GC.Spread.Sheets.ExportCsvOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {string} [encoding] - Deprecated: Currently only support to export in UTF-8 encoding, set the `encoding` attribute will take no effect.
             * @property {string} [rowDelimiter] - The row delimiter that is appended to the end of the row, the default row delimiter is '\r\n'.
             * @property {string} [columnDelimiter] - The column delimiter that is appended to the end of the column, the default column delimiter is ','.
             * @property {object} [range] - The range info.
             * @param {number} [range.sheetIndex] - The sheet index, the default sheet index is current active sheet index.
             * @param {number} [range.row] - The start row, the default row index is 0.
             * @param {number} [range.column] - The start column, the default column index is 0.
             * @param {number} [range.rowCount] - The row count, the default row count is current active sheet row count.
             * @param {number} [range.columnCount] - The column count, the default column count is current active sheet column count.
             
             */
            export type ExportCsvOptions = 
                {
                    encoding?: string;
                    rowDelimiter?: string;
                    columnDelimiter?: string;
                    range?: {
                        sheetIndex: number;
                        row: number;
                        column: number;
                        rowCount: number;
                        columnCount: number;
                    }
                }
            

            /**
             * @typedef {GC.Spread.Sheets.ExportCsvOptions | GC.Spread.Sheets.ExportSSJsonOptions | GC.Spread.Sheets.ExportXlsxOptions} GC.Spread.Sheets.ExportOptions - The options for export function.
             */
            export type ExportOptions = 
                GC.Spread.Sheets.FileOptions & (GC.Spread.Sheets.ExportCsvOptions | GC.Spread.Sheets.ExportSSJsonOptions | GC.Spread.Sheets.ExportXlsxOptions)
            

            /**
             * @typedef GC.Spread.Sheets.ExportSSJsonOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {boolean} [includeBindingSource] - Whether to include the binding source when converting the workbook to json, default false.
             * @property {boolean} [includeStyles] - Whether to include the style when converting the workbook to json, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when converting the workbook to json, default true.
             * @property {boolean} [saveAsView] - Whether to apply the format string to exporting value when converting the workbook to json, default false.
             * @property {boolean} [rowHeadersAsFrozenColumns] - Whether to treat the row headers as frozen columns when converting the workbook to json, default false.
             * @property {boolean} [columnHeadersAsFrozenRows] - Whether to treat the column headers as frozen rows when converting the workbook to json, default false.
             * @property {boolean} [includeAutoMergedCells] - Whether to include the automatically merged cells when converting the workbook to json, default false.
             */
            export type ExportSSJsonOptions = 
                {
                    includeBindingSource?: boolean;
                    includeStyles?: boolean;
                    includeFormulas?: boolean;
                    saveAsView?: boolean;
                    rowHeadersAsFrozenColumns?: boolean;
                    columnHeadersAsFrozenRows?: boolean;
                    includeAutoMergedCells?: boolean;
                    saveR1C1Formula?: boolean;
                }
            

            /**
             * @typedef GC.Spread.Sheets.ExportXlsxOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {boolean} [includeBindingSource] - Whether to include the binding source when do save, default false.
             * @property {boolean} [includeStyles] - Whether to include the style when do save, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when do save, default true.
             * @property {boolean} [saveAsView] - Whether to apply the format string to exporting value when do save, default false.
             * @property {boolean} [rowHeadersAsFrozenColumns] - Whether to treat the row headers as frozen columns when do save, default false.
             * @property {boolean} [columnHeadersAsFrozenRows] - Whether to treat the column headers as frozen rows when do save, default false.
             * @property {boolean} [includeAutoMergedCells] - Whether to include the automatically merged cells when do save, default false.
             * @property {boolean} [includeCalcModelCache] - [deprecated] Whether to include the extra data of calculation. Can be faster when open the file with those data, default false.
             * @property {boolean} [includeUnusedNames] - Whether to include the unused custom name when do save, default true.
             * @property {boolean} [exportSharedFormula] - Whether to include the shared formula when do save, default true.
             * @property {boolean} [includeEmptyRegionCells] - Whether to include any empty cells(cells with no data or only style) outside the used data range, default true.
             * @property {boolean} [losslessEditing] - Whether to include lossless editing content, default true.
             * @property {string} [password] - Set the password to open the workbook.
             * @property {GC.Spread.Sheets.ExcelFileType} [excelFileType] - The exporting excel file type ('XLSX' | 'XLSM' | 'XLTM'), default by XLSX.
             */
            export type ExportXlsxOptions = 
                {
                    includeBindingSource?: boolean;
                    includeStyles?: boolean;
                    includeFormulas?: boolean;
                    saveAsView?: boolean;
                    rowHeadersAsFrozenColumns?: boolean;
                    columnHeadersAsFrozenRows?: boolean;
                    includeUnusedNames?: boolean;
                    exportSharedFormula?: boolean;
                    includeEmptyRegionCells?: boolean;
                    includeAutoMergedCells?: boolean;
                    includeCalcModelCache?: boolean;
                    losslessEditing?: boolean;
                    password?: string;
                    excelFileType?: GC.Spread.Sheets.ExcelFileType;
                }
            

            /**
             * @typedef GC.Spread.Sheets.ExternalPartialValues - Indicates the partial values in each referenced files.
             * @property {GC.Spread.Sheets.PartialValues} filePath - A key-value collection, which key is a file path string, and value is type of GC.Spread.Sheets.PartialValues.
             */
            export type ExternalPartialValues = 
                {
                    [filePath: string]: GC.Spread.Sheets.PartialValues;
                }
            

            /**
             * @typedef GC.Spread.Sheets.FileOptions - The file options for export.
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             */
            export type FileOptions = 
                {
                    /**
                     * The file type.
                     */
                    fileType: GC.Spread.Sheets.FileType
                }
            

            /**
             * @typedef GC.Spread.Sheets.ImportCsvOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {string} [encoding] - The csv encoding type, the default encoding type is 'UTF-8'.
             * @property {string} [rowDelimiter] - The row delimiter that is appended to the end of the row, the default row delimiter is '\r\n'.
             * @property {string} [columnDelimiter] - The column delimiter that is appended to the end of the column, the default column delimiter is ','.
             */
            export type ImportCsvOptions = 
                {
                    rowDelimiter?: string;
                    columnDelimiter?: string;
                    encoding?: string;
                }
            

            /**
             */
            export type ImportOptions = 
                GC.Spread.Sheets.FileOptions & (GC.Spread.Sheets.ImportCsvOptions | GC.Spread.Sheets.ImportSSJsonOptions | GC.Spread.Sheets.ImportXlsxOptions)
            

            /**
             * @typedef GC.Spread.Sheets.ImportSSJsonOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {boolean} [includeStyles] - Whether to include the style when converting json to the workbook, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when converting json to the workbook, default true.
             * @property {boolean} [frozenColumnsAsRowHeaders] - Whether to treat the frozen columns as row headers when converting json to the workbook, default false.
             * @property {boolean} [frozenRowsAsColumnHeaders] - Whether to treat the frozen rows as column headers when converting json to the workbook, default false.
             * @property {boolean} [fullRecalc] - Whether to do full recalculation after loading the json data, default true.
             * @property {boolean | object} [incrementalLoad] - Whether to use the incremental loading or the callbacks of incremental loading when converting json to the workbook, default false.
             * @param {function} [incrementalLoad.loading] - The callback when of the incremental loading progress.
             * @param {function} [incrementalLoad.loaded] - The callback when of the incremental loading finished.
             */
            export type ImportSSJsonOptions = 
                {
                    includeStyles?: boolean;
                    incrementalLoad?: any;
                    frozenColumnsAsRowHeaders?: boolean;
                    frozenRowsAsColumnHeaders?: boolean;
                    includeFormulas?: boolean;
                    fullRecalc?: boolean;
                }
            

            /**
             * @typedef GC.Spread.Sheets.ImportXlsxOptions
             * @property {GC.Spread.Sheets.FileType} fileType - The file type.
             * @property {boolean} [includeStyles] - Whether to include the style when loading, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when loading, default true.
             * @property {boolean} [frozenColumnsAsRowHeaders] - Whether to treat the frozen columns as row headers when loading, default false.
             * @property {boolean} [frozenRowsAsColumnHeaders] - Whether to treat the frozen rows as column headers when loading, default false.
             * @property {boolean} [fullRecalc] - Whether to calculate after loading the json data, false by default.
             * @property {boolean} [dynamicReferences] - Whether to calculate functions with dynamic reference, default true.
             * @property {boolean} [calcOnDemand] - Whether to calculate formulas only when they are demanded, default false.
             * @property {boolean} [incrementalCalculation] - Whether to incremental calculate formulas without blocking UI, default false.
             * @property {boolean} [includeUnusedStyles] - Whether to include the unused name style when converting excel xml to the json, default true.
             * @property {boolean} [convertSheetTableToDataTable] - Whether to convert the sheet tables to data manager tables, default false.
             * @property {string} [password] - The password to open the workbook.
             * @property {GC.Spread.Sheets.OpenMode} [openMode] - The open mode of normal, lazy and incremental. By default is normal.
             * @property {GC.Spread.Sheets.ProgressFunctionType} [progress] - The progress callback function for each open mode.
             * @property {GC.Spread.Sheets.ExcelFileType} [excelFileType] - The importing excel file type ('XLSX' | 'XLSM' | 'XLTM'), default by XLSX.
             */
            export type ImportXlsxOptions = 
                {
                    includeStyles?: boolean;
                    frozenColumnsAsRowHeaders?: boolean;
                    frozenRowsAsColumnHeaders?: boolean;
                    includeFormulas?: boolean;
                    fullRecalc?: boolean;
                    dynamicReferences?: boolean;
                    calcOnDemand?: boolean;
                    incrementalCalculation?: boolean;
                    includeUnusedStyles?: boolean;
                    convertSheetTableToDataTable?: boolean;
                    password?: string;
                    openMode?: GC.Spread.Sheets.OpenMode;
                    progress?: GC.Spread.Sheets.ProgressFunctionType;
                    excelFileType?: GC.Spread.Sheets.ExcelFileType;
                }
            

            /**
             * @typedef GC.Spread.Sheets.OpenOptions
             * @property {boolean} [includeStyles] - Whether to include the style when loading, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when loading, default true.
             * @property {boolean} [fullRecalc] - Whether to calculate after loading the json data, false by default.
             * @property {boolean} [dynamicReferences] - Whether to calculate functions with dynamic reference, default true.
             * @property {boolean} [calcOnDemand] - Whether to calculate formulas only when they are demanded, default false.
             * @property {boolean} [incrementalCalculation] - Whether to incremental calculate formulas without blocking UI, default false.
             * @property {boolean} [includeUnusedStyles] - Whether to include the unused name style when converting excel xml to the json, default true.
             * @property {GC.Spread.Sheets.OpenMode} [openMode] - The open mode of normal, lazy and incremental. By default is normal.
             * @property {GC.Spread.Sheets.ProgressFunctionType} [progress] - The progress callback function for each open mode.
             */
            export type OpenOptions = 
                {
                    includeStyles?: boolean;
                    includeFormulas?: boolean;
                    fullRecalc?: boolean;
                    dynamicReferences?: boolean;
                    calcOnDemand?: boolean;
                    incrementalCalculation?: boolean;
                    includeUnusedStyles?: boolean;
                    openMode?: GC.Spread.Sheets.OpenMode;
                    progress?: GC.Spread.Sheets.ProgressFunctionType;
                }
            

            /**
             * @typedef GC.Spread.Sheets.PartialValues - Indicates the cell data in row data.
             * @property {GC.Spread.Sheets.CellValue} row - A key-value collection, which key is a row number, and value is type of GC.Spread.Sheets.CellValue.
             */
            export type PartialValues = 
                {
                    [row: number]: GC.Spread.Sheets.CellValue;
                }
            

            /**
             * @typedef GC.Spread.Sheets.ProgressArgs
             * @property {string} sheetName - the current loading sheet's name.
             * @property {string} step - the current loading step.
             * @property {number} progress - the current loading progress, from 0 ~ 1.
             */
            export type ProgressArgs = 
                {
                    /**
                     * the current loading sheet's name.
                     */
                    sheetName?: string;
                    /**
                     * the current loading step.
                     */
                    step: string;
                    /**
                     * the current loading progress, from 0 ~ 1.
                     */
                    progress: number;
                }
            

            /**
             * @typedef GC.Spread.Sheets.ProgressFunctionType
             * @param {object} progressArgs - the progress arguments.
             * @param {string} [progressArgs.sheetName] - the current loading sheet's name.
             * @param {string} progressArgs.step - the current loading step.
             * @param {number} progressArgs.progress - the current loading progress, from 0 ~ 1.
             * @description The callback when of the incremental loading progress.
             */
            export type ProgressFunctionType = 
                (progressArgs: GC.Spread.Sheets.ProgressArgs) => void
            

            /**
             * @typedef GC.Spread.Sheets.SaveOptions
             * @property {boolean} [includeBindingSource] - Whether to include the binding source when do save, default false.
             * @property {boolean} [includeStyles] - Whether to include the style when do save, default true.
             * @property {boolean} [includeFormulas] - Whether to include the formula when do save, default true.
             * @property {boolean} [saveAsView] - Whether to apply the format string to exporting value when do save, default false.
             * @property {boolean} [includeAutoMergedCells] - Whether to include the automatically merged cells when do save, default false.
             * @property {boolean} [includeCalcModelCache] - Whether to include the extra data of calculation. Can be faster when open the file with those data, default false.
             * @property {boolean} [includeUnusedNames] - Whether to include the unused custom name when do save, default true.
             * @property {boolean} [includeEmptyRegionCells] - Whether to include any empty cells(cells with no data or only style) outside the used data range, default true.
             * @property {boolean} [saveR1C1Formula] - Whether to save the r1c1 formula in the file, only works in sjs file type, default false.
             */
            export type SaveOptions = 
                {
                    includeBindingSource?: boolean;
                    includeStyles?: boolean;
                    includeFormulas?: boolean;
                    saveAsView?: boolean;
                    includeUnusedNames?: boolean;
                    includeEmptyRegionCells?: boolean;
                    includeAutoMergedCells?: boolean;
                    includeCalcModelCache?: boolean;
                    saveR1C1Formula?: boolean;
                }
            

            /**
             * @typedef {object} GC.Spread.Sheets.SheetTabStyles
             * @property {GC.Spread.Sheets.SheetTabState} sheetTabState
             */
            export type SheetTabStyles = 
                {
                    [sheetTabState in GC.Spread.Sheets.SheetTabState]?: GC.Spread.Sheets.ISheetTabStyle;
                }
            
            /**
             * Specifies how to allow drag header to move.
             * @enum {number}
             */
            export enum AllowDragHeaderToMove{
                /**
                 * Specifies that no drag header to move allowed.
                 */
                none= 0,
                /**
                 * Specifies that only the column header allow to move.
                 */
                column= 1,
                /**
                 * Specifies that only the row header allow to move.
                 */
                row= 2,
                /**
                 * Specifies that both column and row header allow to move.
                 */
                both= 3
            }

            /**
             * Sheet tab all sheets list button visible option.
             * @enum {number}
             */
            export enum AllSheetsListVisibility{
                /**
                 * All sheets list button is hide
                 */
                hide= 0,
                /**
                 * All sheets list button is show
                 */
                show= 1,
                /**
                 * All sheets list button is auto
                 */
                auto= 2
            }

            /**
             * Represents whether the component automatically resizes cells or headers.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the AutoFitType enumeration.
             * spread.options.autoFitType = GC.Spread.Sheets.AutoFitType.cellWithHeader;
             * ```
             */
            export enum AutoFitType{
                /**
                 *  The component autofits cells.
                 */
                cell= 0,
                /**
                 *   The component autofits cells and headers.
                 */
                cellWithHeader= 1
            }

            /**
             * Specifies the type of cellbutton.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the type of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: true,
             *     imageType:GC.Spread.Sheets.ButtonImageType.left,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: false,
             *     imageType:GC.Spread.Sheets.ButtonImageType.right,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             * ```
             */
            export enum ButtonImageType{
                /** Indicates the cellbutton image type is none.
                 * @type {number}
                 */
                none= 0,
                /** Indicates the cellbutton type is custom.
                 * @type {number}
                 */
                custom= 1,
                /** Indicates the cellbutton type is clear.
                 * @type {number}
                 */
                clear= 2,
                /** Indicates the cellbutton type is cancel.
                 * @type {number}
                 */
                cancel= 3,
                /** Indicates the cellbutton type is ok.
                 * @type {number}
                 */
                ok= 4,
                /** Indicates the cellbutton type is dropdown.
                 * @type {number}
                 */
                dropdown= 5,
                /** Indicates the cellbutton type is ellipsis.
                 * @type {number}
                 */
                ellipsis= 6,
                /** Indicates the cellbutton type is left.
                 * @type {number}
                 */
                left= 7,
                /** Indicates the cellbutton type is right.
                 * @type {number}
                 */
                right= 8,
                /** Indicates the cellbutton type is plus.
                 * @type {number}
                 */
                plus= 9,
                /** Indicates the cellbutton type is minus.
                 * @type {number}
                 */
                minus= 10,
                /** Indicates the cellbutton type is undo.
                 * @type {number}
                 */
                undo= 11,
                /** Indicates the cellbutton type is redo.
                 * @type {number}
                 */
                redo= 12,
                /** Indicates the cellbutton type is search.
                 * @type {number}
                 */
                search= 13,
                /** Indicates the cellbutton type is separator.
                 * @type {number}
                 */
                separator= 14,
                /** Indicates the cellbutton type is spinLeft.
                 * @type {number}
                 */
                spinLeft= 15,
                /** Indicates the cellbutton type is spinRight.
                 * @type {number}
                 */
                spinRight= 16,
                /** Indicates the cellbutton type is collapse.
                 * @type {number}
                 */
                collapse= 17,
                /** Indicates the cellbutton type is expand.
                 * @type {number}
                 */
                expand= 18
            }

            /**
             * Specifies the position of cellbutton.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the position of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.right,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.left,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.onSelected,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             * ```
             */
            export enum ButtonPosition{
                /** Indicates the cellbutton is in left of the cell.
                 * @type {number}
                 */
                left= 0,
                /** Indicates the cellbutton is in right of the cell.
                 * @type {number}
                 */
                right= 1
            }

            /**
             * Specifies the visibility of cellbutton.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the visibility of cellbutton.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: true,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.always,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     isLeft: false,
             *     visibility:GC.Spread.Sheets.ButtonVisibility.onSelected,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             * ```
             */
            export enum ButtonVisibility{
                /** Indicates the cellbutton alway visible.
                 * @type {number}
                 */
                always= 0,
                /** Indicates the cellbutton visible when the cell is active.
                 * @type {number}
                 */
                onSelected= 1,
                /** Indicates the cellbutton visible when the cell enter edit.
                 * @type {number}
                 */
                onEditing= 2
            }

            /**
             * Specifies the recalculation behavior of the workbook.
             * @enum {number}
             * @example
             * ```javascript
             * spread.options.calculationMode = GC.Spread.Sheets.CalculationMode.manual;
             * ```
             */
            export enum CalculationMode{
                /**
                 * The default recalculation behavior that calculates formulas every time the relevant data is changed.
                 */
                auto= 0,
                /**
                 * Calculations only occur when the user requests them.
                 */
                manual= 1
            }

            /**
             * Specifies the calculation type of the calculate function.
             * @enum {number}
             * spread.calculate(GC.Spread.Sheets.CalculationType.manual);
             */
            export enum CalculationType{
                /**
                 * The default calculation type that mark cells in the range as dirty for calculate.
                 */
                all= 0,
                /**
                 * This will rebuild all the formulas model in the range and then mark them as dirty for calculate.
                 */
                rebuild= 1,
                /**
                 * Keep the current calc dirty status, and won't mark the volatile cells, circular reference cells.
                 */
                minimal= 2,
                /**
                 * Keep the current calc dirty status, and mark the volatile cells and circular reference cells as dirty for calculate.
                 */
                regular= 3
            }

            /**
             * Specifies the calendar's default page.
             * @enum {number}
             */
            export enum CalendarPage{
                /**
                 * Specifies the calendar's default page is year.
                 */
                year= 1,
                /**
                 * Specifies the calendar's default page is month.
                 */
                month= 2,
                /**
                 * Specifies the calendar's default page is day.
                 */
                day= 3
            }

            /**
             * Specifies the calendar's start day.
             * @enum {number}
             */
            export enum CalendarStartDay{
                /**
                 * Specifies the calendar's start day is Monday.
                 */
                monday= 1,
                /**
                 * Specifies the calendar's start day is Tuesday.
                 */
                tuesday= 2,
                /**
                 * Specifies the calendar's start day is Wednesday.
                 */
                wednesday= 3,
                /**
                 * Specifies the calendar's start day is Thursday.
                 */
                thursday= 4,
                /**
                 * Specifies the calendar's start day is Friday.
                 */
                friday= 5,
                /**
                 * Specifies the calendar's start day is Saturday.
                 */
                saturday= 6,
                /**
                 * Specifies the calendar's start day is Sunday.
                 */
                sunday= 7
            }

            /**
             * Specifies the position of caption.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the position of caption.
             * //create config
             * leftButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.right,
             *     captionAlign:GC.Spread.Sheets.CaptionAlignment.right,
             * };
             * rightButtonConfig1 = {
             *     caption: "left",
             *     enabled: true,
             *     position: GC.Spread.Sheets.ButtonPosition.left,
             *     captionAlign:GC.Spread.Sheets.CaptionAlignment.left,
             * };
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.cellButtons=[
             *      leftButtonConfig1,
             *      rightButtonConfig1
             * ];
             * sheet.setStyle(0, 0, style);
             * ```
             */
            export enum CaptionAlignment{
                /** Indicates the caption is in left of the button.
                 * @type {number}
                 */
                left= 0,
                /** Indicates the caption is in right of the button.
                 * @type {number}
                 */
                right= 1
            }

            /**
             * Specifies the type of CellState.
             * @enum {number}
             */
            export enum CellStatesType{
                /** When mouse hover on the cell , cell state include "hover" state.
                 * @type {number}
                 */
                hover= 1,
                /** When the data-validation conditional evaluate fail, cell state include "invalid" state.
                 * @type {number}
                 */
                invalid= 2,
                /** When the cell is locked in protect worksheet , cell state include "readonly" state.
                 * @type {number}
                 */
                readonly= 4,
                /** When the cell is editing, cell state include "edit" state.
                 * @type {number}
                 */
                edit= 8,
                /** When the cell is focus, cell state include "active" state.
                 * @type {number}
                 */
                active= 16,
                /** When the cell is in the selection range, cell state include "selected" state.
                 * @type {number}
                 */
                selected= 32,
                /** When cell value is changed, cell state include "dirty" state.
                 * @type {number}
                 */
                dirty= 64,
                /** When cell value is invalid formula string, cell state include "invalidFormula" state.
                 * @type {number}
                 */
                invalidFormula= 128
            }

            /**
             * Specifies the type of clearing pending change.
             * @enum {number}
             */
            export enum ClearPendingChangeType{
                /**
                 * Clear dirty change.
                 */
                dirty= 1,
                /**
                 * Clear insert change.
                 */
                insert= 2,
                /**
                 * Clear deleted change.
                 */
                delete= 4
            }

            /**
             * Present the type of clipboard event action
             * @enum {number}
             */
            export enum ClipboardActionType{
                /**
                 * Indicates a paste action
                 */
                copy= 1,
                /**
                 * Indicates a cut action
                 */
                cut= 2,
                /**
                 * Indicates a reset action, the clipboard is reset at this point
                 */
                reset= 3
            }

            /**
             * Specifies what data is pasted from the Clipboard.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the ClipboardPasteOptions enumeration.
             * activeSheet.options.clipBoardOptions = GC.Spread.Sheets.ClipboardPasteOptions.values;
             * ```
             */
            export enum ClipboardPasteOptions{
                /**
                 * Pastes all data objects, including values, formatting, and formulas.
                 */
                all= 0,
                /**
                 * Pastes only values.
                 */
                values= 1,
                /**
                 * Pastes only formatting.
                 */
                formatting= 2,
                /**
                 * Pastes only formulas.
                 */
                formulas= 3,
                /**
                 * Pastes values and formatting.
                 */
                valuesAndFormatting= 4,
                /**
                 * Pastes formulas and formatting.
                 */
                formulasAndFormatting= 5,
                /**
                 * Pastes only comments.
                 */
                comments= 6,
                /**
                 * Paste only validation.
                 */
                validation= 7,
                /**
                 * Paste all data object and apply source theme to style/
                 */
                usingSourceTheme= 8,
                /**
                 * Pastes all cell contents except borders.
                 */
                noBorder= 9,
                /**
                 * Pastes the column widths.
                 */
                columnWidths= 10,
                /**
                 * Paste formulas and number formatting options
                 */
                formulasAndNumberFormatting= 11,
                /**
                 * Paste values and number formatting options
                 */
                valuesAndNumberFormatting= 12,
                /**
                 * Paste values and source formatting options
                 */
                valuesAndSourceFormatting= 13,
                /**
                 * Pastes all data objects and keep column width
                 */
                allAndColumnWidth= 14
            }

            /**
             * Specifies which headers are included when data is copied to or pasted.
             * @enum {number}
             * @example
             * ```javascript
             * //This example allows you to copy and paste headers.
             * spread.options.copyPasteHeaderOptions = GC.Spread.Sheets.CopyPasteHeaderOptions.allHeaders;
             * activeSheet.setRowCount(2,GC.Spread.Sheets.SheetArea.colHeader);
             * activeSheet.setColumnCount(2,GC.Spread.Sheets.SheetArea.rowHeader);
             * activeSheet.setValue(0, 2,"Column",GC.Spread.Sheets.SheetArea.colHeader);
             * activeSheet.setValue(1, 0,"Row",GC.Spread.Sheets.SheetArea.rowHeader);
             * ```
             */
            export enum CopyPasteHeaderOptions{
                /**
                 * Includes neither column nor row headers when data is copied; does not overwrite selected column or row headers when data is pasted.
                 */
                noHeaders= 0,
                /**
                 * Includes selected row headers when data is copied; overwrites selected row headers when data is pasted.
                 */
                rowHeaders= 1,
                /**
                 * Includes selected column headers when data is copied; overwrites selected column headers when data is pasted.
                 */
                columnHeaders= 2,
                /**
                 * Includes selected headers when data is copied; overwrites selected headers when data is pasted.
                 */
                allHeaders= 3
            }

            /**
             * Specifies the copy to option.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the CopyToOption enumeration.
             * activeSheet.getCell(0,0).value("1");
             * activeSheet.copyTo(0,0,1,1,2,2,GC.Spread.Sheets.CopyToOptions.value);
             * ```
             */
            export enum CopyToOptions{
                /**
                 * Indicates the type of data is pure data.
                 */
                value= 1,
                /**
                 * Indicates the type of data is a formula.
                 */
                formula= 2,
                /**
                 * Indicates the type of data is a comment.
                 */
                comment= 4,
                /**
                 * Indicates to copy a range group.
                 */
                outline= 8,
                /**
                 * Indicates the type of data is a sparkline.
                 */
                sparkline= 16,
                /**
                 * Indicates to copy a span.
                 */
                span= 32,
                /**
                 * Indicates the type of data is a style.
                 */
                style= 64,
                /**
                 * Indicates the type of data is a tag.
                 */
                tag= 128,
                /**
                 * Indicates the type of data is a binding path.
                 */
                bindingPath= 256,
                /**
                 * Indicates the type of data is a conditional format.
                 */
                conditionalFormat= 512,
                /**
                 * Indicates the type of data is a hyperlink.
                 */
                hyperlink= 1024,
                /**
                 * Indicates the type of data is a defaultValue.
                 */
                defaultValue= 2048,
                /**
                 * Indicates the type of data is an alternative text.
                 */
                altText= 4096,
                /**
                 * Indicates all types of data.
                 */
                all= 32767
            }

            /**
             * Describe the position of the corner fold in a cell.
             * @enum {number}
             */
            export enum CornerPosition{
                /**
                 * Indicates that the folded corner is in the upper left corner
                 */
                leftTop= 1,
                /**
                 * Indicates that the folded corner is in the upper right corner
                 */
                rightTop= 2,
                /**
                 * indicates that the folded corner is in the lower left corner
                 */
                leftBottom= 4,
                /**
                 * Indicates that the folded corner is in the lower right corner
                 */
                rightBottom= 8
            }

            /**
             * Specifies the cell value type of dropdown list result.
             * @enum {number}
             */
            export enum DropdownListValue{
                /**
                 * Use the string of list select result, use comma to split.
                 */
                string= 0,
                /**
                 * Specifies the sort will move the group.
                 */
                array= 1
            }

            /**
             * Specifies the type of dropdown.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the type of dropdown.
             * //create style
             * var style = new GC.Spread.Sheets.Style();
             * style.dropDowns=[
             *    {
             *      type:GC.Spread.Sheets.DropDownType.dateTimePicker,
             *      option: {
             *          showTime:false
             *      }
             *    }
             * ];
             * sheet.setStyle(0, 0, style);
             * ```
             */
            export enum DropDownType{
                /** Indicates the type of dropdown is colorpicker.
                 * @type {number}
                 */
                colorPicker= 0,
                /** Indicates the type of dropdown is dateTimePicker.
                 * @type {number}
                 */
                dateTimePicker= 1,
                /** Indicates the type of dropdown is timePicker.
                 * @type {number}
                 */
                timePicker= 2,
                /** Indicates the type of dropdown is monthPicker.
                 * @type {number}
                 */
                monthPicker= 3,
                /** Indicates the type of dropdown is list.
                 * @type {number}
                 */
                list= 4,
                /** Indicates the type of dropdown is slider.
                 * @type {number}
                 */
                slider= 5,
                /** Indicates the type of dropdown is calculator.
                 * @type {number}
                 */
                calculator= 6,
                /** Indicates the type of dropdown is workflowList.
                 * @type {number}
                 */
                workflowList= 7,
                /** Indicates the type of dropdown is multiColumn.
                 * @type {number}
                 */
                multiColumn= 8
            }

            /**
             * Specifies the editor status.
             * @enum {number}
             */
            export enum EditorStatus{
                /**
                 * Cell is in Ready mode.
                 */
                ready= 0,
                /**
                 * Cell is in editing mode and can commit the input value and navigate to or select other cells when invoking navigation or selection actions.
                 */
                enter= 1,
                /**
                 * Cell is in editing mode and cannot commit the input value and navigate to or select other cells.
                 */
                edit= 2
            }

            /**
             * The file type for import/export excel file or csv file or ssjson file.
             * @enum {number}
             */
            export enum FileType{
                /**
                 * Import/export excel type file.
                 */
                excel= 0,
                /**
                 * Import/export ssjson type file.
                 */
                ssjson= 1,
                /**
                 * Import/export csv type file.
                 */
                csv= 2
            }

            /**
             * Specifies the method when sort with group.
             * @enum {number}
             */
            export enum GroupSort{
                /**
                 * Specifies the sort don't move the group.
                 */
                none= 0,
                /**
                 * Specifies the sort will move the group.
                 */
                group= 1,
                /**
                 * Specifies that only sort inner the group.
                 */
                child= 2,
                /**
                 * Specifies that sort will do with the group and inner the group.
                 */
                full= 3
            }

            /**
             * Specifies which default labels are displayed in headers.
             * @enum {number}
             */
            export enum HeaderAutoText{
                /**
                 *  Displays blanks in the headers.
                 */
                blank= 0,
                /**
                 *  Displays numbers in the headers.
                 */
                numbers= 1,
                /**
                 *  Displays letters in the headers.
                 */
                letters= 2
            }

            /**
             * Specifies which headers are included when export range data to HTML.
             * @enum {number}
             * @example
             * ```javascript
             * //This example shows you how to export range data to HTML with row header and column header.
             * activeSheet.getRange(-1, -1, -1, -1).toHtml(GC.Spread.Sheets.HeaderOptions.allHeaders);
             * ```
             */
            export enum HeaderOptions{
                /**
                 * Includes neither column nor row headers when export range data to HTML.
                 */
                noHeaders= 0,
                /**
                 * Includes selected row headers when export range data to HTML.
                 */
                rowHeaders= 1,
                /**
                 * Includes selected column headers when export range data to HTML.
                 */
                columnHeaders= 2,
                /**
                 * Includes selected headers when export range data to HTML.
                 */
                allHeaders= 3
            }

            /**
             * Specifies the horizontal alignment.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the HorizontalAlign type.
             * var style = new GC.Spread.Sheets.Style();
             * style.font = "8pt Arial";
             * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
             * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
             * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(1,1).value("B2");
             * ```
             */
            export enum HorizontalAlign{
                /**
                 *  Indicates that the cell content is left-aligned.
                 */
                left= 0,
                /**
                 *  Indicates that the cell content is centered.
                 */
                center= 1,
                /**
                 *  Indicates that the cell content is right-aligned.
                 */
                right= 2,
                /**
                 *  Indicates that the horizontal alignment is based on the value type.
                 */
                general= 3,
                /**
                 *  Indicates that the cell content is center across selection.
                 */
                centerContinuous= 4,
                /**
                 *  Indicates that each 'word' in each line of text inside the cell is evenly distributed across the width of the cell, with flush right and left margins.
                 */
                distributed= 5,
                /**
                 *  Indicates that in each line of a paragraph within the cell, the left-most character is aligned with the left margin and the right-most character is flush with the right margin.
                 */
                justify= 6
            }

            /**
             * Specifies the horizontal position of the cell or column in the component.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the HorizontalPosition enumeration.
             * activeSheet.setActiveCell(10,5);
             * activeSheet.showCell(10, 5, GC.Spread.Sheets.VerticalPosition.top, GC.Spread.Sheets.HorizontalPosition.center);
             * ```
             */
            export enum HorizontalPosition{
                /**
                 *  Positions the cell or column to the left.
                 */
                left= 0,
                /**
                 *  Positions the cell or column in the center.
                 */
                center= 1,
                /**
                 *  Positions the cell or column to the right.
                 */
                right= 2,
                /**
                 *  Positions the cell or column to the nearest edge.
                 */
                nearest= 3
            }

            /**
             * Present the way that user open the hyperlinked document. Default is blank.
             * @enum {number}
             */
            export enum HyperlinkTargetType{
                /**
                 * Opens the hyperlinked document in a new window or tab.
                 */
                blank= 0,
                /**
                 * Opens the hyperlinked document in the same frame where the user clicked.
                 */
                self= 1,
                /**
                 * Opens the hyperlinked document in the parent frame.
                 */
                parent= 2,
                /**
                 * Opens the hyperlinked document in the full body of the window.
                 */
                top= 3
            }

            /**
             * Describe the position of the icon base on a cell.
             * @enum {number}
             */
            export enum IconPosition{
                /**
                 * Indicates that the icon is on left side of the cell
                 */
                left= 0,
                /**
                 * Indicates that the icon is on right side of the cell
                 */
                right= 1,
                /**
                 * Indicates that the icon is on left side of the text in the cell
                 */
                leftOfText= 2,
                /**
                 * Indicates that the icon is on right side of the text in the cell
                 */
                rightOfText= 3,
                /**
                 * Indicates that the icon is on the left outside of the cell
                 */
                outsideLeft= 4,
                /**
                 * Indicates that the icon is on the right outside of the cell
                 */
                outsideRight= 5
            }

            /**
             * Defines the background image layout.
             * @enum {number}
             * @example
             * ```javascript
             * var rowImage = "./css/images/quarter1.png";
             * sheet.getCell(1, -1).backgroundImage(rowImage);
             * sheet.getCell(1, -1).backgroundImageLayout(GC.Spread.Sheets.ImageLayout.center);
             * ```
             */
            export enum ImageLayout{
                /** Specifies that the background image fills the area.
                 * @type {number}
                 */
                stretch= 0,
                /** Specifies that the background image displays in the center of the area.
                 * @type {number}
                 */
                center= 1,
                /** Specifies that the background image displays in the area with its original aspect ratio.
                 * @type {number}
                 */
                zoom= 2,
                /** Specifies that the background image displays in the upper left corner of the area with its original size.
                 * @type {number}
                 */
                none= 3
            }

            /**
             * Defines the IME mode to control the state of the Input Method Editor (IME).
             * @enum {number}
             * @deprecated This enum currently only works in Internet Explorer.
             * @example
             * ```javascript
             * //This example uses the imeMode method.
             * activeSheet.getRange(-1, 2, -1, 1).imeMode(GC.Spread.Sheets.ImeMode.auto);
             * ```
             */
            export enum ImeMode{
                /**
                 * No change is made to the current input method editor state.
                 */
                auto= 1,
                /** All characters are entered through the IME. Users can still deactivate the IME.
                 */
                active= 2,
                /**
                 * All characters are entered without IME. Users can still activate the IME.
                 */
                inactive= 4,
                /**
                 * The input method editor is disabled and may not be activated by the user.
                 */
                disabled= 0
            }

            /**
             * Specifies paste the direction of the insertion finger.
             * @enum {number}
             */
            export enum InsertShiftCell{
                /** Specifies paste the direction right.
                 * @type {number}
                 */
                right= 0,
                /** Specifies paste the direction down.
                 * @type {number}
                 */
                down= 1
            }

            /**
             * Identifies which operation was invalid.
             * @enum {number}
             * @example
             * ```javascript
             * //This example gets the invalid type.
             * activeSheet.getCell(1,1).locked(true);
             * activeSheet.options.isProtected = true;
             * activeSheet.getCell(1,1).value(2);
             * //Bind event
             * activeSheet.bind(GC.Spread.Sheets.Events.InvalidOperation, function (e, info) {
             *         alert("Message (" + info.invalidType + ")");
             * });
             * ```
             */
            export enum InvalidOperationType{
                /**
                 * Specifies the formula is invalid.
                 */
                setFormula= 0,
                /**
                 * Specifies the copy paste is invalid.
                 */
                copyPaste= 1,
                /**
                 * Specifies the drag fill is invalid.
                 */
                dragFill= 2,
                /**
                 * Specifies the drag drop is invalid.
                 */
                dragDrop= 3,
                /**
                 * Specifies the insert row is invalid.
                 */
                changePartOfArrayFormula= 4,
                /**
                 * Specifies the changed sheet name is invalid.
                 */
                changeSheetName= 5,
                /**
                 * Specifies the table action is invalid.
                 */
                table= 6,
                /**
                 * Specifies the add filter is invalid.
                 */
                filter= 7,
                /**
                 * Specifies sort is invalid.
                 */
                sort= 9,
                /**
                 * Specifies the change affect existed pivot table.
                 */
                pivotTable= 10,
                /**
                 * Specifies the pivotTable change will overlap the valued cell.
                 */
                ptOverlapValue= 11,
                /**
                 * Specifies the group can expand and collapse when sheet is unprotected.
                 */
                groupProtected= 12,
                /**
                 * Specifies edit value when cell is locked and sheet is protected.
                 */
                editProtected= 13,
                /**
                 * Specifies cut value when cell is locked and sheet is protected.
                 */
                cutProtected= 14,
                /**
                 * Specifies the value to be set as a custom name is valid.
                 */
                customName= 15,
                /**
                 * Specifies a data object cell type can't insert a data to its adjacent cells on the right.
                 */
                dataObjectCellTypeInsertData= 16,
                /**
                 * The specific event type that is triggered when the size of a file uploaded via the file picker exceeds the limit.
                 */
                sizeLimitExceeded= 18
            }

            /**
             * Specifies the cell label position.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets cell padding, alignment, and other options.
             * var type = new GC.Spread.Sheets.Style();
             * type.watermark = "User name";
             * type.cellPadding = "20";
             * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
             * activeSheet.setStyle(0, 1, type);
             * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * activeSheet.getRange(-1, 1, -1, 1).width(150);
             * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
             * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
             * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
             * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
             * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * ```
             */
            export enum LabelAlignment{
                /**
                 *  Indicates that the cell label position is top-left.
                 */
                topLeft= 0,
                /**
                 *  Indicates that the cell label position is top-center.
                 */
                topCenter= 1,
                /**
                 *  Indicates that the cell label position is top-right.
                 */
                topRight= 2,
                /**
                 *  Indicates that the cell label position is bottom-left.
                 */
                bottomLeft= 3,
                /**
                 *  Indicates that the cell label position is bottom-center.
                 */
                bottomCenter= 4,
                /**
                 *  Indicates that the cell label position is bottom-right.
                 */
                bottomRight= 5
            }

            /**
             * Specifies the cell label visibility.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets label options for the watermark.
             * var type = new GC.Spread.Sheets.Style();
             * type.watermark = "User name";
             * type.cellPadding = "20";
             * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
             * activeSheet.setStyle(0, 1, type);
             * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * activeSheet.getRange(-1, 1, -1, 1).width(150);
             * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
             * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
             * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
             * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
             * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
             * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
             * ```
             */
            export enum LabelVisibility{
                /**
                 *  Specifies to always show the watermark in the padding area and not to show the watermark in the cell area, regardless of the cell value.
                 */
                visible= 0,
                /**
                 *  Specifies to not show the watermark in the padding area, but to show the watermark in the cell area based on a value condition.
                 */
                hidden= 1,
                /**
                 *  Specifies to show the watermark in the padding area when the cell has a value or to show the watermark in the cell area if the cell does not have a value.
                 */
                auto= 2
            }

            /**
             * Specifies the layout direction.
             * @enum {number}
             */
            export enum LayoutDirection{
                /**
                 * Specifies layout direction is horizontal.
                 */
                horizontal= 0,
                /**
                 * Specifies layout direction is vertical.
                 */
                vertical= 1
            }

            /**
             * Specifies the layout display.
             * @enum {number}
             */
            export enum LayoutDisplayAs{
                /**
                 * Specifies layout display is inline.
                 */
                inline= 0,
                /**
                 * Specifies layout display is popup
                 */
                popup= 1,
                /**
                 * Specifies layout display is tree
                 */
                tree= 2
            }

            /**
             * Specifies the line drawing style for the border.
             * @enum {number}
             * @example
             * ```javascript
             * //This example creates a border.
             * var border = new GC.Spread.Sheets.LineBorder
             * border.color = "#7FFFD4";
             * border.style = GC.Spread.Sheets.LineStyle.double;
             * var cell = activeSheet.getCell(1, 1, GC.Spread.Sheets.SheetArea.viewport);
             * cell.borderLeft(border);
             * ```
             */
            export enum LineStyle{
                /**
                 * Indicates a border line without a style.
                 */
                empty= 0,
                /**
                 *  Indicates a border line with a solid thin line.
                 */
                thin= 1,
                /**
                 *  Indicates a medium border line with a solid line.
                 */
                medium= 2,
                /**
                 *  Indicates a border line with dashes.
                 */
                dashed= 3,
                /**
                 *  Indicates a border line with dots.
                 */
                dotted= 4,
                /**
                 *  Indicates a thick border line with a solid line.
                 */
                thick= 5,
                /**
                 *  Indicates a double border line.
                 */
                double= 6,
                /**
                 *  Indicates a border line with all dots.
                 */
                hair= 7,
                /**
                 *  Indicates a medium border line with dashes.
                 */
                mediumDashed= 8,
                /**
                 *  Indicates a border line with dash-dot.
                 */
                dashDot= 9,
                /**
                 *  Indicates a medium border line with dash-dot-dot.
                 */
                mediumDashDot= 10,
                /**
                 *  Indicates a border line with dash-dot-dot.
                 */
                dashDotDot= 11,
                /**
                 *  Indicates a medium border line with dash-dot-dot.
                 */
                mediumDashDotDot= 12,
                /**
                 *  Indicates a slanted border line with dash-dot.
                 */
                slantedDashDot= 13
            }

            /**
             * Represent the list style type of rich text list item
             * @enum {number}
             */
            export enum ListType{
                /**
                 * Indicates the unordered list item, list style type is disc.
                 */
                disc= 0,
                /**
                 * Indicates the unordered list item, list style type is circle.
                 */
                circle= 1,
                /**
                 * Indicates the unordered list item, list style type is square.
                 */
                square= 2,
                /**
                 * Indicates the ordered list item, list style type is decimal.
                 */
                decimal= 3,
                /**
                 * Indicates the ordered list item, list style type is lowerAlpha.
                 */
                lowerAlpha= 4,
                /**
                 * Indicates the ordered list item, list style type is upperAlpha.
                 */
                upperAlpha= 5,
                /**
                 * Indicates the ordered list item, list style type is lowerRoman.
                 */
                lowerRoman= 6,
                /**
                 * Indicates the ordered list item, list style type is upperRoman.
                 */
                upperRoman= 7,
                /**
                 * Indicates the ordered list item, list style type is lowerGreek.
                 */
                lowerGreek= 8
            }

            /**
             * Specifies the context type of mention users happened.
             * @enum {string}
             */
            export enum MentionContextType{
                /**
                 * Specifies that the mention happened in the threded comment.
                 */
                threadedComment= "threadedComment"
            }

            /**
             * Change display mode when date/number data width longer than column width.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the NumbersFitMode enumeration.
             * spread.options.numbersFitMode = GC.Spread.Sheets.NumbersFitMode.overflow;
             * ```
             */
            export enum NumbersFitMode{
                /** Indicates replacing data content with "###" and show tip.
                 * @type {number}
                 */
                mask= 0,
                /** Indicates display data content as a string, if next cell is empty, overflow the content.
                 * @type {number}
                 */
                overflow= 1
            }

            /**
             * The open mode for open sjs function.
             * @enum {number}
             */
            export enum OpenMode{
                /**
                 * normal open mode, without lazy and incremental. When opening file, UI and UI event could be refreshed and responsive at specific time points.
                 */
                normal= 0,
                /**
                 * lazy open mode. When opening file, only the active sheet will be loaded directly. Other sheets will be loaded only when they are be used.
                 */
                lazy= 1,
                /**
                 * incremental open mode. When opening file, UI and UI event could be refreshed and responsive directly.
                 */
                incremental= 2
            }

            /**
             * Specifies the operation options mathematically combine values between the copy and paste ranges
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the PasteOperationOptions enumeration.
             * activeSheet.getCell(0,0).value("1");
             * activeSheet.setSelection(0, 0, 1, 1);
             * spread.commandManager().execute({cmd:"copy", sheetName: activeSheet.name()});
             * activeSheet.setSelection(0, 0, 3, 3);
             * spread.commandManager().execute(
             *     {
             *         cmd:"paste",
             *         sheetName: activeSheet.name(),
             *         pasteSpecialOptions: {
             *             operationOptions: GC.Spread.Sheets.PasteOperationOptions.add
             *         }
             *     }
             * );
             * ```
             */
            export enum PasteOperationOptions{
                /**
                 * Paste the contents of the copy ranges without an operation.
                 */
                none= 0,
                /**
                 * Add the values in the copy ranges to the values in the paste ranges.
                 */
                add= 1,
                /**
                 * Subtract the values in the copy ranges from the values in the paste ranges.
                 */
                subtract= 2,
                /**
                 * Multiply the values in the paste ranges by the values in the copy ranges.
                 */
                multiply= 3,
                /**
                 * Divide the values in the paste ranges by the values in the copy ranges.
                 */
                divide= 4,
                /**
                 * Transform the values in the paste ranges by the formula in the copy ranges.
                 */
                transform= 5
            }

            /**
             * Specifies the cell value type of dropdown list result.
             * @enum {number}
             */
            export enum PatternType{
                /**
                 * Specifies the pattern type is solid.
                 */
                solid= 1,
                /**
                 * Specifies the pattern type is darkGray.
                 */
                darkGray= 2,
                /**
                 * Specifies the pattern type is mediumGray.
                 */
                mediumGray= 3,
                /**
                 * Specifies the pattern type is lightGray.
                 */
                lightGray= 4,
                /**
                 * Specifies the pattern type is gray125.
                 */
                gray125= 5,
                /**
                 * Specifies the pattern type is gray0625.
                 */
                gray0625= 6,
                /**
                 * Specifies the pattern type is darkHorizontal.
                 */
                darkHorizontal= 7,
                /**
                 * Specifies the pattern type is darkVertical.
                 */
                darkVertical= 8,
                /**
                 * Specifies the pattern type is darkDown.
                 */
                darkDown= 9,
                /**
                 * Specifies the pattern type is darkUp.
                 */
                darkUp= 10,
                /**
                 * Specifies the pattern type is darkGrid.
                 */
                darkGrid= 11,
                /**
                 * Specifies the pattern type is darkTrellis.
                 */
                darkTrellis= 12,
                /**
                 * Specifies the pattern type is lightHorizontal.
                 */
                lightHorizontal= 13,
                /**
                 * Specifies the pattern type is lightVertical.
                 */
                lightVertical= 14,
                /**
                 * Specifies the pattern type is lightDown.
                 */
                lightDown= 15,
                /**
                 * Specifies the pattern type is lightUp.
                 */
                lightUp= 16,
                /**
                 * Specifies the pattern type is lightGrid.
                 */
                lightGrid= 17,
                /**
                 * Specifies the pattern type is lightTrellis.
                 */
                lightTrellis= 18
            }

            /**
             * Defines the type of action that raised the RangeChanged event.
             * @enum {number}
             */
            export enum RangeChangedAction{
                /**
                 * Indicates drag drop undo action.
                 */
                dragDrop= 0,
                /**
                 * Indicates drag fill undo action.
                 */
                dragFill= 1,
                /**
                 * Indicates clear range value undo action.
                 */
                clear= 2,
                /**
                 * Indicates paste undo action.
                 */
                paste= 3,
                /**
                 * Indicates sorting a range of cells.
                 */
                sort= 4,
                /**
                 * Indicates setting a formula in a specified range of cells .
                 */
                setArrayFormula= 5,
                /**
                 * Indicates setting a formula in a specified range of cells .
                 */
                evaluateFormula= 6
            }

            /**
             * Specifies the formula reference style.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the reference style for the workbook.
             * workbook.options.referenceStyle = GC.Spread.Sheets.ReferenceStyle.r1c1;
             * ```
             */
            export enum ReferenceStyle{
                /**
                 * Indicates a1 style.
                 */
                a1= 0,
                /**
                 * Indicates r1c1 style.
                 */
                r1c1= 1
            }

            /**
             * Specifies the way resizing use.
             * @enum {number}
             * @example
             * ```javascript
             * //This example shows you how to specify the resize way.
             * spread.options.columnResizeMode = GC.Spread.Sheets.ResizeMode.split;
             * spread.options.rowResizeMode = GC.Spread.Sheets.ResizeMode.normal;
             * ```
             */
            export enum ResizeMode{
                /** Specifies use normal way to resize.
                 * @type {number}
                 */
                normal= 0,
                /** Specifies use split way to resize.
                 * @type {number}
                 */
                split= 1
            }

            /**
             * Specifies the drawing policy of the row or column when it is resized to zero.
             * @enum {number}
             * @example
             * ```javascript
             * //This example displays double grid lines for the column or row with zero width or height.
             * spread.options.resizeZeroIndicator = GC.Spread.Sheets.ResizeZeroIndicator.enhanced;
             * activeSheet.getRange(-1, 2, -1, 1).width(0);
             * activeSheet.getRange(1, -1, 1, -1).height(0);
             * ```
             */
            export enum ResizeZeroIndicator{
                /**
                 *  Uses the current drawing policy when the row or column is resized to zero.
                 */
                default= 0,
                /**
                 * Draws two short lines when the row or column is resized to zero.
                 */
                enhanced= 1
            }

            /**
             * Specifies the type of row and column state.
             * @enum {number}
             */
            export enum RowColumnStates{
                /** When mouse hover on the row and the column , its state include "hover" state.
                 * @type {number}
                 */
                hover= 1,
                /** When the data-validation conditional evaluate fail, its row and column state include "invalid" state.
                 * @type {number}
                 */
                invalid= 2,
                /** When the cell is editing, the cell row and column state include "edit" state.
                 * @type {number}
                 */
                edit= 8,
                /** When the row or the column is focus, its state include "active" state.
                 * @type {number}
                 */
                active= 16,
                /** When the cell is in the selection range, the cell row and column state include "selected" state.
                 * @type {number}
                 */
                selected= 32,
                /** When cell value is changed, cell row and column state include "dirty" state.
                 * @type {number}
                 */
                dirty= 64,
                /** When inserted a row, its state include "inserted" state. This state only support for row.
                 * @type {number}
                 */
                inserted= 128,
                /** When cell value is invalid formula string, cell state include "invalidFormula" state.
                 * @type {number}
                 */
                invalidFormula= 512
            }

            /**
             * Specifies the appearance of both vertical and horizontal scrollbar.
             * @enum {number}
             * @example
             * ```javascript
             * //This example displays a mobile scrollbar mode.
             * spread.options.scrollbarAppearance = GC.Spread.Sheets.ScrollbarAppearance.mobile;
             * ```
             */
            export enum ScrollbarAppearance{
                /** Specifies the excel-like classic scrollbars appearance.
                 * @type {number}
                 */
                skin= 0,
                /** Specifies the fashionable mobile scrollbars appearance which could be customized.
                 * @type {number}
                 */
                mobile= 1
            }

            /**
             * Specifies how users can select items in the control.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the selection policy.
             * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
             * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
             * ```
             */
            export enum SelectionPolicy{
                /**
                 * Allows users to only select single items.
                 */
                single= 0,
                /**
                 * Allows users to select single items and ranges of items, but not multiple ranges.
                 */
                range= 1,
                /**
                 * Allows users to select single items and ranges of items, including multiple ranges.
                 */
                multiRange= 2
            }

            /**
             * Specifies the smallest unit users or the application can select.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the unit type.
             * activeSheet.selectionUnit(GC.Spread.Sheets.SelectionUnit.row);
             * activeSheet.selectionPolicy(GC.Spread.Sheets.SelectionPolicy.range);
             * ```
             */
            export enum SelectionUnit{
                /**
                 * Indicates that the smallest unit that can be selected is a cell.
                 */
                cell= 0,
                /**
                 * Indicates that the smallest unit that can be selected is a row.
                 */
                row= 1,
                /**
                 * Indicates that the smallest unit that can be selected is a column.
                 */
                column= 2
            }

            /**
             * Specifies the sheet area.
             * @enum {number}
             * @example
             * ```javascript
             * //Creates log text describing which area in the sheet was clicked.
             * // Use web browser to see the console log text
             * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
             *         console.log("The column header was clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
             *         console.log("The row header was clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
             *         console.log("The corner header was clicked.");
             *     }
             *     console.log("Clicked column index: " + args.col);
             *     console.log("Clicked row index: " + args.row);
             * });
             * //Bind event
             * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
             *         console.log("The column header was double clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
             *         console.log("The row header was double clicked.");
             *     }
             *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
             *         console.log("The corner header was double clicked.");
             *     }
             *     console.log("Double clicked column index: " + args.col);
             *     console.log("Double clicked row index: " + args.row);
             * });
             * ```
             */
            export enum SheetArea{
                /**
                 * Indicates the sheet corner.
                 */
                corner= 0,
                /**
                 * Indicates the column header.
                 */
                colHeader= 1,
                /**
                 * Indicates the row header.
                 */
                rowHeader= 2,
                /**
                 * Indicates the viewport.
                 */
                viewport= 3
            }

            /**
             * Different states of sheet tab
             * The smaller the value, the higher the style priority for this state
             * @enum {number}
             */
            export enum SheetTabState{
                /**
                 * Indicates hover sheet tab state.
                 */
                hover= 1,
                /**
                 * Indicates selected sheet tab state.
                 */
                selected= 2,
                /**
                 * Indicates active sheet tab state.
                 */
                active= 4,
                /**
                 * Indicates protected sheet tab state.
                 */
                protected= 8,
                /**
                 * Indicates Normal sheet tab state.
                 */
                normal= 16
            }

            /**
             * Worksheet tab visible option.
             * @enum {number}
             */
            export enum SheetTabVisible{
                /**
                 * Sheet tab is hidden
                 */
                hidden= 0,
                /**
                 * Sheet tab is visible
                 */
                visible= 1,
                /**
                 * Sheet tab is veryHidden
                 */
                veryHidden= 2
            }

            /**
             * Present the type of sheet tab.
             * @enum {number}
             */
            export enum SheetType{
                /**
                 * Specifies the sheet tab is table sheet.
                 */
                tableSheet= 0,
                /**
                 * Specifies the sheet tab is gantt sheet.
                 */
                ganttSheet= 1,
                /**
                 * Specifies the sheet tab is report sheet.
                 */
                reportSheet= 3
            }

            /**
             * Defines how the resize tip is displayed.
             * @enum {number}
             * @example
             * ```javascript
             * //This example displays both resize tips.
             * spread.options.showDragDropTip = true;
             * spread.options.showDragFillTip = true;
             * spread.options.showScrollTip = GC.Spread.Sheets.ShowScrollTip.both;
             * spread.options.showResizeTip = GC.Spread.Sheets.ShowResizeTip.both;
             * ```
             */
            export enum ShowResizeTip{
                /** Specifies that no resize tip is displayed.
                 * @type {number}
                 */
                none= 0,
                /** Specifies that only the horizontal resize tip is displayed.
                 * @type {number}
                 */
                column= 1,
                /** Specifies that only the vertical resize tip is displayed.
                 * @type {number}
                 */
                row= 2,
                /** Specifies that horizontal and vertical resize tips are displayed.
                 * @type {number}
                 */
                both= 3
            }

            /**
             * Specifies how the scroll tip is displayed.
             * @enum {number}
             * @example
             * ```javascript
             * //This example displays both resize tips.
             * spread.options.showDragDropTip = true;
             * spread.options.showDragFillTip = true;
             * spread.options.showScrollTip = GC.Spread.Sheets.ShowScrollTip.both;
             * spread.options.showResizeTip = GC.Spread.Sheets.ShowResizeTip.both;
             * ```
             */
            export enum ShowScrollTip{
                /** Specifies that no scroll tip is displayed.
                 * @type {number}
                 */
                none= 0,
                /** Specifies that only the horizontal scroll tip is displayed.
                 * @type {number}
                 */
                horizontal= 1,
                /** Specifies that only the vertical scroll tip is displayed.
                 * @type {number}
                 */
                vertical= 2,
                /** Specifies that horizontal and vertical scroll tips are displayed.
                 * @type {number}
                 */
                both= 3
            }

            /**
             * Specifies the type of sorting to perform.
             * @enum {number}
             * @example
             * ```javascript
             * //This example sets the sort order of the items in the slicer.
             * //create a table
             * datas = [
             *     ["1", "NewYork", "1968/6/8", "80", "180"],
             *     ["4", "NewYork", "1972/7/3", "72", "168"],
             *     ["4", "NewYork", "1964/3/2", "71", "179"],
             *     ["5", "Washington", "1972/8/8","80", "171"],
             *     ["6", "Washington", "1986/2/2", "89", "161"],
             *     ["7", "Washington", "2012/2/15", "71", "240"]];
             * var table = activeSheet.tables.addFromDataSource("table1", 2, 2, datas);
             * dataColumns = ["Name", "City", "Birthday", "Weight", "Height"];
             * table.setColumnName(0, dataColumns[0]);
             * table.setColumnName(1, dataColumns[1]);
             * table.setColumnName(2, dataColumns[2]);
             * table.setColumnName(3, dataColumns[3]);
             * table.setColumnName(4, dataColumns[4]);
             * //create style
             * var style1 = new GC.Spread.Sheets.Slicers.SlicerStyles.light4();
             * //add a slicer to the sheet and return the slicer instance
             * var slicer = activeSheet.slicers.add("slicer1",table.name(),"Height");
             * //set slicer properties
             * slicer.position(new GC.Spread.Sheets.Point(100, 200));
             * slicer.sortState(GC.Spread.Sheets.SortState.descending);
             * slicer.style(style1);
             * ```
             */
            export enum SortState{
                /** Indicates the sorting is disabled.
                 * @type {number}
                 */
                none= 0,
                /** Indicates the sorting is ascending.
                 * @type {number}
                 */
                ascending= 1,
                /** Indicates the sorting is descending.
                 * @type {number}
                 */
                descending= 2
            }

            /**
             * Represents the storage data type.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the StorageType enumeration.
             * activeSheet.getCell(0,0).value("A1");
             * activeSheet.clear(0,0,3,3,GC.Spread.Sheets.SheetArea.viewport,GC.Spread.Sheets.StorageType.data);
             * ```
             */
            export enum StorageType{
                /**
                 *  Indicates the storage data type is pure value.
                 */
                data= 1,
                /**
                 *  Indicates the storage data type is style.
                 */
                style= 2,
                /**
                 *  Indicates the storage data type is comment.
                 */
                comment= 4,
                /**
                 *  Indicates the storage data type is tag.
                 */
                tag= 8,
                /**
                 *  Indicates the storage data type is sparkline.
                 */
                sparkline= 16,
                /**
                 *  Indicates the storage data type is the axis information.
                 */
                axis= 32,
                /**
                 *  Indicates the storage data type is data binding path.
                 */
                bindingPath= 64,
                /**
                 *  Indicates the storage data type is hyperlink.
                 */
                hyperlink= 256,
                /**
                 *  Indicates the storage data type is alternative text.
                 */
                altText= 512,
                /**
                 *  Indicates the storage data type is threaded comment.
                 */
                threadedComment= 1024
            }

            /**
             * Specifies the position of the tab strip relative to the workbook.
             * @enum {number}
             */
            export enum TabStripPosition{
                /**
                 * Specifies the position of the tab strip relative to the bottom of the workbook.
                 */
                bottom= 0,
                /**
                 * Specifies the position of the tab strip relative to the top of the workbook.
                 */
                top= 1,
                /**
                 * Specifies the position of the tab strip relative to the left of the workbook.
                 */
                left= 2,
                /**
                 * Specifies the position of the tab strip relative to the right of the workbook.
                 */
                right= 3
            }

            /**
             * Defines the type of the text decoration.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the TextDecorationType enumeration.
             * activeSheet.getCell(0, 0).textDecoration(GC.Spread.Sheets.TextDecorationType.underline);
             * activeSheet.getRange(1, -1, 1, -1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline);
             * activeSheet.getRange(-1, 1, -1, 1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.lineThrough | GC.Spread.Sheets.TextDecorationType.underline);
             * var style = new GC.Spread.Sheets.Style();
             * style.textDecoration = GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline;
             * activeSheet.setStyle(1, 1, style, GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(0, 0).value("Test");
             * activeSheet.getCell(1, 0).value("Test");
             * activeSheet.getCell(0, 1).value("Test");
             * ```
             */
            export enum TextDecorationType{
                /** Specifies to display a line below the text.
                 */
                underline= 1,
                /** Specifies to display a line through the text.
                 */
                lineThrough= 2,
                /** Specifies to display a line above the text.
                 */
                overline= 4,
                /** Specifies to display double line below the text.
                 */
                doubleUnderline= 8,
                /** Specifies normal text.
                 */
                none= 0
            }

            /**
             * Defines the type of the text direction.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the TextDirectionType enumeration.
             * activeSheet.getCell(0, 0).textDirection(GC.Spread.Sheets.TextDirectionType.rightToLeft);
             * activeSheet.getRange(1, -1, 1, -1).textDirection(GC.Spread.Sheets.TextDirectionType.rightToLeft);
             * var style = new GC.Spread.Sheets.Style();
             * style.textDirection = GC.Spread.Sheets.TextDirectionType.rightToLeft;
             * activeSheet.setStyle(1, 1, style, GC.Spread.Sheets.SheetArea.viewport);
             * ```
             */
            export enum TextDirectionType{
                /** Specifies the text direction is context dependent, which is determined by scanning the text for the first non-whitespace character: if it is a strong right-to-left character, the text direction is right-to-left; otherwise, the text direction is left-to-right.
                 */
                context= 0,
                /** Specifies the text direction is left-to-right in the cell, as in English.
                 */
                leftToRight= 1,
                /** Specifies the text direction is right-to-left in the cell, as in Arabic or Hebrew.
                 */
                rightToLeft= 2
            }

            /**
             * Specifies the scope of transform.
             * @enum {number}
             */
            export enum TransformScope{
                /**
                 * Each cell is considered as a transformation unit.
                 */
                byCell= 0,
                /**
                 * Each row is considered as a transformation unit.
                 */
                byRow= 1,
                /**
                 * Each column is considered as a transformation unit.
                 */
                byColumn= 2,
                /**
                 * The entire cell range is considered as one transformation unit.
                 */
                byRange= 3
            }

            /**
             * Worksheet used range
             * @enum {number}
             */
            export enum UsedRangeType{
                /**
                 * Sheet all used range type
                 */
                all= 4294967295,
                /**
                 * Sheet header used range
                 */
                axis= 1,
                /**
                 * Sheet cell style used range
                 */
                style= 2,
                /**
                 * Sheet row style used range
                 */
                rowStyle= 4,
                /**
                 * Sheet column style used range
                 */
                colStyle= 8,
                /**
                 * Sheet data used range
                 */
                data= 16,
                /**
                 * Sheet formula used range
                 */
                formula= 32,
                /**
                 * Sheet span used range
                 */
                span= 64,
                /**
                 * Sheet shape used range
                 */
                shape= 128,
                /**
                 * Sheet table used range
                 */
                table= 256,
                /**
                 * Sheet sparkLine used range
                 */
                sparkLine= 512,
                /**
                 * Sheet comment used range
                 */
                comment= 1024,
                /**
                 * Sheet slicer used range
                 */
                slicer= 2048,
                /**
                 * Sheet pivot table used range
                 */
                pivottable= 4096,
                /**
                 * Sheet filter used range
                 */
                filter= 8192,
                /**
                 * Sheet dataValidation used range
                 */
                dataValidation= 16384,
                /**
                 * Sheet conditionFormat used range
                 */
                conditionFormat= 32768,
                /**
                 * Sheet chart used range
                 */
                chart= 65536,
                /**
                 * Sheet picture used range
                 */
                picture= 131072,
                /**
                 * Sheet tag used range
                 */
                tag= 262144,
                /**
                 * Sheet data range used range
                 */
                dataRange= 524288,
                /**
                 * Sheet threaded comment used range
                 */
                threadedComment= 1048576
            }

            /**
             * Specifies cell value's type.
             * @enum {number}
             */
            export enum ValueType{
                /**
                 *  indicate normal value type.
                 */
                normal= 0,
                /**
                 *  indicate rich text value type.
                 */
                richText= 1
            }

            /**
             * Defines the type of the text vertAlign.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the VertAlign enumeration.
             * activeSheet.setValue(1, 1, {richText:[{style:{vertAlign:GC.Spread.Sheets.VertAlign.subscript},text:'SpreadJS'}]}, GC.Spread.Sheets.SheetArea.viewport);
             * ```
             */
            export enum VertAlign{
                /**
                 *  indicate normal text align.
                 */
                normal= 0,
                /**
                 *  indicate superscript.
                 */
                superscript= 1,
                /**
                 *  indicate subscript.
                 */
                subscript= 2
            }

            /**
             * Specifies the vertical alignment.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the VerticalAlign type.
             * var style = new GC.Spread.Sheets.Style();
             * style.font = "8pt Arial";
             * style.hAlign = GC.Spread.Sheets.HorizontalAlign.center;
             * style.vAlign = GC.Spread.Sheets.VerticalAlign.center;
             * activeSheet.setStyle(1,1,style,GC.Spread.Sheets.SheetArea.viewport);
             * activeSheet.getCell(1,1).value("B2");
             * ```
             */
            export enum VerticalAlign{
                /**
                 *  Indicates that the cell content is top-aligned.
                 */
                top= 0,
                /**
                 *  Indicates that the cell content is centered.
                 */
                center= 1,
                /**
                 *  Indicates that the cell content is bottom-aligned.
                 */
                bottom= 2,
                /**
                 *  Indicates that each line of text in a paragraph within the cell is evenly distributed across the height of the cell.
                 */
                justify= 3
            }

            /**
             * Specifies the vertical position of the cell or row in the component.
             * @enum {number}
             * @example
             * ```javascript
             * //This example uses the VerticalPosition enumeration.
             * activeSheet.setActiveCell(10,5);
             * activeSheet.showCell(10, 5, GC.Spread.Sheets.VerticalPosition.top, GC.Spread.Sheets.HorizontalPosition.center);
             * ```
             */
            export enum VerticalPosition{
                /**
                 *  Positions the cell or row at the top.
                 */
                top= 0,
                /**
                 *  Positions the cell or row in the center.
                 */
                center= 1,
                /**
                 *  Positions the cell or row at the bottom.
                 */
                bottom= 2,
                /**
                 *  Positions the cell or row at the nearest edge.
                 */
                nearest= 3
            }

            /**
             * Specifies the visual state.
             * @enum {number}
             */
            export enum VisualState{
                /**
                 * Indicates normal visual state.
                 */
                normal= 0,
                /**
                 * Indicates highlight visual state.
                 */
                highlight= 1,
                /**
                 * Indicates selected visual state.
                 */
                selected= 2,
                /**
                 * Indicates active visual state.
                 */
                active= 3,
                /**
                 * Indicates hover visual state.
                 */
                hover= 4
            }


            export class CellRange{
                /**
                 * Represents a cell range in a sheet.
                 * @class
                 * @param {GC.Spread.Sheets.Worksheet} sheet The sheet that contains this cell range.
                 * @param {number} row The row index of the cell.
                 * @param {number} col The column index of the cell.
                 * @param {number} [rowCount] The row count of the cell. If you do not provide this parameter, it defaults to `1`.
                 * @param {number} [colCount] The column count of the cell. If you do not provide this parameter, it defaults to `1`.
                 * @param {GC.Spread.Sheets.SheetArea} [sheetArea] The sheet area. If you do not provide this parameter, it defaults to `viewport`.
                 * If row is -1 and rowCount is -1, the range represents columns. For example, new GC.Spread.Sheets.CellRange(-1,4,-1,6) represents columns "E:J".
                 * If col is -1 and colCount is -1, the range represents rows. For example, new GC.Spread.Sheets.CellRange(4,-1,6,-1) represents rows "5:10".
                 */
                constructor(sheet: GC.Spread.Sheets.Worksheet,  row: number,  col: number,  rowCount?: number,  colCount?: number,  sheetArea?: GC.Spread.Sheets.SheetArea);
                /**
                 * Gets the starting column index.
                 * @type {number}
                 */
                col: number;
                /**
                 * Gets the column count.
                 * @type {number}
                 */
                colCount: number;
                /**
                 * Gets the starting row index.
                 * @type {number}
                 */
                row: number;
                /**
                 * Gets the row count.
                 * @type {number}
                 */
                rowCount: number;
                /**
                 * Gets the sheet that contains this cell range.
                 * @type {GC.Spread.Sheets.Worksheet}
                 */
                sheet: GC.Spread.Sheets.Worksheet;
                /**
                 * Gets the area that contains this cell range.
                 * @type {GC.Spread.Sheets.SheetArea}
                 */
                sheetArea: GC.Spread.Sheets.SheetArea;
                /**
                 * Gets or sets whether the cell can enter edit mode for editing.
                 * @param {boolean} [value] Set to `false` to disallow the cell enter edit mode for editing.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell can enter edit mode for editing; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(0, 0, 2, 3, GC.Spread.Sheets.SheetArea.viewport).allowEditInCell(false);
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).allowEditInCell(false);
                 * ```
                 */
                allowEditInCell(value?: boolean): any;
                /**
                 * Gets or sets the alternative text of the cell for screen readers.
                 * @param {string} value The alternative text of the cell.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the alternative text of the cell; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * var SpreadIcon = {
                 *     FolderOpen: '\ue685',
                 *     InfoFilled: '\ue718',
                 *     Library: '\ue69d',
                 *     NotebookFilled: '\uD800\uDC0F',
                 *     Browse: '\ue626'
                 * };
                 * activeSheet.getCell(1, 1).value(SpreadIcon.FolderOpen).altText("Folder Open Icon");
                 *
                 * // Besides plain text, the alternative text could also contain placeholder {value} or {formatted}, which represents cell value or cell formatted value.
                 * // For example, if the cell value is 1000, and the alt text is "Sales amount is {value}", the final accessible content should be "Sales amount is 1000".
                 * activeSheet.getCell(1, 1).value(1000).altText("Sales amount is {value}");
                 * ```
                 */
                altText(value?: any): any;
                /**
                 * Gets or sets the background color for the cell, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @param {string|undefined|GC.Spread.Sheets.IPatternFill|GC.Spread.Sheets.IGradientFill|GC.Spread.Sheets.IGradientPathFill} [value] The cell background color.
                 * @returns {string|undefined|GC.Spread.Sheets.IPatternFill|GC.Spread.Sheets.IGradientFill|GC.Spread.Sheets.IGradientPathFill|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background color; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backColor("pink");
                 * ```
                 */
                backColor(value?: string | GC.Spread.Sheets.IPatternFill | GC.Spread.Sheets.IGradientFill | GC.Spread.Sheets.IGradientPathFill): any;
                /**
                 * Gets or sets the background image for the cell.
                 * @param {string} [value] The cell background image.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background image; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).backgroundImage("images/example.jpg");
                 * ```
                 */
                backgroundImage(value?: string): any;
                /**
                 * Gets or sets the background image layout for the cell.
                 * @param {GC.Spread.Sheets.ImageLayout} [value] The cell background image layout.
                 * @returns {GC.Spread.Sheets.ImageLayout|GC.Spread.Sheets.CellRange} If no value is set, returns the cell background image layout; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * var layout = GC.Spread.Sheets.ImageLayout.stretch;
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).backgroundImageLayout(layout);
                 * ```
                 */
                backgroundImageLayout(value?: GC.Spread.Sheets.ImageLayout): any;
                /**
                 * Gets or sets the binding path for cell binding.
                 * @param {string} path The binding path for cell binding.
                 * @returns {string | GC.Spread.Sheets.CellRange} If no value is set, returns the binding path for cell binding; otherwise, returns the worksheet.
                 * @example
                 * ```javascript
                 * //This example uses the bindingPath method.
                 * var test = {name: "John"};
                 * activeSheet.getCell(0,0).bindingPath( "name");
                 * activeSheet.setDataSource(new GC.Spread.Sheets.Bindings.CellBindingSource(test));
                 * ```
                 */
                bindingPath(path?: string): any;
                /**
                 * Gets or sets the bottom border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell bottom border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell bottom border line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderBottom(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                borderBottom(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the left border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell left border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell left border line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).borderLeft(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                borderLeft(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the right border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell right border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell right border line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).borderRight(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                borderRight(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the top border of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell top border line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell top border line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("green", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).borderTop(new GC.Spread.Sheets.LineBorder("red", GC.Spread.Sheets.LineStyle.double));
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                borderTop(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the cellButtons of the cell.
                 * @param {Object[]} value the cellButtons of the cell.
                 * @returns {Object[] |GC.Spread.Sheets.CellRange} If no value is set, returns the cellButtons of the cell; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).cellButtons([caption:"Text"]]);
                 * ```
                 * @example
                 * ```javascript
                 * var cellButtons = activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).cellButtons();
                 * ```
                 */
                cellButtons(value?: GC.Spread.Sheets.ICellButton[]): any;
                /**
                 * Gets or sets the cell padding.
                 * @param {string} [value] The cell padding.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the value of the cell padding; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * // This example adds cell padding around the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * ```
                 */
                cellPadding(value?: string): any;
                /**
                 * Gets or sets the cell type of the cell.
                 * @param {GC.Spread.Sheets.CellTypes.Base} [value] The cell type.
                 * @returns {GC.Spread.Sheets.CellTypes.Base|GC.Spread.Sheets.CellRange} If no value is set, returns the cell type; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).cellType(new GC.Spread.Sheets.CellTypes.CheckBox());
                 * ```
                 */
                cellType(value?: GC.Spread.Sheets.CellTypes.Base): any;
                /**
                 * Clears the specified area.
                 * @param {GC.Spread.Sheets.StorageType} type The clear type.
                 */
                clear(type: GC.Spread.Sheets.StorageType): void;
                /**
                 * Gets or sets the comment for the cell.
                 * @param {GC.Spread.Sheets.Comments.Comment} value The comment to set in the cell.
                 * @returns {GC.Spread.Sheets.Comments.Comment | GC.Spread.Sheets.CellRange} If no value is set, returns the comment in the cell; otherwise, returns the cell range.
                 * @example
                 * ```javascript
                 * // This example creates a cell comment.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("yellow");
                 * comment.foreColor("green");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * ```
                 */
                comment(value?: GC.Spread.Sheets.Comments.Comment): any;
                /**
                 * Gets or sets the default value of the cell.
                 * @param {Object} [value] The cell default value.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the cell default value; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).defaultValue(10);
                 * activeSheet.getCell(1,3).defaultValue("=LastYear+1000");
                 * ```
                 */
                defaultValue(value?: any): any;
                /**
                 * Gets or sets the diagonalDown of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell diagonalDown line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell diagonalDown line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalDown(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                diagonalDown(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the diagonalUp of the cell.
                 * @param {GC.Spread.Sheets.LineBorder} [value] The cell diagonalUp line.
                 * @returns {GC.Spread.Sheets.LineBorder|GC.Spread.Sheets.CellRange} If no value is set, returns the cell diagonalUp line; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).diagonalUp(new GC.Spread.Sheets.LineBorder("blue", GC.Spread.Sheets.LineStyle.mediumDashed));
                 * ```
                 */
                diagonalUp(value?: GC.Spread.Sheets.LineBorder): any;
                /**
                 * Gets or sets the dropDowns of the cell.
                 * @param {Object[]} [value] the dropDowns of the cell.
                 * @returns {Object[] |GC.Spread.Sheets.CellRange} If no value is set, returns the dropDowns of the cell; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).dropDowns([caption:"Text"]]);
                 * ```
                 * @example
                 * ```javascript
                 * var dropDowns = activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).dropDowns();
                 * ```
                 */
                dropDowns(value?: GC.Spread.Sheets.IDropdown[]): any;
                /**
                 * Gets or sets the font for the cell, such as "normal normal normal 20px/normal Arial".
                 * @param {string} [value] The cell font.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's font; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).font("12pt Arial");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).font("8pt Arial");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).font("12pt Arial");
                 * ```
                 */
                font(value?: string): any;
                /**
                 * Gets or sets the fontFamily for the cell, such as "Arial".
                 * @param {string} [value] The cell fontFamily.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's fontFamily; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).fontFamily("Arial");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).fontFamily("Arial");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).fontFamily("Arial");
                 * ```
                 */
                fontFamily(value?: string): any;
                /**
                 * Gets or sets the fontSize for the cell, such as "16px".
                 * @param {string} [value] The cell fontSize.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's fontSize; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).fontSize("16px");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).fontSize("16px");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).fontSize("16px");
                 * ```
                 */
                fontSize(value?: string): any;
                /**
                 * Gets or sets the fontStyle for the cell, such as "italic".
                 * @param {string} [value] The cell fontStyle.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's fontStyle; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).fontStyle("italic");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).fontStyle("italic");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).fontStyle("italic");
                 * ```
                 */
                fontStyle(value?: string): any;
                /**
                 * Gets or sets the fontWeight for the cell, such as "bold".
                 * @param {string} [value] The cell fontWeight.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's fontWeight; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).fontWeight("bold");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).fontWeight("bold");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).fontWeight("bold");
                 * ```
                 */
                fontWeight(value?: string): any;
                /**
                 * Gets or sets the color of the text in the cell, such as "red", "#FFFF00", "rgb(255,0,0)", "Accent 5", and so on.
                 * @param {string} [value] The color of the text.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell foreground color; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).foreColor("blue");
                 * ```
                 */
                foreColor(value?: string): any;
                /**
                 * Gets or sets the formatter for the cell.
                 * @param {string | GC.Spread.Formatter.GeneralFormatter} [value] The cell formatter string or object.
                 * @returns {string | GC.Spread.Formatter.GeneralFormatter |GC.Spread.Sheets.CellRange} If no value is set, returns the cell formatter string or object; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).formatter("0.000%");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1, -1).formatter("0.000%");
                 * activeSheet.getCell(1,0).value("2");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(-1, 0).formatter("0.000%");
                 * activeSheet.getCell(0,0).value("2");
                 * ```
                 */
                formatter(value?: string | GC.Spread.Formatter.GeneralFormatter): any;
                /**
                 * Gets or sets the formula for the cell.
                 * @param {string} [value] The cell formula.
                 * @param {boolean} [autoAdjustReference] Whether to adjust the relative reference of each cell. It's false if omitted.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell formula; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(0,2).formula("DATEDIF(DATE(2003,1,1),DATE(2016,1,1),\"Y\")");
                 * activeSheet.getRange(2,2,100,1).formula("=A3+$A$1"); // all the cells are "=A3+$A$1"
                 * activeSheet.getRange(2,2,100,1).formula("=A3+$A$1", true); // the first cell is "=A3+$A$1", the second cell is "=A4+$A$1", ...
                 * ```
                 */
                formula(value?: string,  autoAdjustReference?: boolean): any;
                /**
                 * Gets or sets the horizontal alignment of the contents of the cell.
                 * @param {GC.Spread.Sheets.HorizontalAlign} [value] The horizontal alignment.
                 * @returns {GC.Spread.Sheets.HorizontalAlign|GC.Spread.Sheets.CellRange} If no value is set, returns the horizontal alignment of the contents of the cell; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).hAlign(GC.Spread.Sheets.HorizontalAlign.right);
                 * ```
                 */
                hAlign(value?: GC.Spread.Sheets.HorizontalAlign): any;
                /**
                 * Gets or sets the height of the row in pixels.
                 * @param {number} [value] The cell row height.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the row height; otherwise, returns the row.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(90);
                 * ```
                 */
                height(value?: number): any;
                /**
                 * Gets or sets whether the cell formula is visible.
                 * @param {boolean} [value] Set to `true` to hidden the cell formula.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell formula is hidden; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).hidden(true);
                 * ```
                 */
                hidden(value?: boolean): any;
                /**
                 * Gets or sets the imeMode of the cell.
                 * @param {GC.Spread.Sheets.ImeMode} [value] The cell imeMode.
                 * @deprecated This property currently only works in Internet Explorer.
                 * @returns {GC.Spread.Sheets.ImeMode|GC.Spread.Sheets.CellRange} If no value is set, returns the cell imeMode; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(0, 0).imeMode(GC.Spread.Sheets.ImeMode.disabled);
                 * //or
                 * var style = new GC.Spread.Sheets.Style();
                 * style.imeMode = GC.Spread.Sheets.ImeMode.disabled;
                 * activeSheet.setStyle(0, 0, style);
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2, -1, 1, -1).imeMode(GC.Spread.Sheets.ImeMode.active);
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 2, -1, 1).imeMode(GC.Spread.Sheets.ImeMode.auto);
                 * ```
                 */
                imeMode(value?: GC.Spread.Sheets.ImeMode): any;
                /**
                 * Gets or sets whether the cell's text is vertical.
                 * @param {boolean} [value] Set to `true` to have the cell's text vertical.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell's text vertical; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).isVerticalText(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).isVerticalText(true);
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).isVerticalText(true);
                 * ```
                 */
                isVerticalText(value?: boolean): any;
                /**
                 * Gets or sets the cell label options.
                 * @param {Object} [value] The cell label options.
                 * @param {GC.Spread.Sheets.LabelAlignment} [value.alignment] - The cell label position.
                 * @param {GC.Spread.Sheets.LabelVisibility} [value.visibility] - The cell label visibility.
                 * @param {string} [value.font] - The cell label font.
                 * @param {string} [value.foreColor] - The cell label forecolor.
                 * @param {string} [value.margin] - The cell label margin.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the value of the cell label options; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * //This example sets label options for the watermark.
                 * var type = new GC.Spread.Sheets.Style();
                 * type.watermark = "User name";
                 * type.cellPadding = "20";
                 * type.labelOptions = {alignment:GC.Spread.Sheets.LabelAlignment.topLeft, visibility: GC.Spread.Sheets.LabelVisibility.visible};
                 * activeSheet.setStyle(0, 1, type);
                 * activeSheet.getRange(0, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * activeSheet.getRange(-1, 1, -1, 1).width(150);
                 * var combo = new GC.Spread.Sheets.CellTypes.ComboBox();
                 * combo.items([{ text: "Oranges", value: "11k" }, { text: "Apples", value: "15k" }, { text: "Grape", value: "100k" }]);
                 * combo.editorValueType(GC.Spread.Sheets.CellTypes.EditorValueType.text);
                 * activeSheet.setCellType(2, 1, combo, GC.Spread.Sheets.SheetArea.viewport);
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).watermark("ComboBox Cell Type").cellPadding('10 10 20 10');
                 * activeSheet.getCell(2, 1, GC.Spread.Sheets.SheetArea.viewport).labelOptions({alignment: GC.Spread.Sheets.LabelAlignment.bottomCenter, foreColor: 'yellowgreen', font: 'bold 15px Arial'});
                 * activeSheet.getRange(2, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).height(60);
                 * ```
                 */
                labelOptions(value?: GC.Spread.Sheets.ILabelOptions): any;
                /**
                 * Gets or sets whether the cell is locked. When the sheet is protected, the locked cell cannot be edited.
                 * @param {boolean} [value] Set to `true` to lock the cell.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell is locked; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).locked(true);
                 * ```
                 */
                locked(value?: boolean): any;
                /**
                 * Gets or sets the cell mask.
                 * @param {Object} [value] The cell mask.
                 * @param {string} [value.pattern] - The cell mask pattern.
                 * @param {string} [value.placeholder] - The cell mask placeholder.
                 * @param {string} [value.excludeLiteral] - The cell mask excludeLiteral.
                 * @param {string} [value.excludePlaceholder] - The cell mask excludePlaceholder.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the value of the cell mask; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * //This example sets mask of the cell.
                 * var style = new GC.Spread.Sheets.Style();
                 * var pattern = "[a0_]{8}";
                 * style.pattern = pattern;
                 * activeSheet.setStyle(0, 1, style);
                 * activeSheet.getCell(0, 1, GC.Spread.Sheets.SheetArea.viewport).mask({ pattern: pattern});
                 * ```
                 */
                mask(value?: GC.Spread.Sheets.IMaskType): any;
                /**
                 * Gets or sets the quote prefix of the cell.
                 * @param {boolean} [value] The value of the quote Prefix.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns the content of the quotePrefix; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).quotePrefix(true);
                 * ```
                 */
                quotePrefix(value?: boolean): any;
                /**
                 * Gets or sets whether the row or column can be resized by the user.
                 * @param {boolean} [value] Set to `true` to make the row resizable.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the user can resize the row; otherwise, returns the row or column.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1. GC.Spread.Sheets.SheetArea.viewport).resizable(true);
                 * ```
                 */
                resizable(value?: boolean): any;
                /**
                 * Sets the border for the specified area.
                 * @param {GC.Spread.Sheets.LineBorder} border The border line.
                 * @param {Object} option Determines which part of the cell range to set, the option object contains {all:true, left:true, top:true, right:true, bottom:true, diagonalUp:true, diagonalDown:true, outline:true,inside:true, innerHorizontal:true, innerVertical:true}
                 * @param {boolean} [options.all] - all border.
                 * @param {boolean} [options.left] - left border.
                 * @param {boolean} [options.top] - top border.
                 * @param {boolean} [options.right] - right border.
                 * @param {boolean} [options.bottom] - bottom border.
                 * @param {boolean} [options.outline] - outline border.
                 * @param {boolean} [options.inside] - inside border.
                 * @param {boolean} [options.innerHorizontal] - innerHorizontal border.
                 * @param {boolean} [options.innerVertical] - innerVertical border.
                 * @param {boolean} [options.diagonalUp] - diagonalUp border.
                 * @param {boolean} [options.diagonalDown] - diagonalDown border.
                 * @example
                 * ```javascript
                 * //This example creates borders.
                 * sheet.getCell(1, 1).borderTop(new GC.Spread.Sheets.LineBorder("#F0F8FF",GC.Spread.Sheets.LineStyle.double));
                 * sheet.getCell(1, 1).borderLeft(new GC.Spread.Sheets.LineBorder("#F0F8FF",GC.Spread.Sheets.LineStyle.hair));
                 * sheet.getCell(1, 1).borderRight(new GC.Spread.Sheets.LineBorder("#FAEBD7",GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getCell(1, 1).borderBottom(new GC.Spread.Sheets.LineBorder("#00FFFF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderTop(new GC.Spread.Sheets.LineBorder("#F0FFFF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderLeft(new GC.Spread.Sheets.LineBorder("#F5F5DC",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("#FF02FF", GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getRange(-1, 5, -1, 1, GC.Spread.Sheets.SheetArea.viewport).borderBottom (new GC.Spread.Sheets.LineBorder("#FFE4C4",GC.Spread.Sheets.LineStyle.thin));
                 * sheet.getRange(2, 2, 2, 2, GC.Spread.Sheets.SheetArea.viewport).setBorder(new GC.Spread.Sheets.LineBorder("#8A2BE2",GC.Spread.Sheets.LineStyle.thick), { all:true });
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderTop( new GC.Spread.Sheets.LineBorder("#A52A2A",GC.Spread.Sheets.LineStyle.mediumDashed));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderLeft( new GC.Spread.Sheets.LineBorder("#FF02FF",GC.Spread.Sheets.LineStyle.medium));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderRight(new GC.Spread.Sheets.LineBorder("#5F9EA0", GC.Spread.Sheets.LineStyle.dashDot));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).borderBottom(new GC.Spread.Sheets.LineBorder("#6495ED",GC.Spread.Sheets.LineStyle.dotted));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).diagonalUp(new GC.Spread.Sheets.LineBorder("#FF02FF",GC.Spread.Sheets.LineStyle.dotted));
                 * sheet.getRange(5, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).diagonalDown(new GC.Spread.Sheets.LineBorder("#6495ED",GC.Spread.Sheets.LineStyle.medium));
                 * ```
                 */
                setBorder(border: GC.Spread.Sheets.LineBorder,  option: GC.Spread.Sheets.ISetBorderOptions): void;
                /**
                 * Sets the style for the cell.
                 * @param {GC.Spread.Sheets.Style} value The style.
                 * @example
                 * ```javascript
                 * let style = new GC.Spread.Sheets.Style();
                 * style.formatter = "#,##0.00";
                 * sheet.getRange("E5:F10").setStyle(style);
                 * sheet.getRange(0,0,4,4).setStyle(style);
                 * ```
                 */
                setStyle(value: GC.Spread.Sheets.Style): void;
                /**
                 * Sets the style name for the cell.
                 * @param {string} value The style name.
                 * @example
                 * ```javascript
                 * let style = new GC.Spread.Sheets.Style();
                 * style.name = "bold_style";
                 * style.font = "bold 12px sans-serif";
                 * sheet.addNamedStyle(style);
                 * sheet.getRange("A1:D3").setStyleName("bold_style");
                 * sheet.getRange(5,5,10,10).setStyleName("bold_style");
                 * ```
                 */
                setStyleName(value: string): void;
                /**
                 * Gets or sets the cell ellipsis property.
                 * @param {boolean} [value] Set to `true` to have the cell test show ellipsis.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell ellipsis property.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).showEllipsis(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).showEllipsis(true);
                 * ```
                 */
                showEllipsis(value?: boolean): any;
                /**
                 * Gets or sets whether the cell shrinks the text to fit the cell size.
                 * @param {boolean} [value] Set to `true` to have the cell shrink text to fit.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell shrinks the text to fit; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(2,-1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).shrinkToFit(false);
                 * activeSheet.setText(2,0,"This is a test");
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).shrinkToFit(true);
                 * ```
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).shrinkToFit(true);
                 * ```
                 */
                shrinkToFit(value?: boolean): any;
                /**
                 * Gets or sets a value that indicates whether the user can set focus to the cell using the Tab key.
                 * @param {boolean} [value] Set to `true` to set focus to the cell using the Tab key.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the user can set focus to the cell using the Tab key; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).tabStop(false);
                 * activeSheet.getRange(1, -1, 1, -1, GC.Spread.Sheets.SheetArea.viewport).tabStop(false);
                 * activeSheet.getRange(-1, 1, -1, 1, GC.Spread.Sheets.SheetArea.viewport).tabStop(false);
                 * ```
                 */
                tabStop(value?: boolean): any;
                /**
                 * Gets or sets the tag for the cell.
                 * @param {Object} [value] The tag value.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the tag value; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).tag("cell tag");
                 * ```
                 */
                tag(value?: any): any;
                /**
                 * Gets or sets the formatted text for the cell.
                 * @param {string} [value] The cell text.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).text("cell object");
                 * ```
                 */
                text(value?: string): any;
                /**
                 * Gets or sets the type of the decoration added to the cell's text.
                 * @param {GC.Spread.Sheets.TextDecorationType} [value] The type of the decoration.
                 * @returns {GC.Spread.Sheets.TextDecorationType|GC.Spread.Sheets.CellRange} If no value is set, returns the type of the decoration; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(1, -1, 1, -1).textDecoration(GC.Spread.Sheets.TextDecorationType.overline | GC.Spread.Sheets.TextDecorationType.underline);
                 * ```
                 */
                textDecoration(value?: GC.Spread.Sheets.TextDecorationType): any;
                /**
                 * Gets or sets the type of the direction added to the cell's text.
                 * @param {GC.Spread.Sheets.TextDirectionType} [value] The type of the direction.
                 * @returns {GC.Spread.Sheets.TextDirectionType|GC.Spread.Sheets.CellRange} If no value is set, returns the type of the direction; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(1, -1, 1, -1).textDirection(GC.Spread.Sheets.TextDirectionType.rightToLeft);
                 * ```
                 */
                textDirection(value?: GC.Spread.Sheets.TextDirectionType): any;
                /**
                 * Gets or sets the text indent of the cell.
                 * @param {number} [value] The cell text indent.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text indent; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).textIndent(1);
                 * ```
                 */
                textIndent(value?: number): any;
                /**
                 * Gets or sets the text rotation angle of the cell.
                 * @param {number}  [value] The cell text rotation angle, form -90 to 90 angle.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the cell text rotation angle; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).textOrientation(66);
                 * ```
                 */
                textOrientation(value?: number): any;
                /**
                 * Gets or sets the theme font for the cell.
                 * @param {string} [value] The cell's theme font.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the cell's theme font; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(-1, 0).themeFont("Body");
                 * activeSheet.getCell(0,0).value("Test");
                 * ```
                 */
                themeFont(value?: string): any;
                /**
                 * Gets the HTML content from the specified area.
                 * @param {GC.Spread.Sheets.HeaderOptions} [headerOptions] Indicates whether to include row header and column header or not, when cell range is whole rows or whole columns in viewport area, it defaults to `noHeaders`.
                 * @param {boolean} [includeStyle] Indicates whether to include style and span or not, it defaults to `true`.
                 * @returns {string} The HTML content that contains cell text, cell span and cell style.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(0, 0, 10, 10).toHtml();
                 * ```
                 */
                toHtml(headerOptions?: GC.Spread.Sheets.HeaderOptions,  includeStyle?: boolean): string;
                /**
                 * Gets or sets the data validator for the cell.
                 * @param {GC.Spread.Sheets.DataValidation.DefaultDataValidator} [value] The cell data validator.
                 * @returns {GC.Spread.Sheets.DataValidation.DefaultDataValidator|GC.Spread.Sheets.CellRange} If no value is set, returns the cell data validator; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createDateValidator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, new Date(2012, 11, 31), new Date(2013, 11, 31));
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Enter a date between 12/31/2012 and 12/31/2013.");
                 * dv.inputTitle("Tip");
                 * activeSheet.getCell(0,0).validator(dv);
                 * ```
                 * @example
                 * ```javascript
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createDateValidator(GC.Spread.Sheets.ConditionalFormatting.ComparisonOperators.between, new Date(2012, 11, 31), new Date(2013, 11, 31));
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Enter a date between 12/31/2012 and 12/31/2013.");
                 * dv.inputTitle("Tip");
                 * activeSheet.getCell(1, -1).validator(dv);
                 * ```
                 * @example
                 * ```javascript
                 * spread.options.highlightInvalidData = true;
                 * var dv = GC.Spread.Sheets.DataValidation.createListValidator("1,2,3");
                 * dv.showInputMessage(true);
                 * dv.inputMessage("Value must be 1,2 or 3");
                 * dv.inputTitle("tip");
                 * activeSheet.setDataValidator(-1,0,dv);
                 * ```
                 */
                validator(value?: GC.Spread.Sheets.DataValidation.DefaultDataValidator): any;
                /**
                 * Gets or sets the vertical alignment of the contents of the cell.
                 * @param {GC.Spread.Sheets.VerticalAlign} [value] The vertical alignment.
                 * @returns {GC.Spread.Sheets.VerticalAlign|GC.Spread.Sheets.CellRange} If no value is set, returns the vertical alignment of the contents of the cell; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).vAlign(GC.Spread.Sheets.VerticalAlign.top);
                 * ```
                 */
                vAlign(value?: GC.Spread.Sheets.VerticalAlign): any;
                /**
                 * Gets or sets the unformatted value of the cell.
                 * @param {Object} [value] The cell value.
                 * @returns {Object|GC.Spread.Sheets.CellRange} If no value is set, returns the cell value; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).value(10);
                 * ```
                 */
                value(value?: any): any;
                /**
                 * Gets or sets whether the row or column is displayed.
                 * @param {boolean} [value] Set to `true` to make the row visible.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns the visible of the row or column; otherwise, returns the row or column.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(-1, 0).visible(false);
                 * ```
                 */
                visible(value?: boolean): any;
                /**
                 * Gets or sets the content of the cell watermark.
                 * @param {string} [value] The content of the watermark.
                 * @returns {string|GC.Spread.Sheets.CellRange} If no value is set, returns the content of the watermark; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(1,1).watermark("lightgreen");
                 * ```
                 */
                watermark(value?: string): any;
                /**
                 * Gets or sets the width of the column in pixels.
                 * @param {number} [value] The column width.
                 * @returns {number|GC.Spread.Sheets.CellRange} If no value is set, returns the column width; otherwise, returns the column.
                 * @example
                 * ```javascript
                 * activeSheet.getCell(-1, 0).width(20);
                 * ```
                 */
                width(value?: number): any;
                /**
                 * Gets or sets whether the cell lets text wrap.
                 * @param {boolean} [value] Set to `true` to let text wrap within the cell.
                 * @returns {boolean|GC.Spread.Sheets.CellRange} If no value is set, returns whether the cell lets text wrap; otherwise, returns the cell.
                 * @example
                 * ```javascript
                 * activeSheet.getRange(-1, 3, -1, 1, GC.Spread.Sheets.SheetArea.viewport).wordWrap(true);
                 * ```
                 */
                wordWrap(value?: boolean): any;
            }

            export class ColorScheme{
                /**
                 * Creates a ColorScheme instance.
                 * @constructor
                 * @class
                 * @classdesc Represents the theme color.
                 * @param {string} name The owner that contains the named variable.
                 * @param {string} background1 The theme color for background1.
                 * @param {string} background2 The theme color for background2.
                 * @param {string} text1 The theme color for text1.
                 * @param {string} text2 The theme color for text2.
                 * @param {string} accent1 The theme color for accent1.
                 * @param {string} accent2 The theme color for accent2.
                 * @param {string} accent3 The theme color for accent3.
                 * @param {string} accent4 The theme color for accent4.
                 * @param {string} accent5 The theme color for accent5.
                 * @param {string} accent6 The theme color for accent6.
                 * @param {string} link The color of the link.
                 * @param {string} followedLink The color of the followedLink.
                 * @example
                 * ```javascript
                 * //This example creates colors for the theme.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 1 30");
                 * ```
                 */
                constructor(name: string,  background1: string,  background2: string,  text1: string,  text2: string,  accent1: string,  accent2: string,  accent3: string,  accent4: string,  accent5: string,  accent6: string,  link: string,  followedLink: string);
                /**
                 * Gets or sets the accent1 theme color of the color scheme.
                 * @param {string} value The accent1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent1 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 1 30");
                 * ```
                 */
                accent1(value?: string): any;
                /**
                 * Gets or sets the accent2 theme color of the color scheme.
                 * @param {string} value The accent2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent2 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent2("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 2");
                 * ```
                 */
                accent2(value?: string): any;
                /**
                 * Gets or sets the accent3 theme color of the color scheme.
                 * @param {string} value The accent3 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent3 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent3 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent3("yellow");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 3");
                 * ```
                 */
                accent3(value?: string): any;
                /**
                 * Gets or sets the accent4 theme color of the color scheme.
                 * @param {string} value The accent4 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent4 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent4 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent4("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 4");
                 * ```
                 */
                accent4(value?: string): any;
                /**
                 * Gets or sets the accent5 theme color of the color scheme.
                 * @param {string} value The accent5 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent5 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent5 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent5("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 5");
                 * ```
                 */
                accent5(value?: string): any;
                /**
                 * Gets or sets the accent6 theme color of the color scheme.
                 * @param {string} value The accent6 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the accent6 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the accent6 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().accent6("blue");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("accent 6");
                 * ```
                 */
                accent6(value?: string): any;
                /**
                 * Gets or sets the background1 theme color of the color scheme.
                 * @param {string} value The background1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the background1 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the background1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().background1("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("background 1");
                 * ```
                 */
                background1(value?: string): any;
                /**
                 *  Gets or sets the background2 theme color of the color scheme.
                 * @param {string} value The background2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the background2 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the background2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().background2("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("background 2");
                 * ```
                 */
                background2(value?: string): any;
                /**
                 * Gets or sets the followed hyperlink theme color of the color scheme.
                 * @param {string} value The followed hyperlink theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the followed hyperlink theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the followedHyperline color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 * ```
                 */
                followedHyperlink(value?: string): any;
                /**
                 * Gets the color based on the theme color.
                 * @param {string} name The theme color name.
                 * @returns {string} The theme color.
                 * @example
                 * ```javascript
                 * //This example gets the theme color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * ntheme.colors().name("test");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 * var cname = ntheme.getColor("accent 1");
                 * alert(cname);
                 * ```
                 */
                getColor(name: string): string;
                /**
                 * Gets or sets the hyperlink theme color of the color scheme.
                 * @param {string} value The hyperlink theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the hyperlink theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the hyperline color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().followedHyperlink("orange");
                 * ntheme.colors().hyperlink("red");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("followedHyperlink");
                 * activeSheet.getCell(2, 0).backColor("hyperlink");
                 * ```
                 */
                hyperlink(value?: string): any;
                /**
                 * Gets or sets the name of the color scheme.
                 * @param {string} value The name.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the name; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the theme name.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().name("green theme");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * ```
                 */
                name(value?: string): any;
                /**
                 * Gets or sets the textcolor1 theme color of the color scheme.
                 * @param {string} value The textcolor1 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the textcolor1 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the textColor1 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().textColor1("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("Text 1");
                 * ```
                 */
                textColor1(value?: string): any;
                /**
                 * Gets or sets the textcolor2 theme color of the color scheme.
                 * @param {string} value The textcolor2 theme color string.
                 * @returns {string|GC.Spread.Sheets.ColorScheme} If no value is set, returns the textcolor2 theme color; otherwise, returns the color scheme.
                 * @example
                 * ```javascript
                 * //This example sets the textColor2 color.
                 * var ntheme = new GC.Spread.Sheets.Theme("customThemeColor");
                 * ntheme.colors().accent1("lightgreen");
                 * ntheme.colors().textColor2("orange");
                 * activeSheet.currentTheme(ntheme);
                 * activeSheet.getCell(0, 0).backColor("accent 1");
                 * activeSheet.getCell(1, 0).backColor("Text 2");
                 * ```
                 */
                textColor2(value?: string): any;
            }

            export class CustomDocPropsManager{
                /**
                 * Represents a custom document property manager. It can manage custom properties on all documents.
                 * @class
                 * @param {GC.Spread.Sheets.Workbook} workbook The workbook.
                 */
                constructor(workbook: GC.Spread.Sheets.Workbook);
                /**
                 * Set custom document property.
                 * @param {string} propName Property name.
                 * @param {GC.Spread.Sheets.CustomDocumentPropertyValueType} value Property value.
                 * @param {boolean} isLinkTarget Is it a content link.
                 * @example
                 * ```javascript
                 * spread.docProps.customDocPropsManager.add('prop1', '1');
                 * spread.docProps.customDocPropsManager.add('prop2', 'CellAlias', true);
                 * ```
                 */
                add(propName: string,  value: GC.Spread.Sheets.CustomDocumentPropertyValueType,  isLinkTarget?: boolean): void;
                /**
                 * Get or set custom document properties.
                 * @param {GC.Spread.Sheets.ICustomDocumentProperty[]} props Custom document properties.
                 * @returns {GC.Spread.Sheets.ICustomDocumentProperty[] | undefined} Custom document properties.
                 * @example
                 * ```javascript
                 * spread.docProps.customDocPropsManager.all([{ name: 'prop1', value: '1' }]);
                 * spread.docProps.customDocPropsManager.all();
                 * ```
                 */
                all(props?: GC.Spread.Sheets.ICustomDocumentProperty[]): GC.Spread.Sheets.ICustomDocumentProperty[] | undefined;
                /**
                 * Clear custom document properties.
                 * @example
                 * ```javascript
                 * spread.docProps.customDocPropsManager.clear();
                 * ```
                 */
                clear(): void;
                /**
                 * Get custom document property by property name.
                 * @param {string} propName Property name.
                 * @returns {GC.Spread.Sheets.ICustomDocumentProperty} Custom document property.
                 * @example
                 * ```javascript
                 * spread.docProps.customDocPropsManager.all([{ name: 'prop1', value: '1' }]);
                 * spread.docProps.customDocPropsManager.get('prop1');
                 * ```
                 */
                get(propName: string): GC.Spread.Sheets.ICustomDocumentProperty | undefined;
                /**
                 * Remove custom document property by property name.
                 * @param {string} propName Property name.
                 * @example
                 * ```javascript
                 * spread.docProps.customDocPropsManager.remove('prop1');
                 * ```
                 */
                remove(propName: string): void;
            }

            export class CustomThemeManagerBase{
                /**
                 * Represents a custom theme manager that can manage all custom themes.
                 * @class
                 * @param {GC.Spread.Sheets.Workbook} workbook The workbook.
                 */
                constructor(workbook: GC.Spread.Sheets.Workbook);
            }

            export class Events{
                /**
                 * Defines the events supported in SpreadJS.
                 * @class
                 */
                constructor();
                /**
                 * Occurs when the user has changed the active sheet.
                 * @name GC.Spread.Sheets.Workbook#ActiveSheetChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `oldSheet` The old sheet.
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `newSheet` The new sheet.
                 * @example
                 * ```javascript
                 * //This example creates log text for the ActiveSheetChanged event.
                 * // Use web browser to see the console log text
                 * spread.bind(GC.Spread.Sheets.Events.ActiveSheetChanged, function (sender, args) {
                 *     console.log("Active sheet has been switched.");
                 * });
                 * ```
                 */
                static ActiveSheetChanged: string;
                /**
                 * Occurs when the user is changing the active sheet.
                 * @name GC.Spread.Sheets.Workbook#ActiveSheetChanging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `oldSheet` The old sheet.
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `newSheet` The new sheet.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example stops the active sheet from changing.
                 * spread.bind(GC.Spread.Sheets.Events.ActiveSheetChanging, function (sender, args) {
                 *     //Cancel sheet switching.
                 *     args.cancel = true;
                 * });
                 * ```
                 */
                static ActiveSheetChanging: string;
                /**
                 * Occurs before the print
                 * @name GC.Spread.Sheets.WorkBook.BeforePrint
                 * @event
                 * @param eventParam *Object* `iframe` The print Dom
                 * @param eventParam *boolean* `cancel` Whether cancel print
                 * @example
                 * ```javascript
                 * //This example uses the BeforePrint.
                 * window.onload = function(){
                 *      var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 *      spread.bind(GC.Spread.Sheets.Events.BeforePrint, function (e, data) {
                 *          alert(data.iframe + '\n' + 'cancel: ' + data.cancel);
                 *      });
                 * }
                 * ```
                 */
                static BeforePrint: string;
                /**
                 * Occurs when the user clicks a button, check box, or hyperlink in a cell.
                 * @name GC.Spread.Sheets.Workbook#ButtonClicked
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of the cell.
                 * @param eventParam *number* `col` The column index of the cell.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheet area that contains the cell.
                 * @example
                 * ```javascript
                 * //This example creates a button cell.
                 * var cellType = new GC.Spread.Sheets.CellTypes.Button();
                 * cellType.buttonBackColor("#FFFF00");
                 * cellType.text("this is a button");
                 * activeSheet.setCellType(1,1,cellType);
                 * //bind
                 * spread.bind(GC.Spread.Sheets.Events.ButtonClicked, function (e, args) {
                 *     var sheet = args.sheet, row = args.row, col = args.col;
                 *     var cellType = activeSheet.getCellType(row, col);
                 *     if (cellType instanceof GC.Spread.Sheets.CellTypes.Button) {
                 *         alert("Button Clicked");
                 *     }
                 * });
                 * ```
                 */
                static ButtonClicked: string;
                /**
                 * Occurs when the calculate with incrementalCalculation.
                 * @name GC.Spread.Sheets.Workbook#CalculationProgress
                 * @event
                 * @param eventParam *number* `totalCells` The count of total calculate cells.
                 * @param eventParam *number* `pendingCells` The count of cells needed to be calculated.
                 * @param eventParam *number* `iterate` The count of iterative calculation has done, only available when the iterative calculation.
                 * @example
                 * ```javascript
                 * //This example uses the CalculationProgress event, to log the calculation progress.
                 * spread.options.incrementalCalculation = true;
                 * spread.bind(GC.Spread.Sheets.Events.CalculationProgress, function (e, info) {
                 *      var msg = "Calculate ";
                 *      if (info.pendingCells === 0) {
                 *          msg += "finished";
                 *      } else if (info.iterate >= 0) {
                 *          msg += info.pendingCells + " cells in iterative calculation round " + info.iterate;
                 *      } else {
                 *          msg += (info.totalCells - info.pendingCells) + "/" + info.totalCells + "cells";
                 *      }
                 *      console.log(msg)
                 * });
                 * ```
                 */
                static CalculationProgress: string;
                /**
                 * Occurs when a change is made to a cell in this sheet that may require the cell to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#CellChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of the cell.
                 * @param eventParam *number* `col` The column index of the cell.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheetArea of the cell.
                 * @param eventParam *string* `propertyName` The name of the cell's property that has changed.
                 * @param eventParam *boolean* `isUndo` Whether this event is from a undo operation.
                 * @example
                 * ```javascript
                 * //This example uses the CellChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellChanged, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Cell index (" + info.row + "," + info.col + ")");
                 *     }
                 * });
                 * ```
                 */
                static CellChanged: string;
                /**
                 * Occurs when the user releases the left mouse button in a cell.
                 * @name GC.Spread.Sheets.Worksheet#CellClick
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheet area the clicked cell is in.
                 * @param eventParam *number* `row` The row index of the clicked cell.
                 * @param eventParam *number* `col` The column index of the clicked cell.
                 * @example
                 * ```javascript
                 * //This example uses the CellClick event.
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was clicked.");
                 *     }
                 *     console.log("Clicked column index: " + args.col);
                 *     console.log("Clicked row index: " + args.row);
                 * });
                 * //bind
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was double clicked.");
                 *    }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was double clicked.");
                 *     }
                 *     console.log("Double clicked column index: " + args.col);
                 *     console.log("Double clicked row index: " + args.row);
                 * })
                 * ```
                 */
                static CellClick: string;
                /**
                 * Occurs when the user presses down the left mouse button twice (double-clicks) in a cell.
                 * @name GC.Spread.Sheets.Worksheet#CellDoubleClick
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheet area the clicked cell is in.
                 * @param eventParam *number* `row` The row index of the clicked cell.
                 * @param eventParam *number* `col` The column index of the clicked cell.
                 * @example
                 * ```javascript
                 * //This example uses the CellDoubleClick event.
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was clicked.");
                 *     }
                 *     console.log("Clicked column index: " + args.col);
                 *     console.log("Clicked row index: " + args.row);
                 * });
                 * //bind
                 * activeSheet.bind(GC.Spread.Sheets.Events.CellDoubleClick, function (sender, args) {
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.colHeader){
                 *         console.log("The column header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.rowHeader){
                 *         console.log("The row header was double clicked.");
                 *     }
                 *     if(args.sheetArea === GC.Spread.Sheets.SheetArea.corner){
                 *         console.log("The corner header was double clicked.");
                 *     }
                 *     console.log("Double clicked column index: " + args.col);
                 *     console.log("Double clicked row index: " + args.row);
                 * });
                 * ```
                 */
                static CellDoubleClick: string;
                /**
                 * Occurs when a Clipboard change occurs that affects Spread.Sheets.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *Object* `copyData` The data from Spread.Sheets that has been set into the clipboard.
                 * @param eventParam *string* `copyData.text` The text string of the clipboard.
                 * @param eventParam *string* `copyData.html` The html string of the clipboard.
                 * @param eventParam *Array* `objects` The copied floating objects, it contains picture, custom floating object, slicer, chart and shape.
                 * @example
                 * ```javascript
                 * //This example uses the ClipboardChanged event.
                 * spread.bind(GC.Spread.Sheets.Events.ClipboardChanged, function (sender, args) {
                 *     console.log("ClipboardChanged.", args);
                 * });
                 * ```
                 */
                static ClipboardChanged: string;
                /**
                 * Occurs when the Clipboard is changing due to a Spread.Sheets action.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardChanging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *Object* `copyData` The data from Spread.Sheets that is set into the clipboard.
                 * @param eventParam *string* `copyData.text` The text string of the clipboard.
                 * @param eventParam *string* `copyData.html` The html string of the clipboard.
                 * @param eventParam *{@link GC.Spread.Sheets.Range[]}* `ranges` The source ranges of the source sheet for current cut or copy operation.
                 * @param eventParam *{@link GC.Spread.Sheets.ClipboardActionType}* `action` Indicates the type of the current action.
                 * @param eventParam *Array* `objects` The coping floating objects, it contains picture, custom floating object, slicer, chart and shape.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example uses the ClipboardChanging event.
                 * spread.bind(GC.Spread.Sheets.Events.ClipboardChanging, function (sender, args) {
                 *     console.log("ClipboardChanging", args);
                 * });
                 * ```
                 */
                static ClipboardChanging: string;
                /**
                 * Occurs when the user pastes from the Clipboard.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardPasted
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `cellRange` The range that was pasted.
                 * @param eventParam *{@link GC.Spread.Sheets.ClipboardPasteOptions}* `pasteOption` The paste option that indicates what data is pasted from the clipboard: values, formatting, or formulas.
                 * @param eventParam *Object* `pasteData` The data from the clipboard that will be pasted into Spread.Sheets.
                 * @param eventParam *string* `pasteData.text` The text string of the clipboard.
                 * @param eventParam *string* `pasteData.html` The html string of the clipboard.
                 * @param eventParam *string* `pasteData.image` The image src string of the clipboard.
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `fromSheet` the source sheet of data range.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `[fromRange]` The source range of paste when copy or cut range.
                 * @param eventParam *{@link GC.Spread.Sheets.ClipboardActionType}* `action` Indicates the type of the current action.
                 * @param eventParam *Array* `objects` The pasted floating objects, it contains picture, custom floating object, slicer, chart and shape.
                 * @param eventParam *{@link GC.Spread.Sheets.InsertShiftCell}* `[shiftCells]` The shift cells option, if not shift cell, will be undefined.
                 * @example
                 * ```javascript
                 * //This example uses the ClipboardPasted event.
                 * spread.bind(GC.Spread.Sheets.Events.ClipboardPasted, function (sender, args) {
                 *     console.log("ClipboardPasted", args);
                 * });
                 * ```
                 */
                static ClipboardPasted: string;
                /**
                 * Occurs when the user is pasting from the Clipboard.
                 * @name GC.Spread.Sheets.Worksheet#ClipboardPasting
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `cellRange` The range to paste.
                 * @param eventParam *{@link GC.Spread.Sheets.ClipboardPasteOptions}* `pasteOption` The paste option that indicates what data is pasted from the clipboard: values, formatting, or formulas.
                 * @param eventParam *Object* `pasteData` The data from the clipboard that will be pasted into Spread.Sheets.
                 * @param eventParam *string* `pasteData.text` The text string of the clipboard.
                 * @param eventParam *string* `pasteData.html` The html string of the clipboard.
                 * @param eventParam *string* `pasteData.image` The image src string of the clipboard.
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `fromSheet` the source sheet of data range.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `[fromRange]` The source range of paste when copy or cut range.
                 * @param eventParam *{@link GC.Spread.Sheets.ClipboardActionType}* `action` Indicates the type of the current action.
                 * @param eventParam *Array* `objects` The pasting floating objects, it contains picture, custom floating object, slicer, chart and shape.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @param eventParam *{@link GC.Spread.Sheets.InsertShiftCell}* `[shiftCells]` The shift cells option, if not shift cell, will be undefined.
                 * @example
                 * ```javascript
                 * //This example uses the ClipboardPasting event.
                 * spread.bind(GC.Spread.Sheets.Events.ClipboardPasting, function (sender, args) {
                 *     console.log("ClipboardPasting", args);
                 * });
                 * ```
                 */
                static ClipboardPasting: string;
                /**
                 * After the redo in collaboration.
                 * @name GC.Spread.Sheets.Worksheet#CollaborationEndRedo
                 * @event
                 * @param eventParam *object* `changeSet` The redo operation object.
                 * @example
                 * ```javascript
                 * //This example.
                 * workbook.bind(GC.Spread.Sheets.Events.CollaborationEndRedo, (sender, args) => {
                 *     const changeSet = args.changeSet;
                 *     console.log('redo changeSet is:' changeSet);
                 * });
                 * ```
                 */
                static CollaborationEndRedo: string;
                /**
                 * After the undo in collaboration.
                 * @name GC.Spread.Sheets.Worksheet#CollaborationEndUndo
                 * @event
                 * @param eventParam *object* `changeSet` The undo operation object.
                 * @example
                 * ```javascript
                 * //This example.
                 * workbook.bind(GC.Spread.Sheets.Events.CollaborationEndUndo, (sender, args) => {
                 *     const changeSet = args.changeSet;
                 *     console.log('undo changeSet is:' changeSet);
                 * });
                 * ```
                 */
                static CollaborationEndUndo: string;
                /**
                 * Before the redo in collaboration.
                 * @name GC.Spread.Sheets.Worksheet#CollaborationStartRedo
                 * @event
                 * @param eventParam *object* `changeSet` The redo operation object.
                 * @example
                 * ```javascript
                 * //This example.
                 * workbook.bind(GC.Spread.Sheets.Events.CollaborationStartRedo, (sender, args) => {
                 *     const changeSet = args.changeSet;
                 *     console.log('redo changeSet is:' changeSet);
                 * });
                 * ```
                 */
                static CollaborationStartRedo: string;
                /**
                 * Before the undo in collaboration.
                 * @name GC.Spread.Sheets.Worksheet#CollaborationStartUndo
                 * @event
                 * @param eventParam *object* `changeSet` The undo operation object.
                 * @example
                 * ```javascript
                 * //This example.
                 * workbook.bind(GC.Spread.Sheets.Events.CollaborationStartUndo, (sender, args) => {
                 *     const changeSet = args.changeSet;
                 *     console.log('undo changeSet is:' changeSet);
                 * });
                 * ```
                 */
                static CollaborationStartUndo: string;
                /**
                 * Occurs when a change is made to a column or range of columns in this sheet that may require the column or range of columns to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#ColumnChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The column index.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheetArea of the column.
                 * @param eventParam *string* `propertyName` The name of the column's property that has changed.
                 * @param eventParam *Object* `oldValue` The old value of the property.
                 * @param eventParam *Object* `newValue` The new value of the property.
                 * @param eventParam *number* `[count]` The number of columns affected.
                 * @param eventParam *boolean* `isUndo` Whether this event is from a undo operation.
                 * @example
                 * ```javascript
                 * //This example uses the ColumnChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnChanged, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Index (" + info.col + ")");
                 *     }
                 * });
                 * ```
                 */
                static ColumnChanged: string;
                /**
                 * Occurs when before a change is made to a column or range of columns in this sheet that may require the column or range of columns to be repainted.
                 * @name GC.Spread.Sheets.Worksheet#ColumnChanging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The column index.
                 * @param eventParam *{@link GC.Spread.Sheets.SheetArea}* `sheetArea` The sheetArea of the column.
                 * @param eventParam *string* `propertyName` The name of the column's property that has changed.
                 * @param eventParam *Object* `oldValue` The old value of the property.
                 * @param eventParam *Object* `newValue` The new value of the property.
                 * @param eventParam *number* `[count]` The number of columns affected.
                 * @param eventParam *boolean* `cancel` Whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example uses the ColumnChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnChanging, function (e, info) {
                 *     if(info.sheetArea === GC.Spread.Sheets.SheetArea.viewport){
                 *         alert("Index (" + info.col + ")");
                 *     }
                 * });
                 * ```
                 */
                static ColumnChanging: string;
                /**
                 * Occurs when the column width has changed.
                 * @name GC.Spread.Sheets.Worksheet#ColumnWidthChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *Array* `colList` The list of columns whose widths have changed.
                 * @param eventParam *boolean* `header` Whether the columns are row header columns.
                 * @example
                 * ```javascript
                 * //This example uses the ColumnWidthChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnWidthChanged, function (e, info) {
                 *         alert("Column (" + info.colList + ")");
                 * });
                 * ```
                 */
                static ColumnWidthChanged: string;
                /**
                 * Occurs when the column width is changing.
                 * @name GC.Spread.Sheets.Worksheet#ColumnWidthChanging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *Array* `colList` The list of columns whose widths are changing.
                 * @param eventParam *boolean* `header` Whether the columns are row header columns.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example uses the ColumnWidthChanging event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.ColumnWidthChanging, function (e, info) {
                 *         alert("Column (" + info.colList + ")");
                 * });
                 * ```
                 */
                static ColumnWidthChanging: string;
                /**
                 * Occurs when any comment has changed.
                 * @name GC.Spread.Sheets.Worksheet#CommentChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Comments.Comment}* `comment` The comment that triggered the event.
                 * @param eventParam *string* `propertyName` The name of the comment's property that has changed.
                 * @example
                 * ```javascript
                 * //This example uses the CommentChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentChanged, function (e, info) {
                 *     alert("changed");
                 * });
                 * ```
                 */
                static CommentChanged: string;
                /**
                 * Occurs when the user has removed the comment.
                 * @name GC.Spread.Sheets.Worksheet#CommentRemoved
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Comments.Comment}* `comment` The comment has been removed.
                 * @example
                 * ```javascript
                 * //This example uses the CommentRemoved event.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("orange");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentRemoved, function (e, info) {
                 *        console.log("sheet name: " + info.sheetName);
                 * });
                 * ```
                 */
                static CommentRemoved: string;
                /**
                 * Occurs when the user is removing any comment.
                 * @name GC.Spread.Sheets.Worksheet#CommentRemoving
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Comments.Comment}* `comment` The comment is being removed.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example prevents the comment from being removed.
                 * var comment = new GC.Spread.Sheets.Comments.Comment();
                 * comment.text("new comment!");
                 * comment.backColor("orange");
                 * comment.displayMode(GC.Spread.Sheets.Comments.DisplayMode.alwaysShown);
                 * activeSheet.getCell(5,5).comment(comment);
                 * activeSheet.bind(GC.Spread.Sheets.Events.CommentRemoving, function (e, info) {
                 *       info.cancel = true;
                 * });
                 * ```
                 */
                static CommentRemoving: string;
                /**
                 * Occurs when all tables from data manager fetched completely through the workbook opening or importing the sjs/json format file.
                 * @name GC.Spread.Sheets.Worksheet#DataFetchCompleted
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Workbook}* `spread` The workbook that triggered the event.
                 * @example
                 * ```javascript
                 * //This example uses the DataFetchCompleted event.
                 * //In general, the DataFetchCompleted event is triggered after the successCallback of the workbook open method invoked.
                 * //If doesn\u2019t use any table of data manager, and the Workbook contains a lot of Worksheet, the DataFetchCompleted event could be triggered before the successCallback of the workbook open method invoked.
                 * let isOpenFinished = false, isDataFetchCompleted = false;
                 * spread.bind(GC.Spread.Sheets.Events.DataFetchCompleted, function () {
                 *     isDataFetchCompleted = true;
                 *     if (isOpenFinished) {
                 *         // do something with data fetched
                 *     }
                 * });
                 * spread.open(file, function () {
                 *     isOpenFinished = true;
                 *     if (isDataFetchCompleted) {
                 *         // do something with file opened
                 *     }
                 * }, function () { }, {});
                 * ```
                 */
                static DataFetchCompleted: string;
                /**
                 * Occurs when the user is dragging and dropping a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragDropBlock
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `fromRow` The row index of the top left cell of the source range (range being dragged).
                 * @param eventParam *number* `fromCol` The column index of the top left cell of the source range (range being dragged).
                 * @param eventParam *number* `toRow` The row index of the top left cell of the destination range (where selection is dropped).
                 * @param eventParam *number* `toCol` The column index of the bottom right cell of the destination range (where selection is dropped).
                 * @param eventParam *number* `rowCount` The row count of the cell range being dragged.
                 * @param eventParam *number* `colCount` The column count of the cell range being dragged.
                 * @param eventParam *boolean* `copy` Whether the source range is copied.
                 * @param eventParam *boolean* `insert` Whether the source range is inserted.
                 * @param eventParam *{@link GC.Spread.Sheets.CopyToOptions}* `copyOption` The CopyOption value for the drag and drop operation.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example creates log text for the DragDropBlock event.
                 * // Use web browser to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragDropBlock, function (e, args) {
                 *         console.log("From Column:" + args.fromCol);
                 *         console.log("From Row:" + args.fromRow);
                 *         console.log("To Column:" + args.toCol);
                 *         console.log("To Row:" + args.toRow);
                 *     });
                 * ```
                 */
                static DragDropBlock: string;
                /**
                 * Occurs when the user completes dragging and dropping a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragDropBlockCompleted
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `fromRow` The row index of the top left cell of the source range (range being dragged).
                 * @param eventParam *number* `fromCol` The column index of the top left cell of the source range (range being dragged).
                 * @param eventParam *number* `toRow` The row index of the top left cell of the destination range (where selection is dropped).
                 * @param eventParam *number* `toCol` The column index of the bottom right cell of the destination range (where selection is dropped).
                 * @param eventParam *number* `rowCount` The row count of the cell range being dragged.
                 * @param eventParam *number* `colCount` The column count of the cell range being dragged.
                 * @param eventParam *boolean* `copy` Whether the source range is copied.
                 * @param eventParam *boolean* `insert` Whether the source range is inserted.
                 * @param eventParam *{@link GC.Spread.Sheets.CopyToOptions}* `copyOption` The CopyOption value for the drag and drop operation.
                 * @example
                 * ```javascript
                 * //This example uses the DragDropBlockCompleted event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragDropBlockCompleted, function (e, args) {
                 *         alert("From Column (" + args.fromCol + ")");
                 * });
                 * ```
                 */
                static DragDropBlockCompleted: string;
                /**
                 * Occurs when the user is dragging to fill a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragFillBlock
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `fillRange` The range used for the fill operation.
                 * @param eventParam *{@link GC.Spread.Sheets.Fill.AutoFillType}* `autoFillType` AutoFillType value used for the fill operation.
                 * @param eventParam *{@link GC.Spread.Sheets.Fill.FillDirection}* `fillDirection` The direction of the fill.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example uses the DragFillBlock event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragFillBlock, function (e, info) {
                 *         alert("Direction (" + info.fillDirection + ")");
                 * });
                 * ```
                 */
                static DragFillBlock: string;
                /**
                 * Occurs when the user completes dragging to fill a range of cells.
                 * @name GC.Spread.Sheets.Worksheet#DragFillBlockCompleted
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `fillRange` The range used for the fill operation.
                 * @param eventParam *{@link GC.Spread.Sheets.Fill.AutoFillType}* `autoFillType` AutoFillType value used for the fill operation.
                 * @param eventParam *{@link GC.Spread.Sheets.Fill.FillDirection}* `fillDirection` The direction of the fill.
                 * @example
                 * ```javascript
                 * //This example uses the DragFillBlockCompleted event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragFillBlockCompleted, function (e, info) {
                 *         alert("Type (" + info.autoFillType + ")");
                 * });
                 * ```
                 */
                static DragFillBlockCompleted: string;
                /**
                 * Occurs after user drag merge cells.
                 * @name GC.Spread.Sheets.Worksheet#DragMerged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `mergeRange` The range that will be merged.
                 * @example
                 * ```javascript
                 * //This example returns the row index.
                 * // Press Ctrl key to merge
                 * $(document).keydown(function (e) {
                 *      if (e.keyCode === 17) {
                 *         spread.options.allowUserDragMerge = true;
                 *      }
                 * });
                 * $(document).keyup(function (e) {
                 *      if (e.keyCode === 17) {
                 *         spread.options.allowUserDragMerge = false;
                 *      }
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerging, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerged, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * ```
                 */
                static DragMerged: string;
                /**
                 * Occurs before user drag merge cells.
                 * @name GC.Spread.Sheets.Worksheet#DragMerging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `mergeRange` The range that will be merged.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example returns the row index.
                 * // Press Ctrl key to merge
                 * $(document).keydown(function (e) {
                 *      if (e.keyCode === 17) {
                 *          spread.options.allowUserDragMerge = true;
                 *      }
                 * });
                 * $(document).keyup(function (e) {
                 *      if (e.keyCode === 17) {
                 *          spread.options.allowUserDragMerge = false;
                 *      }
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerging, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.DragMerged, function (e, data) {
                 *      var mergeRange = data.mergeRange;
                 *      alert(mergeRange.row);
                 * });
                 * ```
                 */
                static DragMerging: string;
                /**
                 * Occurs when a cell is in edit mode and the text is changed.
                 * @name GC.Spread.Sheets.Worksheet#EditChange
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of cell.
                 * @param eventParam *number* `col` The column index of cell.
                 * @param eventParam *Object* `editingText` The value from the current editor.
                 * @example
                 * ```javascript
                 * //This example creates log text for the EditChange event.
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditChange, function (sender, args) {
                 *     console.log("Cell (" + args.row + ", " + args.col + ") data has been changed.")
                 * });
                 * ```
                 */
                static EditChange: string;
                /**
                 * Occurs when a cell leaves edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditEnded
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of cell.
                 * @param eventParam *number* `col` The column index of cell.
                 * @param eventParam *Object* `editingText` The value from the current editor.
                 * @param eventParam *boolean* `committed` Whether the value change was committed
                 * @example
                 * ```javascript
                 * //This example creates log text for the EditStarting and EditEnded events.
                 *  // Use web browser to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("Finish cell editing.");
                 * });
                 * ```
                 */
                static EditEnded: string;
                /**
                 * Occurs when a cell is leaving edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditEnding
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of cell.
                 * @param eventParam *number* `col` The column index of cell.
                 * @param eventParam *Object* `editor` The current editor.
                 * @param eventParam *Object* `editingText` The value from the current editor.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @param eventParam *boolean* `committed` Whether the value change was committed
                 * @example
                 * ```javascript
                 * //This example uses the EditEnding event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnding, function (sender, args) {
                 *     console.log("EditEnding event.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("EditEnded event.");
                 * });
                 * ```
                 */
                static EditEnding: string;
                /**
                 * Occurs when the editor's status has changed.
                 * @name GC.Spread.Sheets.Worksheet#EditorStatusChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.EditorStatus}* `oldStatus` The old status of the editor.
                 * @param eventParam *{@link GC.Spread.Sheets.EditorStatus}* `newStatus` The new status of the editor.
                 * @example
                 * ```javascript
                 * //This example uses the EditorStatusChanged event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditorStatusChanged, function (e, info) {
                 *         alert("Column (" + info.newStatus + ")");
                 * });
                 * ```
                 */
                static EditorStatusChanged: string;
                /**
                 * Occurs when a cell is entering edit mode.
                 * @name GC.Spread.Sheets.Worksheet#EditStarting
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of cell.
                 * @param eventParam *number* `col` The column index of cell.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example creates log text for the EditStarting and EditEnded events.
                 * // Use web browser to see the console log text
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditStarting, function (sender, args) {
                 *     console.log("Start cell editing.");
                 * });
                 * activeSheet.bind(GC.Spread.Sheets.Events.EditEnded, function (sender, args) {
                 *     console.log("Finish cell editing.");
                 * });
                 * ```
                 */
                static EditStarting: string;
                /**
                 * Occurs when the focus enters a cell.
                 * @name GC.Spread.Sheets.Worksheet#EnterCell
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of the cell being entered.
                 * @param eventParam *number* `col` The column index of the cell being entered.
                 * @example
                 * ```javascript
                 * //This example uses the EnterCell event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.EnterCell, function (e, info) {
                 *         alert("Cell (" + info.row + ", " + info.col +")");
                 * });
                 * ```
                 */
                static EnterCell: string;
                /**
                 * Occurs when any floating object has changed.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectsChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.FloatingObject}* `floatingObject` The floating object that triggered the event.
                 * @param eventParam *string* `propertyName` The name of the floating object's property that has changed.
                 * @example
                 * ```javascript
                 * //This example uses the FloatingObjectChanged event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectChanged, function (e, info) {
                 *        alert("changed");
                 * });
                 * ```
                 */
                static FloatingObjectChanged: string;
                /**
                 * Occurs when the custom floating object content is loaded.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectLoaded
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.FloatingObject}* `floatingObject` The custom floating object that triggered the event.
                 * @param eventParam *HTMLElement* `element` The HTMLElement of the custom floating object.
                 */
                static FloatingObjectLoaded: string;
                /**
                 * Occurs when the user has removed the floating object.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectRemoved
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.FloatingObject}* `floatingObject` The floating object has been removed.
                 * @example
                 * ```javascript
                 * //This example uses the FloatingObjectRemoved event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectRemoved, function (e, info) {
                 *         alert(info.sheetName);
                 * });
                 * ```
                 */
                static FloatingObjectRemoved: string;
                /**
                 * Occurs when the user is removing any floating object.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectRemoving
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.FloatingObject}* `floatingObject` The floating object is being removed.
                 * @param eventParam *boolean* `cancel` A value that indicates whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example uses the FloatingObjectRemoving event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectRemoving, function (e, info) {
                 *         info.cancel = true;
                 * });
                 * ```
                 */
                static FloatingObjectRemoving: string;
                /**
                 * Occurs when the selections of the floating object have changed.
                 * @name GC.Spread.Sheets.Worksheet#FloatingObjectsSelectionChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.FloatingObject}* `floatingObject` The floating object that triggered the event.
                 * @example
                 * ```javascript
                 * //This example uses the FloatingObjectSelectionChanged event.
                 * var customFloatingObject = new GC.Spread.Sheets.FloatingObjects.FloatingObject("f1", 10, 10, 60, 64);
                 * var btn = document.createElement('button');
                 * btn.style.width = "60px";
                 * btn.style.height = "30px";
                 * btn.innerText = "button";
                 * customFloatingObject.content(btn);
                 * activeSheet.floatingObjects.add(customFloatingObject);
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.FloatingObjectSelectionChanged, function (e, info) {
                 *        console.log("sheet name: " + info.sheetName);
                 * });
                 * ```
                 */
                static FloatingObjectSelectionChanged: string;
                /**
                 * Occurs when the button form control is clicked.
                 * @name GC.Spread.Sheets.Worksheet#FormControlButtonClicked
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Shapes.Shape}* `shape` The shape that triggered the event.
                 * @example
                 * ```javascript
                 * //This example uses the FormControlButtonClicked event.
                 * var shape = sheet.shapes.addFormControl("button", GC.Spread.Sheets.Shapes.FormControlType.button, 50, 50, 150, 100);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FormControlButtonClicked, function (e, info) {
                 *        console.log("event info: " + info);
                 * });
                 * ```
                 */
                static FormControlButtonClicked: string;
                /**
                 * Occurs when the value of the form control have changed.
                 * @name GC.Spread.Sheets.Worksheet#FormControlValueChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.Shapes.Shape}* `shape` The shape that triggered the event.
                 * @param eventParam *any* `newValue` the new value of the form control.
                 * @param eventParam *any* `oldValue` the old value of the form control.
                 * @example
                 * ```javascript
                 * //This example uses the FormControlValueChanged event.
                 * var shape = sheet.shapes.addFormControl('spin', GC.Spread.Sheets.Shapes.FormControlType.spinButton, 50, 50, 150, 100);
                 * activeSheet.bind(GC.Spread.Sheets.Events.FormControlValueChanged, function (e, info) {
                 *        console.log("event info: " + info);
                 * });
                 * ```
                 */
                static FormControlValueChanged: string;
                /**
                 * Occurs when an invalid operation is performed.
                 * @name GC.Spread.Sheets.Worksheet#InvalidOperation
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.InvalidOperationType}* `invalidType` Indicates which operation was invalid.
                 * @param eventParam *string* `message` The description of the invalid operation.
                 * @param eventParam *{@link GC.Spread.Sheets.Range}* `[fillRange]` When invalidType is dragFill, The range used for the fill operation.
                 * @example
                 * ```javascript
                 * //This example uses the InvalidOperation event.
                 * activeSheet.bind(GC.Spread.Sheets.Events.InvalidOperation, function (e, info) {
                 *         alert("Message (" + info.message + ")");
                 * });
                 * ```
                 */
                static InvalidOperation: string;
                /**
                 * Occurs when the focus leaves a cell.
                 * @name GC.Spread.Sheets.Worksheet#LeaveCell
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The row index of the cell being left.
                 * @param eventParam *number* `col` The column index of the cell being left.
                 * @param eventParam *boolean* `cancel` Whether the operation should be canceled.
                 * @example
                 * ```javascript
                 * //This example creates log text for the LeaveCell event.
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.LeaveCell, function (sender, args) {
                 *     console.log("The column index before moving: " + args.col);
                 *     console.log("The row index before moving: " + args.row);
                 * });
                 * ```
                 */
                static LeaveCell: string;
                /**
                 * Occurs when the left column changes.
                 * @name GC.Spread.Sheets.Worksheet#LeftColumnChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `oldLeftCol` The old left column index.
                 * @param eventParam *number* `newLeftCol` The new left column index.
                 * @param eventParam *number* `oldOffset` The old left column offset.
                 * @param eventParam *number* `newOffset` The new left column offset.
                 * @example
                 * ```javascript
                 * //This example synchronizes vertical and horizontal scrolling for sheet 1 and sheet 2.
                 * var sheet1 = spread.getSheet(0),
                 * sheet2 = spread.getSheet(1);
                 * sheet1.bind(GC.Spread.Sheets.Events.TopRowChanged, function (sender, args) {
                 *     //Set the displayed top row of sheet1 to sheet2 (vertical scroll synchronization).
                 *     sheet2.showRow(args.newTopRow, GC.Spread.Sheets.VerticalPosition.top);
                 * });
                 * sheet1.bind(GC.Spread.Sheets.Events.LeftColumnChanged, function (sender, args) {
                 *     //Set the displayed left column of sheet1 to sheet2 (Horizontal scroll synchronization).
                 *     sheet2.showColumn(args.newLeftCol, GC.Spread.Sheets.HorizontalPosition.left);
                 * });
                 * ```
                 */
                static LeftColumnChanged: string;
                /**
                 * Occurs when the outline column check status has changed.
                 * @name GC.Spread.Sheets.Worksheet#OutlineColumnCheckStatusChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The outline column's change row index.
                 * @param eventParam *number* `col` The outline column's change col index.
                 * @param eventParam *boolean* `status` The outline column's change status
                 * @example
                 * ```javascript
                 * //Removing the sparkline causes a change.
                 * activeSheet.bind(GC.Spread.Sheets.Events.OutlineColumnCheckStatusChanged, function (e, info) {
                 *      console.log("status: " + info.status);
                 * });
                 * ```
                 */
                static OutlineColumnCheckStatusChanged: string;
                /**
                 * Occurs when the workbook permission changed.
                 * @name GC.Spread.Sheets.Workbook#PermissionChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Collaboration.IPermission}* `permission` The sheet that triggered the event.
                 * @example
                 * ```javascript
                 * ```javascript
                 * //This example uses the ThreadedCommentChanged event.
                 * spread.bind(GC.Spread.Sheets.Events.PermissionChanged, function (e, info) {
                 *     console.log(info.permission);
                 * });
                 * ```
                 * ```
                 */
                static PermissionChanged: string;
                /**
                 * Occurs when any picture has changed.
                 * @name GC.Spread.Sheets.Worksheet#PictureChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.Picture}* `picture` The picture that triggered the event.
                 * @param eventParam *string* `propertyName` The name of the picture's property that has changed.
                 * @example
                 * ```javascript
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.pictures.add("f2","Event.png",2,2,6,6);
                 * activeSheet.pictures.add("f1","tsoutline.png",3,0,6,6);
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.PictureChanged, function (e, info) {
                 * console.log("Sheet: " + info.sheetName);
                 * console.log("Property: " + info.propertyName);
                 * });
                 * ```
                 */
                static PictureChanged: string;
                /**
                 * Occurs when the selections of the picture have changed.
                 * @name GC.Spread.Sheets.Worksheet#PictureSelectionChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *{@link GC.Spread.Sheets.FloatingObjects.Picture}* `picture` The picture that triggered the event.
                 * @example
                 * ```javascript
                 * //This example uses the PictureSelectionChanged event.
                 * var spread = new GC.Spread.Sheets.Workbook(document.getElementById("ss"),{sheetCount:3});
                 * var activeSheet = spread.getActiveSheet();
                 * activeSheet.pictures.add("f2","Event.png",2,2,6,6);
                 * activeSheet.pictures.add("f1","tsoutline.png",3,0,6,6);
                 * // Use web browser to see the console log text
                 * activeSheet.bind(GC.Spread.Sheets.Events.PictureSelectionChanged, function (e, info) {
                 * console.log("Sheet: " + info.sheetName);
                 * });
                 * ```
                 */
                static PictureSelectionChanged: string;
                /**
                 * After PivotTable filter/sort/collapse/fieldChanged/grandTotal/showNoData/group.
                 * @name GC.Spread.Sheets.Worksheet#PivotTableChanged
                 * @event
                 * @param eventParam *string* `pivotTableName` The PivotTable's name.
                 * @param eventParam *string* `type` The specific operation name("filter" | "sort" | "collapse" | "fieldChanged" | "summarizedValueBy" | "showValueAs" | "dataPositionChanged" | "viewChanged" | "grandTotal" | "showNoData" | "group").
                 * @param eventParam *string* `[fieldName]` Changed fieldName.
                 * @param eventParam *string* `[sourceName]` Changed field sourceName.
                 * @param eventParam *boolean* `[oldValue]` PivotTable changes the value of the previous attribute(collapse, summarizedValueBy, showValueAs).
                 * @param eventParam *boolean* `[newValue]` PivotTable changes the value of the previous attribute(collapse, summarizedValueBy, showValueAs).
                 * @param eventParam *number* `[sortType]` PivotTable sort changes the value of the attribute(sort).
                 * @param eventParam *Object* `[filterInfo]` PivotTable filter changes the value of the attribute(filter).
                 * @param eventParam *number* `[clearType]` PivotTable clear filter changes the value of the attribute(filter).
                 * @param eventParam *number* `[oldArea]` Old PivotTable Field Type(fieldChanged).
                 * @param eventParam *number* `[oldIndex]` Old PivotTable Field index(fieldChanged).
                 * @param eventParam *number* `[newArea]` New PivotTable Field Type(fieldChanged).
                 * @param eventParam *number* `[newIndex]` New PivotTable Field index(fieldChanged).
                 * @param eventParam *string* `[viewName]` Apply PivotTable Views Name(viewChanged).
                 * @example
                 * ```javascript
                 * //This example.
                 * sheet.bind(GC.Spread.Sheets.Events.PivotTableChanged, function (sender, args) {
                 *     alert("pivotTableName: " + args.pivotTableName + "changeType: " + args.type);
                 * });
                 * ```
                 */
                static PivotTableChanged: string;
                /**
                 * Occurs when the cell range has changed.
                 * @name GC.Spread.Sheets.Worksheet#RangeChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `row` The range's row index.
                 * @param eventParam *number* `col` The range's column index.
                 * @param eventParam *number* `rowCount` The range's row count.
                 * @param eventParam *number* `colCount` The range's column count.
                 * @param eventParam *string[]* `tableNames` A collection of table names.
                 * @param eventParam *Object[]* `changedCells` The positions of the cells whose data has changed, each position has row and col.
                 * @param eventParam *{@link GC.Spread.Sheets.RangeChangedAction}* `action` The type of action that raises the RangeChanged event.
                 * @param eventParam *boolean* `isUndo` Whether this event is from a undo operation.
                 * @example
                 * ```javascript
                 * //This example returns the sheet name and action when changing the cell range in Microsoft Internet Explorer.
                 *  activeSheet.bind(GC.Spread.Sheets.Events.RangeChanged, function (sender, args) {
                 *   console.log(args.sheetName, args.action);
                 * });
                 * ```
                 */
                static RangeChanged: string;
                /**
                 * Occurs when a range column has just been clear filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFilterCleared
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The index of the sheet column has just been clear filtered.
                 * @example
                 * ```javascript
                 * //This example uses the RangeFilterCleared event.
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * var cellRange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellRange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFilterCleared, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 * ```
                 */
                static RangeFilterCleared: string;
                /**
                 * Occurs when a range column is about to be automatically clear filter.
                 * @name GC.Spread.Sheets.Worksheet#RangeFilterClearing
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The index of the sheet column to be automatically clear filter.
                 * @example
                 * ```javascript
                 * //This example uses the RangeFilterClearing event.
                 * activeSheet.getCell(0,0).text("Name");
                 * activeSheet.getCell(0,1).text("Value");
                 * activeSheet.getCell(0,2).text("T/F");
                 * activeSheet.getCell(1,0).text("AW");
                 * activeSheet.getCell(1,1).text("5");
                 * activeSheet.getCell(1,2).text("T");
                 * var cellRange =new GC.Spread.Sheets.Range(0, 0, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellRange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFilterClearing, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 * ```
                 */
                static RangeFilterClearing: string;
                /**
                 * Occurs when a column has just been automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFiltered
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The index of the column that was automatically filtered.
                 * @param eventParam *Array* `filterValues` The values by which the column was filtered.
                 * @param eventParam *Array* `conditionInfo` The condition rule info that which the column was filtered.
                 * @example
                 * ```javascript
                 * //This example uses the RangeFiltered event.
                 * var cellrange =new GC.Spread.Sheets.Range(0, 2, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFiltered, function (e, info) {
                 *         alert("Col (" + info.col + ")");
                 * });
                 * ```
                 */
                static RangeFiltered: string;
                /**
                 * Occurs when a column is about to be automatically filtered.
                 * @name GC.Spread.Sheets.Worksheet#RangeFiltering
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *number* `col` The index of the column to be automatically filtered.
                 * @param eventParam *Array* `filterValues` The values by which to filter the column.
                 * @param eventParam *Object* `conditionInfo` The condition rule info by which to filter the column.
                 * @example
                 * ```javascript
                 * //This example uses the RangeFiltering event.
                 * var cellrange =new GC.Spread.Sheets.Range(0, 2, 5, 1);
                 * var hideRowFilter =new GC.Spread.Sheets.Filter.HideRowFilter(cellrange);
                 * activeSheet.rowFilter(hideRowFilter);
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeFiltering, function (e, info) {
                 *        alert("Col (" + info.col + ")");
                 * });
                 * ```
                 */
                static RangeFiltering: string;
                /**
                 * Occurs when the user has changed the outline state (range group) for rows or columns.
                 * @name GC.Spread.Sheets.Worksheet#RangeGroupStateChanged
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Worksheet}* `sheet` The sheet that triggered the event.
                 * @param eventParam *string* `sheetName` The sheet's name.
                 * @param eventParam *boolean* `isRowGroup` Whether the outline (range group) is a group of rows.
                 * @param eventParam *number* `index` The index of the RangeGroupInfo object whose state has changed.
                 * @param eventParam *number* `level` The level of the RangeGroupInfo object whose state has changed.
                 * @example
                 * ```javascript
                 * //This example uses the RangeGroupStateChanged event.
                 * activeSheet.suspendPaint();
                 * activeSheet.setRowCount(34);
                 * activeSheet.setValue(0,0,"Western");
                 * activeSheet.setValue(0,1,"Western");
                 * activeSheet.setValue(0,2,"Western");
                 * activeSheet.setValue(1,0,"A");
                 * activeSheet.setValue(1,1,"B");
                 * activeSheet.setValue(1,2,"C");
                 * activeSheet.setValue(2,0,"A");
                 * activeSheet.setValue(2,1,"B");
                 * activeSheet.setValue(2,2,"C");
                 * activeSheet.rowOutlines.group(0,2);
                 * activeSheet.columnOutlines.group(0,1);
                 * activeSheet.resumePaint();
                 * activeSheet.bind(GC.Spread.Sheets.Events.RangeGroupStateChanged, function (e, info) {
                 *         alert("Level (" + info.level + ")");
                 * });
                 * ```
                 */
                static RangeGroupStateChanged: string;
                /**
                 * Occurs before the user changes the outline state (range group) for rows or columns.
                 * @name GC.Spread.Sheets.Worksheet#RangeGroupStateChanging
                 * @event
                 * @param eventParam *{@link GC.Spread.Sheets.Work