/**
 * Document Solutions Data Viewer control.
 */
export declare class DsDataViewer {
    /**
     * Document Solutions Data Viewer constructor.
     * @param element root container element or selector pattern used to select the root container element.
     */
	constructor(element: HTMLElement | string);

	/**
     * Use this method to close and release resources occupied by the DsDataViewer.
     */
	dispose(): void;

	/**
     * Product license key.
     * @example
     *  ```
     *<script>
     *  // Add your license
     *  DsDataViewer.LicenseKey = 'XXX';
     *  // Add your code
     *  const viewer = new DsDataViewer("#viewer");
     *</script>
     *```
     */
    static LicenseKey: string;

	/**
     * Gets the viewer instance using the host element or host element selector
     * @example
     * ```javascript
     * var viewer = DsDataViewer.findControl("#root");
     * ```
     * @param selector
     */
    static findControl(selector: string | HTMLElement): DsDataViewer | undefined;

	/**
     * Open data file.
     * @param file The data file.
     * @param fileType The type of the imported data file.
     * @param openOptions The options for imported data file.
     * @example
     * ```javascript
     * viewer.openFile("Documents/HelloWorld.xlsx", FileType.XLSX, {loadHiddenSheets: true});
     * ```
     */
    openFile(file: Blob, fileType: FileType, openOptions?: XlsxOpenOptions | SSJsonOpenOptions | CsvOpenOptions | SjsOpenOptions): void;

    /**
     * Get the current toolbar layout information.
     * @example
     * ```javascript
     * viewer.toolbarLayout
     * ```
     */
    get toolbarLayout(): DataToolbarLayout;

    /**
     * Sets the toolbar layout information and modify layout.
     * @param buttons The toolbar layout information.
     * @example
     * ```javascript
     * viewer.toolbarLayout = { default: ['open', 'zoom', 'fullscreen', 'theme-change'] }
     * ```
     */
    set toolbarLayout(buttons: DataToolbarLayout);


    /**
     * Fetches the specified sheet based on the index.
     * @param index The index of the sheet to return.
     * @returns The specified sheet.
     * @example
     * ```javascript
     * viewer.getSheet(0)
     * ```
     */
    getSheet(index: number): WorkSheet;

    /**
     * Fetches the sheet with the specified name.
     * @param name The sheet name.
     * @returns The sheet with the specified name.
     * @example
     * ```javascript
     * viewer.getSheetFromName("Sheet1")
     * ```
     */
    getSheetFromName(name: string): WorkSheet;

    /**
     * Fetches the active sheet.
     * @returns The active sheet instance.
     * @example
     * ```javascript
     * viewer.getActiveSheet()
     * ```
     */
    getActiveSheet(): WorkSheet;


    /**
     * Get the current sidebar layout information.
     * @example
     * ```javascript
     * viewer.sidebarLayout
     * ```
     */
    get sidebarLayout(): string[];

    /**
     * Sets the sidebar layout information and modify layout.
     * Currently supported: SearchPanel.
     * @param sidebarLayout: The sidebar layout information.
     * @example
     * ```javascript
     * // Open SearchPanel
     * viewer.sidebarLayout = ['SearchPanel'];
     * // Clear all sidebars
     * viewer.sidebarLayout = [''];
     * // Default sidebar layout
     * viewer.sidebarLayout = [];
     * ```
     */
    set sidebarLayout(sidebarLayout: string[]);

    /**
     * Set the visibility of the sidebar.
     * @param show The visibility of the sidebar.
     * @example
     * ```javascript
     * viewer.showSidebar(true)
     * ```
     */
    showSidebar(show: boolean): void;
}

export declare class WorkSheet {
    /**
     * Retrieves formatted text in the cell based on the desired row and column index.
     * @param row The row index.
     * @param col The column index.
     * @returns Returns the formatted text of the cell.
     * @sample
     * ```javascript
     * var text = worksheet.getText(0, 0);
     * ```
     */
    getText(row: number, col: number): string;

    /**
     * Retrieves unformatted data from the specified cell based on the desired row and column index.
     * @param row The row index.
     * @param col The column index.
     * @returns Object Returns the value of the cell.
     * @sample
     * ```javascript
     * var value = worksheet.getValue(0, 0);
     * ```
     */
    getValue(row: number, col: number): any;

    /**
     * Retrieves the selections in the current sheet.
     * @returns The type Range is stored in an Array.
     * @sample
     * ```javascript
     * var selection = worksheet.getSelections();
     * ```
     */
    getSelections(): Range[] | undefined;
}

/**
 * Represents a range, which is described by the row index, column index, row count, and column count.
 */
export declare class Range {
    /**
     * The row count.
     */
    row: number;
    /**
     * The column index.
     */
    col: number;
    /**
     * The row count.
     */
    rowCount: number;
    /**
     * The column count.
     */
    colCount: number;
}

/**
 * The file type.
 */
export declare enum FileType {
	/**
     * XLSX.
     */
    XLSX = "XLSX",
	/**
     * SpreadJS JSON.
     */
    SSJSON = "SSJSON",
	/**
     * CSV.
     */
    CSV = "CSV",
    /**
     * SpreadJS SJS.
     */
    SJS = "SJS",
    /**
     * ARROW. 
     **/
    ARROW = 'ARROW',
    /**
     * PARQUET. 
     **/
    PARQUET = 'PARQUET'
}

 /**
 * Options to open the XLSX file.
 **/
 export type XlsxOpenOptions = {
     /**
     * Optional. Whether to show the hidden and very hidden sheets in XLSX file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {showHiddenSheets: true});
     * ```
     **/
     showHiddenSheets? : boolean;

     /**
     * Optional. Whether to show the hidden rows in XLSX file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {showHiddenRows: true});
     * ```
     **/
     showHiddenRows? : boolean;

     /**
     * Optional. Whether to show the hidden columns in XLSX file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {showHiddenColumns: true});
     * ```
     **/
     showHiddenColumns? : boolean;

     /**
     * Optional. Whether to show the filters in XLSX file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {showFilters: false});
     * ```
     **/
     showFilters? : boolean;

     /**
     * Optional. Whether to show the row groups when loading a XLSX file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {keepRowGroups: false});
     * ```
     **/
     keepRowGroups? : boolean;

     /**
     * Optional. Whether to show the column groups when loading a XLSX file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {keepColumnGroups: false});
     * ```
     **/
     keepColumnGroups? : boolean;

     /**
     * Optional. For decrypting password-protected XLSX file.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.xlsx', FileType.XLSX, {password: "123"});
     * ```
     **/
     password? : string;
 };

 /**
 * Options to open the SSJSON file.
 **/
 export type SSJsonOpenOptions = {
     /**
     * Optional. Whether to show the hidden and very hidden sheets in SSJSON file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {showHiddenSheets: true});
     * ```
     **/
     showHiddenSheets? : boolean;

     /**
     * Optional. Whether to show the hidden rows in SSJSON file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {showHiddenRows: true});
     * ```
     **/
     showHiddenRows? : boolean;

     /**
     * Optional. Whether to show the hidden columns in SSJSON file. The default value is false.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {showHiddenColumns: true});
     * ```
     **/
     showHiddenColumns? : boolean;

     /**
     * Optional. Whether to show the filters in SSJSON file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {showFilters: false});
     * ```
     **/
     showFilters? : boolean;

     /**
     * Optional. Whether to show the row groups when loading a SSJSON file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {keepRowGroups: false});
     * ```
     **/
     keepRowGroups? : boolean;

     /**
     * Optional. Whether to show the column groups when loading a SSJSON file. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.ssjson', FileType.SSJSON, {keepColumnGroups: false});
     * ```
     **/
     keepColumnGroups? : boolean;
 };

 /**
* Options to open the CSV file.
**/
export type CsvOpenOptions = {
     /**
     * Optional. Column separator for CSV file. The default value is ','.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.CSV', FileType.CSV, {columnSeparator: ','});
     * ```
     **/
     columnSeparator? : string;

     /**
     * Optional. Row separator for CSV file. The default value is '\r\n'.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.CSV', FileType.CSV, {rowSeparator: '\r\n'});
     * ```
     **/
     rowSeparator? : string;

     /**
     * Optional. Encoding for CSV file. The default value is 'UTF-8'.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.CSV', FileType.CSV, {encoding: 'ANSI'});
     * ```
     **/
     encoding? : string;

     /**
     * Optional. Whether the column data in the CSV file has a header. The default value is true.
     * @example
     * ```javascript
     * viewer.openFile('Documents/HelloWorld.CSV', FileType.CSV, {columnHasHeader: false});
     * ```
     **/
     columnHasHeader? : boolean;
 };

 /**
 * Options to open the SJS file.
 **/
 export type SjsOpenOptions = {
    /**
    * Optional. Whether to show the hidden and very hidden sheets in SJS file. The default value is false.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {showHiddenSheets: true});
    * ```
    **/
    showHiddenSheets? : boolean;

    /**
    * Optional. Whether to show the hidden rows in SJS file. The default value is false.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {showHiddenRows: true});
    * ```
    **/
    showHiddenRows? : boolean;

    /**
    * Optional. Whether to show the hidden columns in SJS file. The default value is false.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {showHiddenColumns: true});
    * ```
    **/
    showHiddenColumns? : boolean;

    /**
    * Optional. Whether to show the filters in SJS file. The default value is true.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {showFilters: false});
    * ```
    **/
    showFilters? : boolean;

    /**
    * Optional. Whether to show the row groups when loading a SJS file. The default value is true.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {keepRowGroups: false});
    * ```
    **/
    keepRowGroups? : boolean;

    /**
    * Optional. Whether to show the column groups when loading a SJS file. The default value is true.
    * @example
    * ```javascript
    * viewer.openFile('Documents/HelloWorld.sjs', FileType.SJS, {keepColumnGroups: false});
    * ```
    **/
    keepColumnGroups? : boolean;
};


 /**
  * Describes the toolbar items layout (order and visibility) for different view modes.
  * These items are available in the toolbar layout: open, zoom, fullscreen, theme-change, toggle-note, about.
  */
export type DataToolbarLayout = {
    /** Default (desktop) view mode. Also applied when other modes are not specified.
     * @example
     * ```javascript
     * viewer.toolbarlayout.default
     * ```
     */
    default?: string[];
    /** The layout for the full-screen mode.
     * @example
     * ```javascript
     * viewer.toolbarlayout.fullscreen
     * ```
     */
    fullscreen?: string[];
    /** The toolbar layout for mobile devices.
     * @example
     * ```javascript
     * viewer.toolbarlayout.mobile
     * ```
     */
    mobile?: string[];
}
